package com.ericsson.cms.epgmgmt.client.task;

import static com.ericsson.cms.epgmgmt.client.model.UITaskSetting.SUBSCRIBE_TO;

import java.util.ArrayList;
import java.util.Date;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import com.ericsson.cms.epgmgmt.client.EPGFormContainer;
import com.ericsson.cms.epgmgmt.client.i18n.EPGComponentConstants;
import com.ericsson.cms.epgmgmt.client.i18n.EPGComponentMessages;
import com.ericsson.cms.epgmgmt.client.model.UIOrderedTask;
import com.ericsson.cms.epgmgmt.client.model.UITask;
import com.ericsson.cms.epgmgmt.client.model.UITaskSetting;
import com.ericsson.cms.epgmgmt.client.model.UITaskTrigger;
import com.ericsson.cms.epgmgmt.client.model.enumeration.JobType;
import com.ericsson.cms.epgmgmt.client.model.enumeration.ScheduleType;
import com.ericsson.cms.epgmgmt.client.model.enumeration.TriggerType;
import com.ericsson.cms.epgmgmt.client.model.enumeration.WeekEnum;
import com.ericsson.cms.epgmgmt.client.rpc.IEPGTaskService;
import com.ericsson.cms.epgmgmt.client.rpc.IEPGTaskServiceAsync;
import com.ericsson.cms.epgmgmt.client.utils.DefaultAsyncCallback;
import com.ericsson.cms.epgmgmt.client.widget.TimePickerPopupPanel;
import com.google.gwt.core.client.GWT;
import com.google.gwt.event.dom.client.ChangeEvent;
import com.google.gwt.event.dom.client.ChangeHandler;
import com.google.gwt.event.dom.client.ClickEvent;
import com.google.gwt.event.dom.client.ClickHandler;
import com.google.gwt.event.logical.shared.ValueChangeEvent;
import com.google.gwt.event.logical.shared.ValueChangeHandler;
import com.google.gwt.i18n.client.NumberFormat;
import com.google.gwt.i18n.client.TimeZone;
import com.google.gwt.user.client.Window;
import com.google.gwt.user.client.ui.FlexTable;
import com.google.gwt.user.client.ui.HasVerticalAlignment;
import com.google.gwt.user.client.ui.TextBox;
import com.google.gwt.user.client.ui.Widget;
import com.tandbergtv.neptune.widgettoolkit.client.remote.NeptuneAsyncCallback;
import com.tandbergtv.neptune.widgettoolkit.client.widget.basic.CheckBoxWidget;
import com.tandbergtv.neptune.widgettoolkit.client.widget.basic.ImageWidget;
import com.tandbergtv.neptune.widgettoolkit.client.widget.basic.LabelWidget;
import com.tandbergtv.neptune.widgettoolkit.client.widget.basic.ListBoxWidget;
import com.tandbergtv.neptune.widgettoolkit.client.widget.container.HorizontalContainer;
import com.tandbergtv.neptune.widgettoolkit.client.widget.container.VerticalContainer;
import com.tandbergtv.neptune.widgettoolkit.client.widget.datatype.TypedListBoxWidget;
import com.tandbergtv.neptune.widgettoolkit.client.widget.datatype.converter.IntegerConverter;
import com.tandbergtv.neptune.widgettoolkit.client.widget.datatype.converter.StringConverter;
import com.tandbergtv.neptune.widgettoolkit.client.widget.servertime.internal.IServerTimeService;
import com.tandbergtv.neptune.widgettoolkit.client.widget.servertime.internal.IServerTimeServiceAsync;
import com.tandbergtv.neptune.widgettoolkit.client.widget.servertime.internal.ServerTime;

public class TaskTriggerSettingForm extends EPGFormContainer {

    private static final int MINUTES_PER_HOUR = 60;
    private static final int MAX_SCHEDULE_COUNT = 5;
    private static final String ICON_WIDHT = "18px";
    private static final String ICON_HEIGHT = "15px";
    private static final String ADD_ICON = "cms_epgmgmt_ui/images/add.png";
    private static final String REMOVE_ICON = "cms_epgmgmt_ui/images/remove-1.png";
    private UITaskTrigger taskTriggerModel;
    private final JobType type;
    private final EPGComponentConstants constants = GWT.create(EPGComponentConstants.class);
    private final EPGComponentMessages constantsMsg = (EPGComponentMessages) GWT.create(EPGComponentMessages.class);
    private final IEPGTaskServiceAsync taskService = GWT.create(IEPGTaskService.class);
    private final IServerTimeServiceAsync serverTimeService = GWT.create(IServerTimeService.class);
    private final TypedListBoxWidget<String> triggerTypeList = new TypedListBoxWidget<String>(new StringConverter());
    private FrequencyTimePicker frequencyTimePicker;
    private TextBox frequency;
    private final VerticalContainer scheduleContainer = new VerticalContainer();
    private final TypedListBoxWidget<String> scheduleTypeList = new TypedListBoxWidget<String>(new StringConverter());
    private final ImageWidget addScheduleIcon = new ImageWidget();
    private ClickHandler removeScheduleHandler;
    private final FlexTable scheduleValueTable = new FlexTable();
    private final VerticalContainer manualContainer = new VerticalContainer();
    private final CheckBoxWidget subscriberCheckbox = new CheckBoxWidget();
    private final ListBoxWidget<UIOrderedTask> subscriberTask = new ListBoxWidget<UIOrderedTask>();
    private Long providerId;
    private UITask model;
    private UITaskSetting subscribeSetting;
    private ImageWidget manualHelpIcon;

    public TaskTriggerSettingForm(JobType type, UITask model) {
        super(HorizontalContainer.ALIGN_LEFT);
        this.type = type;
        this.taskTriggerModel = model.getUiTaskTrigger();
        this.subscribeSetting = model.getTaskSettings().get(SUBSCRIBE_TO);
        this.model = model;
        bindEventHandler();
        initForm();
    }

    public void loadModel(UITask model) {
        this.model = model;
        this.taskTriggerModel = model.getUiTaskTrigger();
        this.subscribeSetting = model.getTaskSettings().get(SUBSCRIBE_TO);
        triggerTypeList.setSelectedItem(taskTriggerModel.getTriggerType().toString());
        frequencyTimePicker.loadModel(model.getUiTaskTrigger());
        scheduleTypeList.setSelectedItem(taskTriggerModel.getScheduleType().toString());

        subscriberCheckbox.setValue(taskTriggerModel.getTriggeredByPartner() != -1);
        if (taskTriggerModel.getTriggeredByPartner() != -1) {
            getTasks(taskTriggerModel.getTriggeredByPartner(), taskTriggerModel.getTriggeredByTask());
        }
        cascadeTriggerTypeChanges();
        cascadeScheduleTypeChanges();
    }

    private void initForm() {
        initTriggerTypeList();
        initFrequency();
        initSchedule();
        initManual();

        cascadeTriggerTypeChanges();
        cascadeScheduleTypeChanges();
    }

    private void initTriggerTypeList() {
        for (TriggerType triggerType : TriggerType.values()) {
            triggerTypeList.addItem(triggerType.toString(), triggerType.toString());
        }
        triggerTypeList.setSelectedItem(taskTriggerModel.getTriggerType().toString());

        HorizontalContainer triggerTypeContainer = new HorizontalContainer();
        triggerTypeContainer.add(triggerTypeList);
        manualHelpIcon = buildHelpIcon(constants.manualTips());
        triggerTypeContainer.setSpacing(4);// NOSONAR
        triggerTypeContainer.add(manualHelpIcon);
        triggerTypeContainer.setVerticalAlignment(HasVerticalAlignment.ALIGN_TOP);

        LabelWidget triggerTypeLabel = new LabelWidget(constants.providerTaskTriggerTypeLable());
        addRow(triggerTypeLabel, triggerTypeContainer, true);

        buildServerDifferenceTimeZoneTip(triggerTypeContainer);
    }

    private void buildServerDifferenceTimeZoneTip(final HorizontalContainer triggerTypeContainer) {
        serverTimeService.getCurrentTime(new NeptuneAsyncCallback<ServerTime>() {
            @Override
            public void onNeptuneFailure(Throwable caught) {
                Window.alert("Get server time failed!! \n " + caught);
            }

            @Override
            public void onNeptuneSuccess(ServerTime serverTime) {
                String tip = getScheduleTimeTimeZoneTip(serverTime);
                if (tip != null) {
                    ImageWidget warningButtonIcon = buildWarningIcon(tip);
                    triggerTypeContainer.add(warningButtonIcon);
                }
            }
        });
    }

    private String getScheduleTimeTimeZoneTip(ServerTime serverTime) {
        Date serverDate = serverTime.getDate();
        Date clientDate = new Date();
        TimeZone clientTimeZone = TimeZone.createTimeZone(clientDate.getTimezoneOffset());
        if (serverDate.getTimezoneOffset() != clientDate.getTimezoneOffset()) {
            String tip = constantsMsg.timeZoneDifferenceTip(serverTime.getTimeZone().getID(), clientTimeZone.getID(),
                    timeDifferenceDesc(serverDate, clientDate));
            return tip;
        }
        return null;
    }

    private String timeDifferenceDesc(Date date1, Date date2) {
        int d1Offset = date1.getTimezoneOffset();
        int d2Offset = date2.getTimezoneOffset();
        NumberFormat nf = NumberFormat.getDecimalFormat();
        nf.overrideFractionDigits(1);
        int offset = date1.getTimezoneOffset() - date2.getTimezoneOffset();
        String timeZoneOffset = "early ";
        if (offset < 0) {
            timeZoneOffset = "later ";
        }

        timeZoneOffset += nf.format((d1Offset - d2Offset) / 60);
        return timeZoneOffset;
    }

    private ImageWidget buildWarningIcon(String tip) {
        ImageWidget warningButtonIcon = new ImageWidget();
        warningButtonIcon.setUrl("cms_epgmgmt_ui/images/warning.png");
        warningButtonIcon.setTitle(tip);
        return warningButtonIcon;
    }

    private ImageWidget buildHelpIcon(String tip) {
        ImageWidget cleanButtonIcon = new ImageWidget();
        cleanButtonIcon.setUrl("cms_epgmgmt_ui/images/help.png");
        cleanButtonIcon.setTitle(tip);

        return cleanButtonIcon;
    }

    private void initFrequency() {
        frequencyTimePicker = new FrequencyTimePicker(taskTriggerModel);
        frequency = frequencyTimePicker.getTextBox();
        addRow(constants.frequency(), frequency, true);
    }

    private void initSchedule() {
        initScheduleList();
        addRow(constants.triggerScheduleType(), scheduleContainer, true);
    }

    private void initScheduleList() {
        for (ScheduleType schdeuleType : ScheduleType.values()) {
            scheduleTypeList.addItem(schdeuleType.toString(), schdeuleType.toString());
        }

        scheduleTypeList.setSelectedItem(taskTriggerModel.getScheduleType().toString());

        addScheduleIcon.setUrl(ADD_ICON);
        addScheduleIcon.setWidth(ICON_WIDHT);
        addScheduleIcon.setHeight(ICON_HEIGHT);
        addScheduleIcon.setTitle(constants.addScheduleItem());

        HorizontalContainer scheduleTypeContainer = new HorizontalContainer();
        scheduleTypeContainer.add(scheduleTypeList);
        scheduleTypeContainer.add(addScheduleIcon);
        scheduleTypeContainer.setCellVerticalAlignment(addScheduleIcon, HasVerticalAlignment.ALIGN_MIDDLE);

        scheduleContainer.add(scheduleTypeContainer);
        scheduleContainer.add(scheduleValueTable);
    }

    private void initManual() {
        if (JobType.EXPORT.equals(type)) {
            initSubscriberRow();
        }

        addRow("", manualContainer);
    }

    private void initSubscriberRow() {
        subscriberTask.setWidth("120px");
        subscriberTask.addChangeHandler(new SubscriberTaskChangeHandler());
        HorizontalContainer subscriberContainer = new HorizontalContainer();
        subscriberCheckbox.setText(constants.manualSubscriberLabel());
        subscriberContainer.add(subscriberCheckbox);
        subscriberContainer.setSpacing(4);// NOSONAR
        subscriberContainer.add(subscriberTask);
        subscriberContainer.setCellVerticalAlignment(subscriberCheckbox, HasVerticalAlignment.ALIGN_MIDDLE);
        subscriberContainer.setCellVerticalAlignment(subscriberTask, HasVerticalAlignment.ALIGN_MIDDLE);

        manualContainer.add(subscriberContainer);
        subscriberCheckbox.setEnabled(taskTriggerModel.getTriggeredByPartner() != -1);
        if (taskTriggerModel.getTriggeredByPartner() != -1) {
            getTasks(taskTriggerModel.getTriggeredByPartner(), taskTriggerModel.getTriggeredByTask());
        }
    }

    private void cascadeTriggerTypeChanges() {
        if (taskTriggerModel.getTriggerType() != null) {
            setFieldRowVisible(getFieldRowIndex(frequency),
                    taskTriggerModel.getTriggerType().equals(TriggerType.FREQUENCY));
            setFieldRowVisible(getFieldRowIndex(scheduleContainer),
                    taskTriggerModel.getTriggerType().equals(TriggerType.SCHEDULE));
            setFieldRowVisible(getFieldRowIndex(manualContainer),
                    taskTriggerModel.getTriggerType().equals(TriggerType.ONDEMAND));

            manualHelpIcon.setVisible(taskTriggerModel.getTriggerType() == TriggerType.ONDEMAND);
        }
    }

    private void cascadeScheduleTypeChanges() {
        scheduleValueTable.removeAllRows();

        long scheduleCount = taskTriggerModel.getScheduleCount();

        for (int i = 0; i < scheduleCount; i++) {
            addScheduleRow(taskTriggerModel.getScheduleHours()[i], taskTriggerModel.getScheduleMinutes()[i],
                    taskTriggerModel.getScheduleDaysOrWeeks()[i]);
        }

        if (scheduleCount == 0) {
            addScheduleRow(-1L, -1L, -1L);
        }

        scheduleCount = getScheduleCount();

        if (scheduleCount == 1) {
            getFirstScheduleTriggerIcon().setVisible(false);
        }

        addScheduleIcon.setVisible(scheduleCount != MAX_SCHEDULE_COUNT);
    }

    private void addScheduleRow(long selectedHour, long selectedMinute, long selectedWeekOrMonth) {
        int row = scheduleValueTable.getRowCount();
        int column = 0;
        scheduleValueTable.setWidget(row, column++, createTimePickerWidget(selectedHour, selectedMinute));
        if (ScheduleType.WEEKLY.equals(taskTriggerModel.getScheduleType())) {
            scheduleValueTable.setWidget(row, column++, createWeeksList(selectedWeekOrMonth));
        } else if (ScheduleType.MONTHLY.equals(taskTriggerModel.getScheduleType())) {
            scheduleValueTable.setWidget(row, column++, createMonthsList(selectedWeekOrMonth));
        }
        scheduleValueTable.setWidget(row, column, buildRemoveScheduleIconWidget());
    }

    private ImageWidget buildRemoveScheduleIconWidget() {
        final ImageWidget imageWidget = new ImageWidget();
        imageWidget.setUrl(REMOVE_ICON);
        imageWidget.setWidth(ICON_WIDHT);
        imageWidget.setHeight(ICON_HEIGHT);
        imageWidget.setTitle(constants.removeScheduleItem());
        imageWidget.addClickHandler(removeScheduleHandler);
        return imageWidget;
    }

    private Widget createTimePickerWidget(final long selectedHour, final long selectedMinute) {
        TextBox textBox = new TimePickerPopupPanel(selectedHour, selectedMinute).getTextBox();
        textBox.setWidth("60px");
        return textBox;
    }

    private TypedListBoxWidget<Long> createWeeksList(long selectedValue) {
        TypedListBoxWidget<Long> weeks = new TypedListBoxWidget<Long>(new IntegerConverter());
        weeks.clear();
        weeks.addItem("", -1L);
        for (int i = 1; i <= WeekEnum.values().length; i++) {
            weeks.addItem(WeekEnum.getTextByValue(i), (long) i);
        }
        weeks.setSelectedItem(selectedValue);
        return weeks;
    }

    private TypedListBoxWidget<Long> createMonthsList(long selectedValue) {
        Map<Integer, String> monthMap = new HashMap<Integer, String>();
        monthMap.put(1, "1st");
        monthMap.put(2, "2nd");
        monthMap.put(3, "3rd");
        TypedListBoxWidget<Long> months = new TypedListBoxWidget<Long>(new IntegerConverter());
        months.clear();
        months.addItem("", -1L);
        for (int i = 1; i <= 31; i++) {
            String itemValue = monthMap.get(i) == null ? i + "th" : monthMap.get(i);
            months.addItem(itemValue, (long) i);
        }
        months.setSelectedItem(selectedValue);
        return months;
    }

    private TypedListBoxWidget<Long> createList(TypedListBoxWidget<Long> l, long maxValue, long selectedValue) {
        l.clear();
        for (long i = 0L; i < maxValue; i++) {
            l.addItem(String.valueOf(i), i);
        }
        l.setSelectedItem(selectedValue);
        return l;
    }

    private void bindEventHandler() {
        triggerTypeList.addValueChangeHandler(new ValueChangeHandler<String>() {
            @Override
            public void onValueChange(ValueChangeEvent<String> event) {
                TriggerType triggerType = TriggerType.fromValue(event.getValue());
                taskTriggerModel.setTriggerType(triggerType);
                cascadeTriggerTypeChanges();
            }
        });
        scheduleTypeList.addValueChangeHandler(new ValueChangeHandler<String>() {
            @Override
            public void onValueChange(ValueChangeEvent<String> event) {
                ScheduleType scheduleType = ScheduleType.fromValue(event.getValue());
                taskTriggerModel.setScheduleType(scheduleType);
                taskTriggerModel.setScheduleCount(0);
                cascadeScheduleTypeChanges();
            }
        });
        addScheduleIcon.addClickHandler(new ClickHandler() {
            @Override
            public void onClick(ClickEvent event) {
                addScheduleRow(-1L, -1L, -1L);
                getFirstScheduleTriggerIcon().setVisible(true);

                int scheduleCount = getScheduleCount();
                if (scheduleCount == MAX_SCHEDULE_COUNT) {
                    addScheduleIcon.setVisible(false);
                }
            }
        });

        removeScheduleHandler = new ClickHandler() {
            @Override
            public void onClick(ClickEvent event) {
                int rowIndex = scheduleValueTable.getCellForEvent(event).getRowIndex();
                scheduleValueTable.removeRow(rowIndex);
                addScheduleIcon.setVisible(true);

                int scheduleCount = getScheduleCount();
                if (scheduleCount == 1) { // NOSONAR
                    getFirstScheduleTriggerIcon().setVisible(false);
                }
            }
        };

        subscriberCheckbox.addClickHandler(new ClickHandler() {
            @Override
            public void onClick(ClickEvent event) {
                subscriberTask.setEnabled(subscriberCheckbox.getValue());
            }
        });
    }

    private Widget getFirstScheduleTriggerIcon() {
        int columnIndex = taskTriggerModel.getScheduleType() == ScheduleType.DAILY ? 1 : 2;
        return scheduleValueTable.getWidget(0, columnIndex);
    }

    public List<String> validateField() {
        List<String> validationMessages = new ArrayList<String>();

        if ((triggerTypeList.getTextValue() == null) || triggerTypeList.getTextValue().isEmpty()) {
            validationMessages.add(constantsMsg.requiredTriggerTypeName());
        }

        if (taskTriggerModel.getTriggerType().equals(TriggerType.FREQUENCY)) {
            if ((taskTriggerModel.getHours() == 0) && (taskTriggerModel.getMinutes() == 0)) {
                validationMessages.add(constantsMsg.defaultFrequency());
            }
        } else if (taskTriggerModel.getTriggerType().equals(TriggerType.SCHEDULE)) {
            int scheduleCount = getScheduleCount();
            if (scheduleCount == 0) {
                validationMessages.add(constantsMsg.scheduleCountZero());
            } else {
                boolean hasDuplicated = false;
                boolean hasUnconfigured = false;
                long[] scheduleInMinutes = getScheduleInMinutes();
                if (taskTriggerModel.getScheduleType().equals(ScheduleType.DAILY)) {
                    hasDuplicated = validateDuplication(scheduleInMinutes);
                    hasUnconfigured = hasUnconfigured(scheduleInMinutes);
                } else {
                    long[] scheduleDayOrWeeks = getScheduleDayOrWeeks();
                    hasDuplicated = validateDuplication(scheduleInMinutes, scheduleDayOrWeeks);
                    hasUnconfigured = hasUnconfigured(scheduleInMinutes) || hasUnconfigured(scheduleDayOrWeeks);
                }

                if (hasDuplicated) {
                    validationMessages.add(constantsMsg.scheduleDuplication());
                }

                if (hasUnconfigured) {
                    validationMessages.add(constantsMsg.scheduleUnconfigured());
                }
            }
        }
        return validationMessages;
    }

    private boolean hasUnconfigured(long[] array) {
        boolean unconfigured = false;
        for (int i = 0; i < array.length; i++) {
            if (array[i] == -1) {
                unconfigured = true;
            }
        }
        return unconfigured;
    }

    private boolean validateDuplication(long[] array) {
        boolean duplicated = false;
        for (int i = 0; i < array.length; i++) {
            for (int j = i + 1; j < array.length; j++) {
                if (array[i] == array[j]) {
                    duplicated = true;
                }
            }
        }
        return duplicated;
    }

    // Array1 & Array2 should have same length
    private boolean validateDuplication(long[] array1, long[] array2) {
        boolean duplicated = false;

        for (int i = 0; i < array1.length; i++) {
            for (int j = i + 1; j < array1.length; j++) {
                if ((array1[i] == array1[j]) && (array2[i] == array2[j])) {
                    duplicated = true;
                }
            }
        }
        return duplicated;
    }

    public void populateModel() {
        taskTriggerModel.setTriggerType(TriggerType.fromValue(triggerTypeList.getSelectedItem()));

        if (taskTriggerModel.getTriggerType().equals(TriggerType.FREQUENCY)) {
            taskTriggerModel.setHours(frequencyTimePicker.getHour());
            taskTriggerModel.setMinutes(frequencyTimePicker.getMinute());
        } else if (taskTriggerModel.getTriggerType().equals(TriggerType.SCHEDULE)) {
            populateScheduleValue();
        } else {
            if (subscriberCheckbox.getValue()) {
                Map<String, Object> map = new HashMap<String, Object>();
                if (this.subscribeSetting == null) {
                    this.subscribeSetting = new UITaskSetting();
                    this.subscribeSetting.setKey(SUBSCRIBE_TO);
                    this.model.getTaskSettings().put(SUBSCRIBE_TO, this.subscribeSetting);
                }
                map.put(SUBSCRIBE_TO, String.valueOf(this.subscriberTask.getSelectedItem().getId()));
                this.subscribeSetting.setSetting(map);
            } else {
                this.model.getTaskSettings().remove(SUBSCRIBE_TO);
            }
        }
    }

    private void populateScheduleValue() {
        taskTriggerModel.setScheduleHours(getScheduleHours());
        taskTriggerModel.setScheduleMinutes(getScheduleMinutes());
        taskTriggerModel.setScheduleDaysOrWeeks(getScheduleDayOrWeeks());
        taskTriggerModel.setScheduleCount(getScheduleCount());
    }

    private int getScheduleCount() {
        return scheduleValueTable.getRowCount();
    }

    private long[] getScheduleHours() {

        long[] scheduleInMinutes = getScheduleInMinutes();
        if (scheduleInMinutes == null) {
            return null;
        }

        long[] scheduleHours = new long[scheduleInMinutes.length];
        for (int i = 0; i < scheduleInMinutes.length; i++) {
            scheduleHours[i] = -1L;
            if (scheduleInMinutes[i] != -1L) {
                scheduleHours[i] = scheduleInMinutes[i] / MINUTES_PER_HOUR;
            }
        }

        return scheduleHours;
    }

    private long[] getScheduleMinutes() {

        long[] scheduleInMinutes = getScheduleInMinutes();
        if (scheduleInMinutes == null) {
            return null;
        }

        long[] scheduleHours = new long[scheduleInMinutes.length];
        for (int i = 0; i < scheduleInMinutes.length; i++) {
            scheduleHours[i] = -1L;
            if (scheduleInMinutes[i] != -1L) {
                scheduleHours[i] = scheduleInMinutes[i] % MINUTES_PER_HOUR;
            }
        }

        return scheduleHours;
    }

    private long[] getScheduleInMinutes() {
        int scheduleCount = getScheduleCount();
        if (scheduleCount <= 0) {
            return null;
        }

        long[] scheduleInMinutes = new long[scheduleCount];
        for (int row = 0; row < scheduleValueTable.getRowCount(); row++) {
            scheduleInMinutes[row] = -1L;
            String time = ((TextBox) scheduleValueTable.getWidget(row, 0)).getText();
            if ((time != null) && time.contains(TimePickerPopupPanel.TIME_SPERATOR)) {
                String[] times = time.split(TimePickerPopupPanel.TIME_SPERATOR);
                scheduleInMinutes[row] = (Long.parseLong(times[0]) * MINUTES_PER_HOUR) + Long.parseLong(times[1]);
            }
        }
        return scheduleInMinutes;
    }

    @SuppressWarnings("unchecked")
    private long[] getScheduleDayOrWeeks() {
        int scheduleCount = getScheduleCount();
        if ((scheduleCount <= 0) || taskTriggerModel.getScheduleType().equals(ScheduleType.DAILY)) {
            return null;
        }

        long[] scheduleDayOrWeeks = new long[scheduleCount];
        for (int row = 0; row < scheduleValueTable.getRowCount(); row++) {
            Long weekOrMonth = ((TypedListBoxWidget<Long>) scheduleValueTable.getWidget(row, 1)).getSelectedItem();
            scheduleDayOrWeeks[row] = weekOrMonth;
        }
        return scheduleDayOrWeeks;
    }

    private void getTasks(final long partnerId, final long selectedItem) {
        subscriberCheckbox.setValue(false);
        subscriberCheckbox.setEnabled(true);
        subscriberTask.setEnabled(false);
        taskService.getTasksByPartnerId(partnerId, new DefaultAsyncCallback<List<UITask>>(null) {
            @Override
            public void onNeptuneSuccess(List<UITask> tasks) {
                subscriberTask.clear();
                List<UIOrderedTask> sortedTasks = IngestTaskSorter.sortByFollowUpOrder(tasks);
                String subscribedTaskId = getSubscribedTask();
                UIOrderedTask findSubscribedTask = null;
                for (UIOrderedTask task : sortedTasks) {
                    if (!task.isEnabled()) {
                        continue;
                    }
                    subscriberTask.addItem(task.getName(), task);
                    if (String.valueOf(task.getId()).equals(subscribedTaskId)) {
                        findSubscribedTask = task;
                    }
                }

                if (findSubscribedTask != null) {
                    subscriberCheckbox.setValue(true);
                    subscriberTask.setSelectedItem(findSubscribedTask);
                    subscriberTask.setEnabled(true);
                }
            }
        });
    }

    private String getSubscribedTask() {
        String taskId = null;
        if ((subscribeSetting != null) && (subscribeSetting.getSetting() != null)) {
            taskId = (String) subscribeSetting.getSetting().get(SUBSCRIBE_TO);
        }
        return taskId;
    }

    public Long getProviderId() {
        return providerId;
    }

    public void setProviderId(Long providerId) {
        this.providerId = providerId;
        getTasks(providerId, -1);
    }

    private final class SubscriberTaskChangeHandler implements ChangeHandler {
        @Override
        public void onChange(ChangeEvent event) {
            UIOrderedTask task = subscriberTask.getSelectedItem();
            int index = subscriberTask.getSelectedIndex();
            if ((task != null) && (index < (subscriberTask.getItemCount() - 1))) {
                UIOrderedTask nextTask = subscriberTask.getItem(index + 1);
                if (nextTask.getOrder() > task.getOrder()) {
                    Window.alert(constantsMsg.alertExportSubscriber());
                }
            }
        }
    }
}