package com.ericsson.cms.me.client.view.custom;

import java.util.ArrayList;
import java.util.List;

import com.google.gwt.event.dom.client.ClickEvent;
import com.google.gwt.event.dom.client.ClickHandler;
import com.google.gwt.user.client.Window;
import com.google.gwt.user.client.ui.Button;
import com.google.gwt.user.client.ui.Widget;
import com.tandbergtv.neptune.widgettoolkit.client.widget.basic.LabelWidget;
import com.tandbergtv.neptune.widgettoolkit.client.widget.container.FlexTableContainer;

/**
 * @author CMeyer
 *
 */
public class ResizeableTable extends FlexTableContainer{
    private List<List<Widget>> valueTable = new ArrayList<List<Widget>>();

    private TemplateRowGenerator templateRowGen;
    private boolean hasHeaders = false;
    private int maxWidth = 0;
    private static final boolean CONFIRM_REMOVE = false;

    private boolean editable = true;

    public ResizeableTable(){}

    /**
     * Constructs a ResizeableTable using the specified TemplateRowGenerator to generate new rows
     * @param templateRowGen
     */
    public ResizeableTable(TemplateRowGenerator templateRowGen){
        this.templateRowGen = templateRowGen;
        addRow(templateRowGen.getTemplateRow());
        resetAddButtons();
    }

    /**
     * Constructs a ResizeableTable using the specified TemplateRowGenerator to generate new rows,
     * and displaying the specified existing values
     * @param templateRowGen
     * @param existingValues
     * @param editable
     */
    public ResizeableTable(TemplateRowGenerator templateRowGen, List<List<Widget>> existingValues, boolean editable){
        this.templateRowGen = templateRowGen;
        setEditable(editable);
        this.valueTable = existingValues;
        maxWidth = calcMaxWidth(valueTable);

        if(existingValues.size() > 0){
            int i=0;
            for(List<Widget> curList : existingValues){
                setRow(i, curList);
                i++;
            }
        }else{
            setRow(0, templateRowGen.getTemplateRow());
        }

        resetAddButtons();
    }

    /**
     * Set the header row of the tabl
     * @param headers
     */
    public void setHeaders(List<String> headers){
        if(!hasHeaders){
            insertRow(0);
            hasHeaders = true;
        }

        int i = 0;
        for(String header : headers){
            LabelWidget label = new LabelWidget(header);
            setWidget(0, i, label);
            i++;
        }
    }

    private void setRow(int rowIndex, List<Widget> row){
        if(row != null){
            int i = 0;
            for(Widget curWidget : row){
                setWidget(rowIndex, i, curWidget);
                i++;
            }

            if(row instanceof TableRow){
                TableRow tableRow = (TableRow)row;
                if(tableRow.isRemovable()){
                    setWidget(rowIndex, i, new ResizeButton(Operation.remove));
                }
            } else{
                setWidget(rowIndex, i, new ResizeButton(Operation.remove));
            }
        }
    }

    /**
     * Return all of the value widgets contained in the table
     * @return
     */
    public List<List<Widget>> getValues(){
        return this.valueTable;
    }

    /**
     * Add a new row of widgets to the end of the table
     * @param row
     */
    public void addRow(List<Widget> row){
        valueTable.add(row);
        setRow(valueTable.size()-1+getHeaderOffset(), row);
        maxWidth = calcMaxWidth(valueTable);
        resetAddButtons();
    }

    /**
     * Determine whether the table may have rows added to/removed from it
     * @param editable
     */
    public void setEditable(boolean editable){
        this.editable = editable;
    }

    private int calcMaxWidth(List<List<Widget>> valueTable){
        int max = 0;
        for(List<Widget> curList : valueTable){
            max = curList.size() > max ? curList.size() : max;
        }

        return max;
    }

    private int getHeaderOffset(){
        return hasHeaders ? 1 : 0;
    }

    private void resetAddButtons(){
        int headerOffset = getHeaderOffset();
        for(int i=headerOffset; i<getRowCount(); i++){
            if(getCellCount(i) >= maxWidth+2){
                removeCell(i, maxWidth+1);
            }
        }

        setWidget(getRowCount()-1, maxWidth+1, new ResizeButton(Operation.add));
    }

    class ResizeButton extends Button{
        private static final String REMOVE_LABEL = "-";
        private static final String ADD_LABEL = "+";

        Operation operation;
        int rowNum;
        protected ResizeButton(Operation oper){
            super();
            this.operation = oper;
            String label = REMOVE_LABEL;
            switch(operation){
            case add:
                label = ADD_LABEL;
                break;
            case remove:
                label = REMOVE_LABEL;
                break;
            }

            setHTML(label);
            addClickHandler(new ResizeButtonClickHandler());
            setVisible(editable);
        }

        protected int getRowNum(){
            return rowNum;
        }

        protected Operation getOperation(){
            return operation;
        }

    }

    class ResizeButtonClickHandler implements ClickHandler{
        @Override
        public void onClick(ClickEvent event){
            ResizeButton source = (ResizeButton)event.getSource();

            switch(source.getOperation()){
            case add:
                try{
                    addRow(templateRowGen.getTemplateRow());
                    source.setVisible(false);
                } catch(Exception e){
                    Window.alert("Unable to add row (" + e.getMessage() + ")");
                }
                break;

            case remove:
                if (!CONFIRM_REMOVE || Window.confirm("Are you sure you want to remove this row?")) {
                    if (valueTable.size() > 1) {  // NOSONAR
                        Cell clickedCell = getCellForEvent(event);
                        removeRow(clickedCell.getRowIndex());

                        valueTable.remove(clickedCell.getRowIndex() - getHeaderOffset());
                        resetAddButtons();
                    }
                }
                break;

            default:
                Window.alert("Unknown operation: " + source.getOperation());
            }
        }
    }

    enum Operation{
        add,
        remove
    }

    /**
     * Interface for classes to add new (blank or default) rows to a ResizeableTable.
     * @author CMeyer
     *
     */
    public interface TemplateRowGenerator{
        public List<Widget> getTemplateRow();
    }

    public class TableRow<T> extends ArrayList<T>{
        private boolean removable = true;

        public TableRow(boolean removable){
            setRemovable(removable);
        }

        public void setRemovable(boolean removable){
            this.removable = removable;
        }
        public boolean isRemovable(){
            return this.removable;
        }
    }
}
