package com.tandbergtv.cms.rules.ui.client;

import java.util.ArrayList;
import java.util.List;
import java.util.Map;

import com.ericsson.cms.scheduling.ui.client.IUISchedulerClient;
import com.google.gwt.core.client.GWT;
import com.google.gwt.event.shared.HandlerManager;
import com.google.gwt.user.client.History;
import com.tandbergtv.cms.portal.ui.title.client.criteria.FieldTree.AssetTreeObject;
import com.tandbergtv.cms.rules.ui.client.data.Packages;
import com.tandbergtv.cms.rules.ui.client.ruleServiceErrorHandling.RuleSetDoesntExist;
import com.tandbergtv.neptune.widgettoolkit.client.application.ClientAuthorizationManager;
import com.tandbergtv.neptune.widgettoolkit.client.application.PortletFactory;
import com.tandbergtv.neptune.widgettoolkit.client.application.Service;
import com.tandbergtv.neptune.widgettoolkit.client.application.ServiceLoader;
import com.tandbergtv.neptune.widgettoolkit.client.component.ComponentBase;
import com.tandbergtv.neptune.widgettoolkit.client.event.EventListenerRegistry;
import com.tandbergtv.neptune.widgettoolkit.client.event.EventSink;
import com.tandbergtv.neptune.widgettoolkit.client.menu.GroupMenuItem;
import com.tandbergtv.neptune.widgettoolkit.client.menu.MenuItemBase;
import com.tandbergtv.neptune.widgettoolkit.client.menu.WidgetMenuItem;
import com.tandbergtv.neptune.widgettoolkit.client.remote.NeptuneAsyncCallback;
import com.tandbergtv.neptune.widgettoolkit.client.security.NeptuneSecurity;
import com.tandbergtv.neptune.widgettoolkit.client.widget.composite.BusyIndicator;

public class RulesComponent extends ComponentBase {

	private static final String SCHEDULER_LICENSE_KEY = "SchedulerLicense";

	private RulesConstants myConstants = (RulesConstants) GWT.create(RulesConstants.class);

	public static final String ruleAnchor = "Rules";
	public static final String newAnchor = "Create";
	public static final String existingAnchor = "id";
	public static final String SITEID = "siteid";

	private static RulesComponent instance = new RulesComponent();
	private NeptuneSecurity security;
	private HandlerManager handlerManager;
	private RulesTab tabNormalize;
	private RulesTab tabValidate;
	private RulesTab tabLicensing;
	private RulesTab tabProcess;
	private RulesTab tabTargeting;
	private RulesTab tabPrioritization;
	private RulesTab tabMsoOverride;
	private boolean hasErrors;
	private Packages currentPackage = Packages.NORMALIZATION;
	private BusyIndicator busyIndicator = new BusyIndicator();
	private UiRulesServiceAsync rulesService = GWT.create(UiRulesService.class);
	private boolean schedulerLicensed;

	public RulesComponent() {
		this.hasErrors = false;
		this.tabNormalize = new RulesTab(Packages.NORMALIZATION);
		this.tabValidate = new RulesTab(Packages.VALIDATION);
		this.tabLicensing = new RulesTab(Packages.LICENSING);
		this.tabProcess = new RulesTab(Packages.PROCESSING);
		this.tabTargeting = new RulesTab(Packages.TARGETING);
		this.tabPrioritization = new RulesTab(Packages.PRIORITIZATION);
		this.tabMsoOverride = new RulesTab(Packages.MSOOVERRIDE);
		this.handlerManager = new HandlerManager(this, true);
	}

	/**
	 * Only allow one for rules component so tab panels don't get confused
	 * 
	 * @return
	 */
	public static RulesComponent getInstance() {
		return instance;
	}

	/* (non-Javadoc)
	 * @see com.tandbergtv.neptune.widgettoolkit.client.component.ComponentBase#init(com.tandbergtv.neptune.widgettoolkit.client.event.EventListenerRegistry, com.tandbergtv.neptune.widgettoolkit.client.event.EventSink, com.tandbergtv.neptune.widgettoolkit.client.application.ServiceLoader, java.util.Map)
	 */
	@Override
	public void init(EventListenerRegistry eventListenerRegistry,
			EventSink eventSink, ServiceLoader serviceLoader,
			Map<String, String> info) {
		super.init(eventListenerRegistry, eventSink, serviceLoader, info);
		
		schedulerLicensed = getBooleanValue(SCHEDULER_LICENSE_KEY);
	}

	/**
	 * Get the container for a particular package
	 * 
	 * @param panel
	 * @return
	 */
	public RulesTab getTab(Packages panel) {
		switch (panel) {
		case NORMALIZATION:
			return tabNormalize;
		case VALIDATION:
			return tabValidate;
		case PROCESSING:
			return tabProcess;
		case TARGETING:
			return tabTargeting;
		case PRIORITIZATION:
			return tabPrioritization;
		case LICENSING:
			return tabLicensing;
		case MSOOVERRIDE:
			return tabMsoOverride;
		}
		return null; // throw exception perhaps...
	}
	
	public RulesTab getCurrentTab() {
		return getTab(currentPackage);
	}

	public void setCurrentPackage(Packages pkg) {
		currentPackage = pkg;
	}

	/**
	 * {@inheritDoc}
	 */
	public List<MenuItemBase> listTabs(NeptuneSecurity security) {
		List<MenuItemBase> tabs = new ArrayList<MenuItemBase>();
		this.security = security;

		/* Always add the group menu item */
		List<MenuItemBase> rulesGroup = new ArrayList<MenuItemBase>();
		tabs.add(new GroupMenuItem(ruleAnchor, myConstants.ruleLabel(),
				rulesGroup));

		// if have appropriate permissions then add tab items to menu
		if (security.isUserInRole("RulesManagement_View")) {
			rulesGroup.add(createNormalisationPanel());
			rulesGroup.add(createValidationPanel());
			if (ClientAuthorizationManager.isAuthorized("ContractManager_View"))
				rulesGroup.add(createLicensingPanel());
			rulesGroup.add(createProcessingPanel());
			if (RulesUtils.isFeatureAvailable(RulesUtils.FEATURE_NAME_SITES)) {
				rulesGroup.add(createTargetingPanel());
			}
			rulesGroup.add(createPrioritizationPanel());
			rulesGroup.add(createMsoOverridePanel());
		}

		return tabs;
	}

	/**
	 * {@inheritDoc}
	 */
	public List<PortletFactory> listPortlets(NeptuneSecurity security) {
		return null;
	}

	public HandlerManager getHandlerManager() {
		return this.handlerManager;
	}

	/**
	 * {@inheritDoc}
	 */
	@SuppressWarnings("unchecked")
	public <E extends Service> List<E> listServices(Class<E> clazz,
			NeptuneSecurity security) {
		List<E> services = new ArrayList<E>();
		if(clazz.equals(IUISchedulerClient.class)) {
			services.add((E) new RulesUISchedulerClient(security));
		}
		return services;
	}

	/**
	 * Get security parameters of the rules component
	 * 
	 * @return
	 */
	public NeptuneSecurity getSecurity() {
		return this.security;
	}

	/**
	 * Gets the rules constants only
	 * 
	 * @return
	 */
	public RulesConstants getConstants() {
		return this.myConstants;
	}

	/**
	 * returns whether the rules component is ready. So far only dependent on
	 * the spec
	 * 
	 * @return
	 */
	public boolean isReady() {
		return AssetTreeObject.isInitialized();
	}

	/**
	 * Indicates has errors while loading component
	 * 
	 * @return
	 */
	public boolean hasErrors() {
		return this.hasErrors;
	}

	/*
	 * create and add normalisation tab that links to the panel
	 */
	private WidgetMenuItem createNormalisationPanel() {
		String label = myConstants.normalizationLabel();
		return new WidgetMenuItem(Packages.NORMALIZATION.toString(), label,
				tabNormalize, tabNormalize.getAnchorChangeListener());
	}

	/*
	 * create and add validation panel to tabs group
	 */
	private WidgetMenuItem createValidationPanel() {
		String label = myConstants.validationLabel();
		return new WidgetMenuItem(Packages.VALIDATION.toString(), label,
				tabValidate, tabValidate.getAnchorChangeListener());
	}
	
	/*
	 * create and add validation panel to tabs group
	 */
	private WidgetMenuItem createLicensingPanel() {
		String label = myConstants.liceningAssignment();
		return new WidgetMenuItem(Packages.LICENSING.toString(), label,
				tabLicensing, tabLicensing.getAnchorChangeListener());
	}

	/*
	 * create and add content Processing panel to tabs group
	 */
	private WidgetMenuItem createProcessingPanel() {
		String label = myConstants.ContentProcessingLabel();
		return new WidgetMenuItem(Packages.PROCESSING.toString(), label,
				tabProcess, tabProcess.getAnchorChangeListener());
	}
	
	/*
	 * create and add content Processing panel to tabs group
	 */
	private WidgetMenuItem createMsoOverridePanel() {
		String label = myConstants.msoOverrideLabel();
		return new WidgetMenuItem(Packages.MSOOVERRIDE.toString(), label,
				tabMsoOverride, tabMsoOverride.getAnchorChangeListener());
	}

	/*
	 * create and add Targeting panel to tabs group
	 */
	private WidgetMenuItem createTargetingPanel() {
		String label = myConstants.targetingLabel();
		return new WidgetMenuItem(Packages.TARGETING.toString(), label,
				tabTargeting, tabTargeting.getAnchorChangeListener());
	}

	/*
	 * create and add Targeting panel to tabs group
	 */
	private WidgetMenuItem createPrioritizationPanel() {
		String label = myConstants.prioritizationLabel();
		return new WidgetMenuItem(Packages.PRIORITIZATION.toString(), label,
				tabPrioritization, tabPrioritization.getAnchorChangeListener());
	}

	/*
	 * Asynchronously loads the specification for the rules component
	 */
	public void loadCache(final String anchor) {
		busyIndicator.center();
		AssetTreeObject.initialize(new NeptuneAsyncCallback<Void>() {

			@Override
			public void onNeptuneFailure(Throwable caught) {
				setInErrorState();
			}

			@Override
			public void onNeptuneSuccess(Void result) {
				checkCacheLoaded(anchor);
			}
			
		});

	}

	public void checkCacheLoaded(String anchor) {
		if(isReady()) {
			busyIndicator.hide();
			getCurrentTab().populateTab(anchor);
		}
	}

	private void setInErrorState() {
		busyIndicator.hide();
		hasErrors = true;
		getCurrentTab().setPanelInError();
	}

	// for testing
	public void setSecurity(NeptuneSecurity security) {
		this.security = security;
	}

	public void setNewRuleAnchor(Packages pkg) {
		History.newItem(ruleAnchor + "." + getPackageAnchor(pkg) + "."
				+ newAnchor);
	}

	public void setNewRuleAnchor(Packages pkg, int siteId) {
		History.newItem(ruleAnchor + "." + getPackageAnchor(pkg) + "."
				+ newAnchor+"?"+ SITEID + "=" +siteId);
	}

	public void setViewExistingAnchor(final int ruleSetId) {
		busyIndicator.center();
		rulesService.getPackageType(ruleSetId, new NeptuneAsyncCallback<Packages>() {

			@Override
			public void onNeptuneFailure(Throwable caught) {
				busyIndicator.hide();
				if (caught instanceof RuleSetDoesntExist) {
					getCurrentTab().setPanelInError(
							myConstants.invalidRuleSetId());
				} else {
					getCurrentTab().setPanelInError(
							myConstants.errLoadingRuleSet());
				}
			}

			@Override
			public void onNeptuneSuccess(Packages result) {
				busyIndicator.hide();
				String historyToken = ruleAnchor + "."
						+ getPackageAnchor(result) + "?" + existingAnchor
						 + "="+ ruleSetId;
				History.newItem(historyToken);
			}

		});

	}

	/**
	 * Loads a existing rule. Use this method if you already know the package
	 * type. It will save one server call.
	 * 
	 * @param pkg
	 * @param ruleSetId
	 */
	public void setViewExistingAnchor(Packages pkg, int ruleSetId) {
		String historyToken = ruleAnchor + "." + getPackageAnchor(pkg) + "?"
				+ existingAnchor + "=" + ruleSetId;
		History.newItem(historyToken);
	}

	private String getPackageAnchor(Packages pkg) {
		if (pkg.equals(Packages.NORMALIZATION)) {
			return Packages.NORMALIZATION.toString();
		} else if (pkg.equals(Packages.VALIDATION)) {
			return Packages.VALIDATION.toString();
		} else if (pkg.equals(Packages.PROCESSING)) {
			return Packages.PROCESSING.toString();
		} else if (pkg.equals(Packages.TARGETING)) {
			return Packages.TARGETING.toString();
		} else if (pkg.equals(Packages.PRIORITIZATION)) {
			return Packages.PRIORITIZATION.toString();
		} else if (pkg.equals(Packages.LICENSING)) {
			return Packages.LICENSING.toString();
		} else if (pkg.equals(Packages.MSOOVERRIDE)) {
			return Packages.MSOOVERRIDE.toString();
		} 
		return "";
	}

	/**
	 * Changes the page and anchor to a listing page based on package type.
	 */
	public void setViewListingAnchor(Packages pkg) {
		History.newItem(ruleAnchor + "." + getPackageAnchor(pkg));
	}

	@Override
	public String getName() {
		return "RULES";
	}

	@Override
	public String getDisplayName() {
		return myConstants.componentDisplayName();
	}

	/**
	 * @return the schedulerLicensed
	 */
	public boolean isSchedulerLicensed() {
		return schedulerLicensed;
	}

	/*
	 * Get the boolean value for a property
	 */
	private boolean getBooleanValue(String property) {
		return Boolean.parseBoolean(info.get(property));
	}

}
