package com.tandbergtv.cms.rules.ui.client.common.rulelisting;

import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;

import com.google.gwt.core.client.GWT;
import com.google.gwt.core.client.Scheduler;
import com.google.gwt.core.client.Scheduler.ScheduledCommand;
import com.google.gwt.event.dom.client.ClickEvent;
import com.google.gwt.event.dom.client.ClickHandler;
import com.google.gwt.event.logical.shared.ResizeEvent;
import com.google.gwt.event.logical.shared.ResizeHandler;
import com.google.gwt.event.shared.HandlerRegistration;
import com.google.gwt.user.client.Window;
import com.google.gwt.user.client.ui.Widget;
import com.tandbergtv.cms.rules.ui.client.RulesComponent;
import com.tandbergtv.cms.rules.ui.client.RulesConstants;
import com.tandbergtv.cms.rules.ui.client.RulesMessages;
import com.tandbergtv.cms.rules.ui.client.UiRulesService;
import com.tandbergtv.cms.rules.ui.client.UiRulesServiceAsync;
import com.tandbergtv.cms.rules.ui.client.common.Permissions;
import com.tandbergtv.cms.rules.ui.client.data.Packages;
import com.tandbergtv.cms.rules.ui.client.data.UIRuleListItem;
import com.tandbergtv.cms.rules.ui.client.data.UIRuleSet;
import com.tandbergtv.cms.rules.ui.client.ruleServiceErrorHandling.ErrorMessageBuilder;
import com.tandbergtv.cms.rules.ui.client.ruleServiceErrorHandling.RuleServiceException;
import com.tandbergtv.neptune.widgettoolkit.client.menu.WidgetMenuItem;
import com.tandbergtv.neptune.widgettoolkit.client.remote.NeptuneAsyncCallback;
import com.tandbergtv.neptune.widgettoolkit.client.security.NeptuneSecurity;
import com.tandbergtv.neptune.widgettoolkit.client.widget.basic.ButtonWidget;
import com.tandbergtv.neptune.widgettoolkit.client.widget.basic.HTMLWidget;
import com.tandbergtv.neptune.widgettoolkit.client.widget.basic.ImageWidget;
import com.tandbergtv.neptune.widgettoolkit.client.widget.composite.BusyIndicator;
import com.tandbergtv.neptune.widgettoolkit.client.widget.composite.HeaderPanel;
import com.tandbergtv.neptune.widgettoolkit.client.widget.composite.messagearea.MessageArea;
import com.tandbergtv.neptune.widgettoolkit.client.widget.composite.orderabletable.OrderableTable;
import com.tandbergtv.neptune.widgettoolkit.client.widget.composite.orderabletable.TableRow;
import com.tandbergtv.neptune.widgettoolkit.client.widget.composite.table.TableConstants;
import com.tandbergtv.neptune.widgettoolkit.client.widget.container.HorizontalContainer;
import com.tandbergtv.neptune.widgettoolkit.client.widget.container.ScrollContainer;
import com.tandbergtv.neptune.widgettoolkit.client.widget.container.TabContainer;
import com.tandbergtv.neptune.widgettoolkit.client.widget.container.VerticalContainer;
import com.tandbergtv.neptune.widgettoolkit.client.widget.dnd.TableRowBeforeMoveEvent;
import com.tandbergtv.neptune.widgettoolkit.client.widget.dnd.TableRowMoveEvent;
import com.tandbergtv.neptune.widgettoolkit.client.widget.dnd.TableRowMoveHandler;
import com.tandbergtv.neptune.widgettoolkit.client.widget.style.StyleNames;

/**
 * Rules listing panel to show the listing, allow users to get details of row
 * 
 * @author rchu
 * 
 */
public class RulesListingPanelBase extends VerticalContainer implements
		WidgetMenuItem.AnchorChangeListener {

	private static final String DISABLED_RULE_IMAGE_URL = "cms_rules_ui/images/disabled_rule.png";
	private static final String ENABLED_RULE_IMAGE_URL = "cms_rules_ui/images/enabled_rule.png";

	private static final String TAB_CONTAINER_STYLE = "rules-tabContainer";

	private static final int CMS_HEADER_MENU_SIZE_PX = 115;

	protected UiRulesServiceAsync rulesService = GWT.create(UiRulesService.class);
	protected RulesConstants constants = RulesComponent.getInstance()
			.getConstants();
	protected RulesMessages messages = (RulesMessages) GWT.create(RulesMessages.class);
	protected OrderableTable table;
	protected ScrollContainer scrollPanel;
	protected TabContainer tabContainer;
	protected MessageArea messageArea;
	protected Packages pkgType;
	protected HorizontalContainer buttonContainer;
	protected BusyIndicator busyIndicator;
	protected ButtonWidget saveOrdbutton;
	protected HandlerRegistration windowRegistration = null;
	protected boolean hasTab = false;
	
	public RulesListingPanelBase(Packages packages, boolean isOrderable) {
		this(packages, isOrderable, false);
	}
	
	/**
	 * creates listing panel based on package type
	 * 
	 * @param packages
	 * @param serviceLoader
	 * @param security
	 */
	public RulesListingPanelBase(Packages packages, boolean isOrderable, boolean hasTab) {
		setStylePrimaryName("rules-RulesListingPanel");

		// setup label to display errors
		this.pkgType = packages;
		this.messageArea = new MessageArea();
		this.buttonContainer = new HorizontalContainer();
		this.table = createTable(isOrderable);
		this.scrollPanel = new ScrollContainer();
		this.tabContainer = new TabContainer();

		createAndAddButtons();

		buttonContainer.setStylePrimaryName("rules-RulesListingPanel-buttonContainer");
		busyIndicator = new BusyIndicator();

		/*
		 * show the busy indicator. In the callback from rules service it will
		 * be hidden
		 */
		busyIndicator.center();
		refreshTable();

		// add widgets to this container
		add(buildHeader(packages));
		add(messageArea);
		if (hasTab) {
			scrollPanel.setWidget(table);
			
			VerticalContainer mainPanel = new VerticalContainer();
			mainPanel.add(scrollPanel);
			mainPanel.add(buttonContainer);
			tabContainer.add(mainPanel, getTabText());
			tabContainer.selectTab(0);
			tabContainer.addStyleName(TAB_CONTAINER_STYLE);
			
			add(tabContainer);

			Scheduler.get().scheduleDeferred(new ScheduledCommand() {
				@Override
				public void execute() {
					updateScrollPanelSize();
				}
			});

		}
		else {
			add(table);
			add(buttonContainer);
		}
		
	}

	/*
	 * Builds the header.
	 */
	private Widget buildHeader(Packages packages) {
		String headerTxt = buildHeaderText(packages);
		return new HeaderPanel(headerTxt);
	}
	
	private String buildHeaderText(Packages packages) {
		String headerTxt = "";
		if (packages.equals(Packages.NORMALIZATION)) {
			headerTxt = constants.normalizationRuleListingHeader();
		} else if (packages.equals(Packages.VALIDATION)) {
			headerTxt = constants.validationRuleListingHeader();
		} else if (packages.equals(Packages.PROCESSING)) {
			headerTxt = constants.contentProccRuleListingHeader();
		} else if (packages.equals(Packages.TARGETING)) {
			headerTxt = constants.targetingRuleListingHeader();
		} else if (packages.equals(Packages.PRIORITIZATION)) {
			headerTxt = constants.prioritizationRuleListingHeader();
		} else if (packages.equals(Packages.LICENSING)) {
			headerTxt = constants.licensingRuleListingHeader();
		} else if (packages.equals(Packages.MSOOVERRIDE)) {
			headerTxt = constants.msoOverideListingHeader();
		}
		return headerTxt;
	}

	private String getTabText() {
		return constants.rulesListingTabHeader();
	}

	/*
	 * Setup a rules listing panel
	 */
	protected OrderableTable createTable(boolean isOrderable) {
		NeptuneSecurity security = RulesComponent.getInstance().getSecurity();
		boolean hasChkBox = security.isUserInRole(Permissions.MODIFY) || security.isUserInRole(Permissions.DELETE);
		
		// create table and add the columns and buttons
		OrderableTable tbl = new OrderableTable("id", isOrderable, hasChkBox, 0);
		tbl.addColumn(constants.ruleListViewRuleSetId(), "id");
		tbl.addColumn("Enabled", "enabled");
		tbl.addColumn(constants.ruleListViewRuleSetName(), "name");
		tbl.addColumn("Update Date", "updateDate");
		return tbl;
	}

	protected void createAndAddButtons() {
		// create click handlers for delete and create buttons
		ClickHandler deleteClick = new ClickHandler() {
			@Override
			public void onClick(ClickEvent event) {
				deleteIds();
			}
		};
		ClickHandler createClick = new ClickHandler() {
			@Override
			public void onClick(ClickEvent event) {
				RulesComponent.getInstance().setNewRuleAnchor(pkgType);
			}
		};

		ClickHandler createEnable = new ClickHandler() {
			@Override
			public void onClick(ClickEvent event) {
				enableRuleSets();
			}
		};

		ClickHandler createDisable = new ClickHandler() {
			@Override
			public void onClick(ClickEvent event) {
				disableRuleSets();
			}
		};

		
		// add buttons depending on security
		NeptuneSecurity security = RulesComponent.getInstance().getSecurity();
		if (security.isUserInRole(Permissions.MODIFY) && table.isReorderable()) {
			createSaveOrderButton(StyleNames.DO_BUTTON_STYLE);
		}
		if (security.isUserInRole(Permissions.MODIFY)) {
			addButton(constants.enable(), StyleNames.DO_BUTTON_STYLE, createEnable);
		}
		if (security.isUserInRole(Permissions.CREATE)) {
			addButton(constants.createRuleSetButtonLabel(), StyleNames.ACT_TOWARDS_SAVE_BUTTON_STYLE, createClick);
		}
		if (security.isUserInRole(Permissions.MODIFY)) {
			addButton(constants.disable(), StyleNames.CAUTION_BUTTON_STYLE, createDisable);
		}
		if (security.isUserInRole(Permissions.DELETE)) {
			addButton(constants.delete(), StyleNames.DATALOSS_BUTTON_STYLE, deleteClick);
		}

	}

	private void createSaveOrderButton(String style) {
		ClickHandler saveOrderHandler = new ClickHandler() {
			@Override
			public void onClick(ClickEvent event) {
				saveListOrder();
			}
		};

		saveOrdbutton = new ButtonWidget(constants.saveOrder(),
				saveOrderHandler);
		saveOrdbutton.addStyleDependentName(style);
		this.buttonContainer.add(saveOrdbutton);

		saveOrdbutton.setVisible(table.isDirty());

		TableRowMoveHandler handler  = new TableRowMoveHandler() {
			@Override
			public void onBeforeRowMove(TableRowBeforeMoveEvent event) {
			}

			@Override
			public void onRowMove(TableRowMoveEvent event) {
				saveOrdbutton.setVisible(true);
			}
		};
		table.addMoveHandler(handler);
	}

	protected void enableRuleSets() {
		List<Integer> ids = table.getSelectedIds();
		if (ids.size() > 0) {
			rulesService.enableRuleSets(ids, new NeptuneAsyncCallback<Object>() {
	
				@Override
				public void onNeptuneFailure(Throwable caught) {
					if (caught instanceof RuleServiceException) {
						List<String> msgs = ErrorMessageBuilder
								.getErrorMessages((RuleServiceException) caught);
						setErrorMessages(msgs);
					} else {
						setErrorMessage(messages.errEnableRuleSets(caught.getLocalizedMessage()));
					}
				}
	
				@Override
				public void onNeptuneSuccess(Object o) {
					refreshTable();
					setInfoMessage(messages.ruleSetsEnabled());
				}
	
			});
		} else {
			setErrorMessage("No rules selected!");
		}
	}

	protected void disableRuleSets() {
		final List<Integer> ids = table.getSelectedIds();
		if (ids.size() > 0) {
			rulesService.disableRuleSets(ids, new NeptuneAsyncCallback<Object>() {
	
				@Override
				public void onNeptuneFailure(Throwable caught) {
					if (caught instanceof RuleServiceException) {
						List<String> msgs = ErrorMessageBuilder
								.getErrorMessages((RuleServiceException) caught);
						setErrorMessages(msgs);
					} else {
						setErrorMessage(
								messages.errDisableRuleSets(caught.getLocalizedMessage()));
					}
				}
	
				@Override
				public void onNeptuneSuccess(Object o) {
					refreshTable();
					setInfoMessage(messages.ruleSetsDisabled());
				}
	
			});
		} else {
			setErrorMessage("No rules selected!");
		}
	}

	/*
	 * Add a button to the bottom of this list. A click handler must be supplied
	 * along with title and style
	 * 
	 * @param title
	 * 
	 * @param style
	 * 
	 * @param click
	 */
	protected void addButton(String title, String style, ClickHandler click) {
		ButtonWidget button = new ButtonWidget(title, click);
		button.addStyleDependentName(style);
		this.buttonContainer.add(button);
	}

	private NeptuneAsyncCallback<Object> getSaveOrderCallback() {
		return new NeptuneAsyncCallback<Object>() {
			@Override
			public void onNeptuneFailure(Throwable caught) {
				setErrorMessage("Unable to save error");
			}

			@Override
			public void onNeptuneSuccess(Object o) {
				saveOrdbutton.setVisible(false);
				setInfoMessage("Order Saved");
			}
		};
	}

	/*
	 * Call back to handle when user deletes rulesets
	 */
	private NeptuneAsyncCallback<List<UIRuleSet>> getDeleteCallback() {
		return new NeptuneAsyncCallback<List<UIRuleSet>>() {
			@Override
			public void onNeptuneFailure(Throwable caught) {
				if (caught instanceof RuleServiceException) {
					List<String> msgs = ErrorMessageBuilder
							.getErrorMessages((RuleServiceException) caught);
					setErrorMessages(msgs);
				} else {
					setErrorMessage("Error Deleting");
				}
			}

			@Override
			public void onNeptuneSuccess(List<UIRuleSet> result) {
				refreshTable();
				setInfoMessage(messages.ruleSetsDeleted());
			}
		};
	}

	protected void refreshTable() {
		messageArea.reset();
		table.clear();
		rulesService.getRuleSetsByPackage(pkgType.toString(),
				getDataRetrieveCallback());
		//if user doesn't have permission to modify then the saveOrdbutton is null 
		if (saveOrdbutton != null) {
			saveOrdbutton.setVisible(false);
		}
	}

	/*
	 * call back to handle listing data retrieval and populate our list
	 */
	private NeptuneAsyncCallback<List<UIRuleListItem>> getDataRetrieveCallback() {
		return new NeptuneAsyncCallback<List<UIRuleListItem>>() {

			@Override
			public void onNeptuneFailure(Throwable caught) {
				setErrorMessage("Data could not be retrieved");
				busyIndicator.hide();
			}

			@Override
			public void onNeptuneSuccess(List<UIRuleListItem> result) {
				populateTable(table, result);
				busyIndicator.hide();
			}
		};
	}

	protected void populateTable(OrderableTable table, List<UIRuleListItem> result) {
		NeptuneSecurity security = RulesComponent.getInstance().getSecurity();

		for (int i = 0; i < result.size(); i++) {
			UIRuleListItem item = result.get(i);
			try {
				ImageWidget enabledImage = getEnabledImage(item
						.isEnabled());

				HTMLWidget id = new HTMLWidget("" + item.getId());
				HTMLWidget name = new HTMLWidget(item.getName());
				HTMLWidget updateDate = new HTMLWidget(item
						.getUpdateDate());

				// link the rules only if view permissions are enabled.
				if (security.isUserInRole(Permissions.VIEW)) {
					id.addClickHandler(getDetailHandler(item.getId()));
					name.addClickHandler(getDetailHandler(item.getId()));

					id.setStyleName(TableConstants.STYLE_DATA_LINK);
					name.setStyleName(TableConstants.STYLE_DATA_LINK);
				}

				table.addItem(i, "id", id, TableConstants.STYLE_DATACELL_NUMERIC);
				table.addItem(i, "enabled", enabledImage, TableConstants.STYLE_DATACELL_ICON);
				table.addItem(i, "name", name, TableConstants.STYLE_DATACELL_TEXT);
				table.addItem(i, "updateDate", updateDate, TableConstants.STYLE_DATACELL_DATE);
			} catch (Exception e) {
				// exception probably means back column name
				String msg = "Error populating table: ";
				setErrorMessage(msg + e.getMessage());
			}
		}
	}
	
	protected ImageWidget getEnabledImage(boolean enabled) {
		String enabledUrl;
		if (enabled) {
			enabledUrl = ENABLED_RULE_IMAGE_URL;
		} else {
			enabledUrl = DISABLED_RULE_IMAGE_URL;
		}

		ImageWidget enabledImage = new ImageWidget(enabledUrl);
		return enabledImage;
	}

	protected ClickHandler getDetailHandler(final int ruleSetId) {
		return new ClickHandler() {
			@Override
			public void onClick(ClickEvent event) {
				//HERE
				RulesComponent.getInstance().setViewExistingAnchor(pkgType, ruleSetId);
			}
		};
	}

	/*
	 * Saves the order of the list
	 */
	private void saveListOrder() {
		Iterator<TableRow> rows = table.rowIterator();
		List<UIRuleListItem> list = new ArrayList<UIRuleListItem>();
		int order = 0;
		while (rows.hasNext()) {
			// exception means cannot parse int and our table is
			// horribly misconfigured and something is wrong with code
			TableRow row = rows.next();
			UIRuleListItem item = new UIRuleListItem();
			item.setId(Integer.parseInt(row.getValue("id")));
			item.setOrder(order++);
			item.setName(row.getValue("name"));
			list.add(item);
		}
		setInfoMessage("Saving order...");
		rulesService.saveRuleSetOrder(pkgType.name(), list,
				getSaveOrderCallback());
		table.resetDirty();
	}

	/*
	 * calls the service to delete this list of IDs
	 */
	private void deleteIds() {
		final List<Integer> ids = table.getSelectedIds();
		if (ids.size() > 0) {
			String msg = "Are you sure you want to delete?";
			if(Window.confirm(msg)) {
				setInfoMessage("Deleting...");
				rulesService.deleteRuleSet(ids, getDeleteCallback());
			}
		} else {
			setErrorMessage("No rules selected!");
		}

	}

	@Override
	public void anchorChanged(String anchor) {
		// TODO Auto-generated method stub
	}
	
	protected void setErrorMessage(String errMsg) {
		messageArea.setErrorMessage(errMsg);
		updateScrollPanelSize();
	}
	
	protected void setInfoMessage(String infoMsg) {
		messageArea.setInfoMessage(infoMsg);
		updateScrollPanelSize();
	}
	
	protected void setErrorMessages(List<String> msgs) {
		String errMsg = null;
		if ((msgs != null) && (msgs.size() > 0)) {
			errMsg = "";
			for (String msg : msgs) {
				errMsg += msg + "\\n";
			}
		}
		
		setErrorMessage(errMsg);
	}
	
	private void updateScrollPanelSize() {
		int height = Window.getClientHeight()
			- CMS_HEADER_MENU_SIZE_PX
			- messageArea.getOffsetHeight()
			- buttonContainer.getOffsetHeight()
			- 90;	//adjustment for borders and spacing at the bottom and header panel at the top
		if (hasTab) {
			height -= tabContainer.getTabBar().getOffsetHeight();
		}
		scrollPanel.setHeight(height + "px");
	}
	
	// ==============================================================
	// ===================== WIDGET OVERRIDES
	// ==============================================================

	@Override
	protected void onLoad() {
		super.onLoad();

		windowRegistration = Window.addResizeHandler(new ResizeHandler() {
			public void onResize(ResizeEvent event) {
				updateScrollPanelSize();
			}
		});
	};

	@Override
	protected void onUnload() {
		windowRegistration.removeHandler();
		windowRegistration = null;

		if (busyIndicator != null)
			busyIndicator.hide();

		super.onUnload();
	}

}
