package com.tandbergtv.cms.rules.ui.client.common.site;

import java.util.ArrayList;
import java.util.LinkedList;
import java.util.List;
import java.util.Queue;

import com.google.gwt.core.client.GWT;
import com.google.gwt.event.dom.client.ClickEvent;
import com.google.gwt.event.dom.client.ClickHandler;
import com.google.gwt.event.logical.shared.ValueChangeEvent;
import com.google.gwt.event.logical.shared.ValueChangeHandler;
import com.google.gwt.event.shared.HandlerRegistration;
import com.google.gwt.user.client.ui.CheckBox;
import com.google.gwt.user.client.ui.HTML;
import com.google.gwt.user.client.ui.HasValue;
import com.google.gwt.user.client.ui.HorizontalPanel;
import com.google.gwt.user.client.ui.Image;
import com.google.gwt.user.client.ui.Label;
import com.google.gwt.user.client.ui.Tree;
import com.google.gwt.user.client.ui.TreeItem;
import com.google.gwt.user.client.ui.VerticalPanel;
import com.tandbergtv.cms.rules.ui.client.RulesComponent;
import com.tandbergtv.cms.rules.ui.client.RulesConstants;
import com.tandbergtv.cms.rules.ui.client.UiRulesSiteFetcher;
import com.tandbergtv.cms.rules.ui.client.UiRulesSiteFetcherAsync;
import com.tandbergtv.neptune.widgettoolkit.client.application.NeptuneApplication;
import com.tandbergtv.neptune.widgettoolkit.client.remote.NeptuneAsyncCallback;
import com.tandbergtv.neptune.widgettoolkit.client.security.NeptuneSecurity;
import com.tandbergtv.neptune.widgettoolkit.client.widget.composite.BusyIndicator;

public class SiteSelector extends VerticalPanel {

	private RulesConstants myConstants = (RulesConstants) GWT
			.create(RulesConstants.class);

	private Tree tree = new Tree();
	private BusyIndicator busyIndicator = new BusyIndicator();
	private UiRulesSiteFetcherAsync siteFetcher = GWT
			.create(UiRulesSiteFetcher.class);

	private NeptuneSecurity security;

	public SiteSelector() {
		busyIndicator.center();
		siteFetcher.getSites(new NeptuneAsyncCallback<List<UISite>>() {

			@Override
			public void onNeptuneFailure(Throwable caught) {
				RulesComponent.getInstance().getCurrentTab().setPanelInError(
						"Error loading sites from server.");
				busyIndicator.hide();
			}

			@Override
			public void onNeptuneSuccess(List<UISite> result) {
				busyIndicator.hide();
				buildUi(result);
			}

		});
	}

	@Override
	protected void onUnload() {
		super.onUnload();
		busyIndicator.hide();
	}

	public SiteSelector(List<UISite> sites) {
		super();
		buildUi(sites);
	}

	private void buildUi(List<UISite> sites) {
		security = NeptuneApplication.getApplication().getSecurity();
		boolean enabled = security.isUserInRole("Sites_Modify");
		populateTree(sites, enabled);
		if (enabled) {
			HorizontalPanel checkPanel = new HorizontalPanel();
			String labelText = myConstants.siteSelectLabel();
			HTML checkLabel = new HTML("<div style=\"margin-right: 5px;\">"
					+ labelText + "</div>");
			checkPanel.add(checkLabel);
			checkPanel.add(buildCheckWidget(myConstants.all(), true));
			checkPanel.add(buildCheckWidget(myConstants.none(), false));
			add(checkPanel);
		}
		add(tree);
		tree.setStylePrimaryName("site-select-tree");
		addStyleName("site-selection-panel");
		fireEvent(new SitesLoadedEvent(hasSites()));
	}

	private Label buildCheckWidget(String text, final boolean checked) {
		Label label = new Label(text);
		label.setStyleName("site-selector-check-all");
		label.addClickHandler(new ClickHandler() {

			@Override
			public void onClick(ClickEvent event) {
				for (int i = 0; i < tree.getItemCount(); i++) {
					TreeItem child = tree.getItem(i);
					((SiteTreeItem) child.getWidget()).setValue(checked, false);
					setChildrenState(child, checked);
				}
			}

		});
		return label;

	}

	private void populateTree(List<UISite> sites, boolean enabled) {
		for (UISite site : sites) {
			TreeItem item = createSiteTreeItem(site, enabled);
			populateTree(item, site.getChildren(), enabled);
			tree.addItem(item);
			((SiteTreeItem)item.getWidget()).setValue(site.isSelected(), true);
		}
		ensureSelectedItemsVisible();
	}

	private void populateTree(TreeItem parent, List<UISite> sites, boolean enabled) {
		for (UISite site : sites) {
			TreeItem item = createSiteTreeItem(site, enabled);
			populateTree(item, site.getChildren(), enabled);
			parent.addItem(item);
			((SiteTreeItem)item.getWidget()).setValue(site.isSelected(), true);
		}
	}
	
	private void ensureSelectedItemsVisible() {
		for(int i = 0; i < tree.getItemCount(); i++) {
			TreeItem child = tree.getItem(i);
			if(!((SiteTreeItem)child.getWidget()).getValue()) {
				child.setState(isChildSelected(child), false);
			}
		}
	}
	
	private boolean isChildSelected(TreeItem treeItem) {
		for(int i=0; i < treeItem.getChildCount(); i++) {
			TreeItem child = treeItem.getChild(i);
			if(((SiteTreeItem)child.getWidget()).getValue()) {
				return true;
			} else if (isChildSelected(child)) {
				treeItem.setState(true,false);
				return true;
			}
		}
		return false;
	}

	private TreeItem createSiteTreeItem(UISite site, boolean enabled) {
		SiteTreeItem siteTreeItem = new SiteTreeItem(site, enabled);
		final TreeItem item = new TreeItem(siteTreeItem);
		item.setUserObject(site);
		siteTreeItem.addValueChangeHandler(new ValueChangeHandler<Boolean>() {

			@Override
			public void onValueChange(ValueChangeEvent<Boolean> event) {
				if (!event.getValue()) {
					setParentState(item, false);
				}
				setChildrenState(item, event.getValue());
			}

		});
		return item;
	}

	private void setParentState(TreeItem item, boolean value) {
		TreeItem parent = item.getParentItem();
		if (parent != null) {
			((SiteTreeItem) parent.getWidget()).setValue(value, false);
			setParentState(parent, value);
		}

	}

	private void setChildrenState(TreeItem item, boolean value) {
		for (int i = 0; i < item.getChildCount(); i++) {
			TreeItem child = item.getChild(i);
			((SiteTreeItem) child.getWidget()).setValue(value, false);
			setChildrenState(child, value);
		}
	}
	
	public boolean hasSelectedSites() {
		Queue<TreeItem> ll = new LinkedList<TreeItem>();
		for (int i = 0; i < tree.getItemCount(); i++) {
			TreeItem item = tree.getItem(i);
			if(((SiteTreeItem) item.getWidget()).getValue())
				return true;
			else {
				for (int j = 0; j < item.getChildCount(); j++) {
					ll.add(item.getChild(j));
				}
			}
		}
		while(!ll.isEmpty()) {
			TreeItem item = ll.remove();
			if(((SiteTreeItem) item.getWidget()).getValue())
				return true;
			else {
				for (int j = 0; j < item.getChildCount(); j++) {
					ll.add(item.getChild(j));
				}
			}
		}
		return false;
	}

	public List<UISite> getSelectedSites() {
		List<UISite> sites = new ArrayList<UISite>();
		for (int i = 0; i < tree.getItemCount(); i++) {
			recurseTree(tree.getItem(i), sites);
		}
		return sites;
	}

	private void recurseTree(TreeItem item, List<UISite> sites) {
		if (((SiteTreeItem) item.getWidget()).getValue()) {
			sites.add((UISite) item.getUserObject());
		} else {
			for (int i = 0; i < item.getChildCount(); i++) {
				recurseTree(item.getChild(i), sites);
			}
		}
	}
	
	public boolean hasSites() {
		return tree.getItemCount() > 0;
	}
	
	public HandlerRegistration registerSiteLoadedEvent(SitesLoadedHandler handler) {
		return addHandler(handler, SitesLoadedEvent.TYPE);
	}
	
	private class SiteTreeItem extends HorizontalPanel implements HasValue<Boolean> {
		private static final String ENABLED_LOGICAL_SITE_URL = "cms_rules_ui/images/logicalSite.gif";
		private static final String DISABLED_LOGICAL_SITE_URL = "cms_rules_ui/images/logicalSiteDisabled.gif";
		private static final String ENABLED_DISTRIBUTION_SITE_URL = "cms_rules_ui/images/distributionSite.gif";
		private static final String DISABLED_DISTRIBUTION_SITE_URL = "cms_rules_ui/images/distributionSiteDisabled.gif";
		

		private CheckBox checkBox;

		public SiteTreeItem(UISite site, boolean enabled) {
			super();
			String name = site.getName();
			checkBox = new CheckBox("");
			checkBox.setEnabled(enabled);
			add(checkBox);
			Image image = getImage(site);
			image.addStyleName("site-tree-item-image");
			add(image);
			add(new Label(name));
		}
		
		private Image getImage(UISite site) {
			Image image = null;
			if(site.isActive() && site.getSiteType().equals(SiteType.LOGICAL)) {
				image = new Image(ENABLED_LOGICAL_SITE_URL);
			} else if(site.isActive() && site.getSiteType().equals(SiteType.DISTRIBUTION)) {
				image = new Image(ENABLED_DISTRIBUTION_SITE_URL);
			} else if(!site.isActive() && site.getSiteType().equals(SiteType.LOGICAL)) {
				image = new Image(DISABLED_LOGICAL_SITE_URL);
			} else if(!site.isActive() && site.getSiteType().equals(SiteType.DISTRIBUTION)) {
				image = new Image(DISABLED_DISTRIBUTION_SITE_URL);
			}
			
			if (site.getSiteType().equals(SiteType.LOGICAL)) {
				image.setTitle(myConstants.logicalSite());
			} else if (site.getSiteType().equals(SiteType.DISTRIBUTION)) {
				image.setTitle(myConstants.distributionSite());
			}
			return image;
		}

		@Override
		public Boolean getValue() {
			return checkBox.getValue();
		}

		@Override
		public void setValue(Boolean value) {
			checkBox.setValue(value);
			
		}

		@Override
		public void setValue(Boolean value, boolean fireEvents) {
			checkBox.setValue(value, fireEvents);
		}

		@Override
		public HandlerRegistration addValueChangeHandler(
				ValueChangeHandler<Boolean> handler) {
			return checkBox.addValueChangeHandler(handler);
		}
	}

}
