package com.tandbergtv.cms.rules.ui.client.normalization;

import java.util.ArrayList;
import java.util.HashSet;
import java.util.List;
import java.util.Set;

import com.google.gwt.event.dom.client.ChangeEvent;
import com.google.gwt.event.dom.client.ChangeHandler;
import com.google.gwt.event.dom.client.ClickEvent;
import com.google.gwt.event.dom.client.ClickHandler;
import com.google.gwt.event.shared.HandlerRegistration;
import com.google.gwt.user.client.ui.Widget;
import com.tandbergtv.cms.portal.ui.title.client.criteria.CriteriaListingPanel;
import com.tandbergtv.cms.portal.ui.title.client.criteria.CriteriaPanelType;
import com.tandbergtv.cms.portal.ui.title.client.criteria.CriteriaType;
import com.tandbergtv.cms.portal.ui.title.client.criteria.PredicateRow;
import com.tandbergtv.cms.portal.ui.title.client.criteria.UiCriteriaMode;
import com.tandbergtv.cms.portal.ui.title.client.criteria.FieldTree.IFieldTreeObject;
import com.tandbergtv.cms.portal.ui.title.client.model.search.RulesOperatorFull;
import com.tandbergtv.cms.portal.ui.title.client.model.search.UITitleFilterListNode;
import com.tandbergtv.cms.rules.ui.client.RulesComponent;
import com.tandbergtv.cms.rules.ui.client.RulesConstants;
import com.tandbergtv.cms.rules.ui.client.common.events.RuleDeleteEvent;
import com.tandbergtv.cms.rules.ui.client.common.events.RuleDeleteHandler;
import com.tandbergtv.cms.rules.ui.client.data.UIAction;
import com.tandbergtv.cms.rules.ui.client.data.UiStandardRule;
import com.tandbergtv.neptune.widgettoolkit.client.widget.basic.ButtonWidget;
import com.tandbergtv.neptune.widgettoolkit.client.widget.basic.LabelWidget;
import com.tandbergtv.neptune.widgettoolkit.client.widget.basic.TextBoxWidget;
import com.tandbergtv.neptune.widgettoolkit.client.widget.container.SimpleContainer;
import com.tandbergtv.neptune.widgettoolkit.client.widget.container.VerticalContainer;
import com.tandbergtv.neptune.widgettoolkit.client.widget.style.StyleNames;

/**
 * This represents a single Normalization Rule within the UI and should be
 * contained within the NormalizationRulePanel
 * 
 * @author dweiner
 * 
 */
public class NormalizationRule extends CriteriaListingPanel implements INormalizationRuleForm, IMatchedItemProvider {

	// Tree of conditions
	// List of actions
	private RulesConstants myConstants = RulesComponent.getInstance()
			.getConstants();
	private UiStandardRule rule;
	private VerticalContainer actionBoxes = new VerticalContainer();

	private Set<IFieldTreeObject> variables = new HashSet<IFieldTreeObject>();
	private TextBoxWidget indexWidget;
	private int index;
	private boolean sourceIsMso;

	public NormalizationRule(int index, boolean enableReordering, boolean sourceIsMso) {
		this(new UiStandardRule(), index, enableReordering, sourceIsMso);
	}

	public NormalizationRule(UiStandardRule rule,
			int index, boolean enableReordering, boolean sourceIsMso) {
		super(UiCriteriaMode.EPISODE);
		maxLevels = 1;
		this.sourceIsMso = sourceIsMso;
		this.operatorProvider = new RulesOperatorFull();
		this.rule = rule;
		this.index = index;
		putTheUiTogether();
		enableReorder(enableReordering);
	}

	/**
	 * Returns the rule. If the rule is empty it will return null.
	 * 
	 * @return
	 */
	public UiStandardRule getRule() {
		UiStandardRule result = rule == null ? new UiStandardRule() : rule;

		// Set the condtion
		UITitleFilterListNode list = getCriteria();
		result.setUiTitleFilterNode(list);

		// Set the actions
		List<UIAction> actions = new ArrayList<UIAction>();
		for (Widget w : actionBoxes) {
			NormalizationActionBox abox = (NormalizationActionBox) w;
			actions.add(abox.getAction());
		}
		result.setActions(actions);

		return result;
	}

	private Widget buildReorder() {
		SimpleContainer reorderContainer = new SimpleContainer();
		reorderContainer.setStyleName("rule-order-pos");

		// 4chars == MAX 999 rules
		indexWidget = new TextBoxWidget((index + 1) + "");
		indexWidget.setVisibleLength(4);
		indexWidget.setMaxLength(4);
		indexWidget.addChangeHandler(new ChangeHandler() {
			@Override
			public void onChange(ChangeEvent event) {
				String text = indexWidget.getText();
				int newIndex;
				try {
					newIndex = Integer.parseInt(text) - 1;
				} catch (NumberFormatException ex) {
					indexWidget.setText((index + 1) + "");
					return;
				}
				
				fireEvent(new RuleReorderEvent(NormalizationRule.this, newIndex));
			}
		});

		reorderContainer.add(indexWidget);
		return reorderContainer;
	}

	public void enableReorder(boolean reorderEnabled) {
		// if the ruleEnabled is not true set read only
		indexWidget.setReadOnly(!reorderEnabled);
		String style = reorderEnabled ? "rule-order-editable"
				: "rule-order-noneditable";
		indexWidget.setStyleName(style);
	}

	private void putTheUiTogether() {

		getContainer().clear();

		// This sets the border for each rule

		getContainer().add(buildReorder());
		getContainer().add(super.getWarningMessageBox());
		getContainer().add(
				new LabelWidget(myConstants.normalizationPanelLabel1()));

		ButtonWidget delButton = deleteButton();
		initializeTree(this.rule.getUiTitleFilterNode());
		getContainer().add(
				new LabelWidget(myConstants.normalizationPanelLabel2()));
		getContainer().add(actionBoxes);
		getContainer().add(delButton);

		initializeActions();

		// SET STYLES
		getContainer().setStyleName(CRITERIA_PANEL_CONTAINER_STYLE);
		setStyleName(CRITERIA_PANEL_STYLE);
	}
	/*
	 * 
	 */
	private void initializeActions() {
		if (rule.getActions() == null || rule.getActions().size() == 0) {
			// brand new action for new rule
			addAction();
		} else {
			// go through and add in all the actions
			for (UIAction uiaction : rule.getActions()) {
				NormalizationActionBox abox = new NormalizationActionBox(uiaction, this);
				addAction(abox);
			}
		}
	}

	public void addAction() {
		NormalizationActionBox abox = new NormalizationActionBox(this);
		actionBoxes.add(abox);
	}

	public void addActionAfter(NormalizationActionBox actionBox) {
		NormalizationActionBox abox = new NormalizationActionBox(this);
		int index = actionBoxes.getWidgetIndex(actionBox)+1;
		actionBoxes.insert(abox,index);
	}

	private void addAction(NormalizationActionBox abox) {
		actionBoxes.add(abox);
	}

	public void removeAction(NormalizationActionBox abox) {
		actionBoxes.remove(abox);
		if (actionBoxes.getWidgetCount() == 0)
			addAction();
	}

	@Override
	public boolean validate() {
		boolean result = super.validate();
		for (Widget w : actionBoxes) {
			NormalizationActionBox abox = (NormalizationActionBox) w;
			if (!abox.validate())
				result = false;
		}
		return result;
	}

	/*
	 * create a button to delete this rule
	 */
	private ButtonWidget deleteButton() {
		ButtonWidget button = new ButtonWidget(myConstants.delete(),
				new ClickHandler() {
					public void onClick(ClickEvent event) {
						fireEvent(new RuleDeleteEvent(NormalizationRule.this)); 
					}
				});
		button.addStyleDependentName(StyleNames.DATALOSS_BUTTON_STYLE);
		return button;
	}

	/**
	 * determines whether a predicate row can be deleted from the criteria
	 * listing panel.
	 * 
	 * @param row
	 *            (&& )
	 * @return
	 */
	@Override
	public boolean canChangePredicateRow(PredicateRow row) {

		/*
		 * The goal here is to determine if this predicate row is referenced by
		 * an action. For example if there is one predicate row with field
		 * assetNameand an action references assetName as a Matched Item than it
		 * can't bedeleted until no actions reference it. However if there are
		 * two predicaterows with assetName than we are all go for a delete
		 * since the action stillmakes sense.
		 */

		// determine if there are multiple actions with a reference to that
		// field
		String msg = myConstants.ruleValidationRemoveActionBeforeChange();
		removeWarningMessage(msg);
		if (getRoot().predicateRowsWithTreeObject(row.getLhsTreeObject()) > 1)
			return true;
		// if only one action references it than make sure no actions reference
		// it
		if(actionHasReferencesForField(row.getLhsTreeObject())) {
			addWarningMessage(msg);
			return false;
		}
		return true;
	}
	
	public boolean actionHasReferencesForField(IFieldTreeObject field) {
		for (Widget w : actionBoxes) {
			NormalizationActionBox abox = (NormalizationActionBox) w;
			if (!abox.canRemoveReferencesForField(field)) {
				return true;
			}
		}
		return false;
	}

	@Override
	public void notifyChange(IFieldTreeObject newItem) {
		// add the new variable
		if (newItem.supportsMatched()
				&& (newItem.supportsActions() || newItem.childSupportsActions())) {
			variables.add(newItem);
		}
	}

	@Override
	public void notifyRemoval(IFieldTreeObject treeObject) {
		// remove the old object
		if (treeObject != null && !this.getRoot().contains(treeObject)) {
			variables.remove(treeObject);
		}
	}

	public void setCurrentIndex(int index) {
		this.index = index;
		indexWidget.setText((index + 1) + "");
	}

	public int getIndex() {
		return index;
	}
	
	public HandlerRegistration addDeleteHandler(RuleDeleteHandler handler) {
		return addHandler(handler, RuleDeleteEvent.TYPE);
	}

	@Override
	public HandlerRegistration addReorderHandler(RuleReorderHandler handler) {
		return addHandler(handler, RuleReorderEvent.TYPE);
	}

	@Override
	public CriteriaPanelType getCriteriaPanelType() {
		return CriteriaPanelType.RULES;
	}

	public Set<IFieldTreeObject> getVariables() {
		return variables;
	}

	public CriteriaType getActionLhsCriteriaType() {
		return sourceIsMso ? CriteriaType.MSO_ACTION_LHS : CriteriaType.NORM_ACTION_LHS;
	}
}
