package com.tandbergtv.cms.rules.ui.client.validation;

import java.util.ArrayList;
import java.util.List;

import com.google.gwt.core.client.GWT;
import com.google.gwt.event.dom.client.ChangeEvent;
import com.google.gwt.event.dom.client.ChangeHandler;
import com.google.gwt.event.dom.client.ClickEvent;
import com.google.gwt.event.dom.client.ClickHandler;
import com.google.gwt.event.shared.HandlerRegistration;
import com.google.gwt.user.client.ui.Button;
import com.google.gwt.user.client.ui.HorizontalPanel;
import com.google.gwt.user.client.ui.Label;
import com.google.gwt.user.client.ui.TextBox;
import com.tandbergtv.cms.portal.ui.title.client.TitleStyleNames;
import com.tandbergtv.cms.portal.ui.title.client.criteria.CriteriaListingPanel;
import com.tandbergtv.cms.portal.ui.title.client.criteria.UiCriteriaMode;
import com.tandbergtv.cms.portal.ui.title.client.criteria.CriteriaPanelType;
import com.tandbergtv.cms.portal.ui.title.client.model.search.IUITitleFilterNode;
import com.tandbergtv.cms.portal.ui.title.client.model.search.RulesOperatorFull;
import com.tandbergtv.cms.portal.ui.title.client.model.search.UITitleFilterListNode;
import com.tandbergtv.cms.rules.ui.client.RulesComponent;
import com.tandbergtv.cms.rules.ui.client.RulesConstants;
import com.tandbergtv.cms.rules.ui.client.common.IRuleForm;
import com.tandbergtv.cms.rules.ui.client.common.events.RuleDeleteEvent;
import com.tandbergtv.cms.rules.ui.client.common.events.RuleDeleteHandler;
import com.tandbergtv.cms.rules.ui.client.data.ActionParams;
import com.tandbergtv.cms.rules.ui.client.data.UIAction;
import com.tandbergtv.cms.rules.ui.client.data.UIActionParam;
import com.tandbergtv.cms.rules.ui.client.data.UiRule;
import com.tandbergtv.cms.rules.ui.client.data.UiStandardRule;
import com.tandbergtv.neptune.widgettoolkit.client.widget.basic.ListBoxWidget;
import com.tandbergtv.neptune.widgettoolkit.client.widget.style.StyleNames;

public class ValidationRule extends CriteriaListingPanel implements IRuleForm {

	private static final String MESSAGE_STYLE = "big-txt-box";
	private static final String SMALL_TEXT = "small-txt";
	private RulesConstants myConstants = (RulesConstants) GWT
			.create(RulesConstants.class);

	private TextBox actionText;
	private ListBoxWidget<UIAction> actionsList;
	private int ruleId = -1;

	private static List<UIAction> actions;

	/**
	 * Constuctor used when loading a brand new rule.
	 * 
	 * @param parentPanel
	 */
	public ValidationRule(UiCriteriaMode critMode) {
		super(critMode);
		this.operatorProvider = new RulesOperatorFull();
		this.actionText = new TextBox();
		this.actionText.addStyleName(MESSAGE_STYLE);
		this.actionText.addStyleName(SMALL_TEXT);

		//no condition exists because this is a brand new rule so pass null
		putTheUiTogether(null);
	}

	/**
	 * Constructor used when loading a pre-existing rule.
	 * @param contentClasses 
	 * 
	 * @param condition
	 *            The condition associated with the rule.
	 * @param action
	 *            The action associated with the rule.
	 * @param parentPanel
	 *            the ValidationCriteriaPanel that contains this object
	 */
	public ValidationRule(UiCriteriaMode critMode, UiStandardRule rule) {
		super(critMode);
		this.ruleId = rule.getId();
		this.operatorProvider = new RulesOperatorFull();
		this.actionText = new TextBox();
		this.actionText.addStyleName(MESSAGE_STYLE);
		this.actionText.addStyleName(SMALL_TEXT);
		// For validation the number of actions should be 1
		if (rule.getActions().size() == 1) {
			UIAction action = rule.getActions().get(0);
			putTheUiTogether(rule.getUiTitleFilterNode());
			setActionSelected(action);
		} else {
			String msg = "Unexpected error: validation can only have 1 action.";
			RulesComponent.getInstance().getCurrentTab().setPanelInError(msg);
		}
	}

	/**
	 * UI externally depends on a rule and list of available actions before it
	 * can proceed. This does not occur until action async callback happens
	 * 
	 * @param condition
	 *            the condition to be displayed. When null it is assumed this is
	 *            a new condition.
	 */
	private void putTheUiTogether(IUITitleFilterNode condition) {

		// populate action list box
		initializeActionDropDown();

		getContainer().add(super.getWarningMessageBox());
		getContainer().add(new Label(myConstants.validationActionLabel()));
		
		initializeTree(condition); // Put the Tree in

		// action selection row
		HorizontalPanel messageRow = new HorizontalPanel();
		messageRow.add(new Label(myConstants.returnThis() + " "));
		messageRow.add(actionsList);
		messageRow.add(new Label(myConstants.message() + " "));
		messageRow.add(actionText);

		Button delButton = deleteButton();

		getContainer().add(messageRow);
		getContainer().add(delButton);

		//SET STYLES
		getContainer().setStyleName(CRITERIA_PANEL_CONTAINER_STYLE);
		setStyleName(CRITERIA_PANEL_STYLE);
	}

	/**
	 * Initializes the action drop down using the action cache
	 */
	private void initializeActionDropDown() {
		actionsList = new ListBoxWidget<UIAction>();
		actionsList.addStyleName(SMALL_TEXT);
		for (UIAction action : actions) {
			String label = myConstants.getString(action.getLabelKey());
			actionsList.addItem(label, action);
		}
	}

	/**
	 * on loading a ruleset this is a helper to load the action. Must be called
	 * after populateUi because it depends on the actionsList being initialized.
	 * 
	 * @param action
	 * 
	 */
	private void setActionSelected(UIAction action) {
		if (action.containsParam(ActionParams.MESSAGE)) {
			UIActionParam param = action.getParam(ActionParams.MESSAGE);
			actionText.setText(param.getValue());
		}
		actionsList.setSelectedItem(action);
	}

	@Override
	public UiRule getRule() {
		UITitleFilterListNode tree = getCriteria();
		UIAction action = actionsList.getSelectedItem().clone();

		if (action.containsParam(ActionParams.MESSAGE)) {
			UIActionParam param = action.getParam(ActionParams.MESSAGE);
			param.setValue(actionText.getText());
		}

		List<UIAction> actions = new ArrayList<UIAction>();
		actions.add(action);
		return new UiStandardRule(ruleId, tree, actions);
		
	}

	/**
	 * create a button to delete this rule
	 */
	private Button deleteButton() {
		Button button = new Button(myConstants.delete(),
				new ClickHandler() {
					public void onClick(ClickEvent event) {
						fireEvent(new RuleDeleteEvent(ValidationRule.this));
					}
				});
		button.addStyleDependentName(StyleNames.DATALOSS_BUTTON_STYLE);
		return button;
	}
	
	public HandlerRegistration addDeleteHandler(RuleDeleteHandler handler) {
		return addHandler(handler, RuleDeleteEvent.TYPE);
	}

	@Override
	public boolean validate() {
		boolean result = super.validate();
		if (actionText.getText().equals("")) {
			this.addWarningMessage(myConstants.ruleValidationBlankMessage());
			actionText.addStyleName(TitleStyleNames.PINK_BOX_STYLE);
			actionText.addChangeHandler(new ActionTextChangeHandler());
			result = false;
		}
		else if (actionText.getText().length() > 1024) {
			this.addWarningMessage(myConstants.ruleValidationTooLongMessage());
			actionText.addStyleName(TitleStyleNames.PINK_BOX_STYLE);
			actionText.addChangeHandler(new ActionTextChangeHandler());
			result = false;
		}
		return result;
	}

	/**
	 * sets the cached actions. Setter should be called from ValidationRulePanel
	 * 
	 * @param cachedActions
	 *            the action list to be stored in the cache.
	 */
	public static void setCachedActions(List<UIAction> cachedActions) {
		actions = cachedActions;
	}
	
	public static List<UIAction> getCachedActions() {
		return actions;
	}

	/**
	 * returns the cached actions. Setter should be called from
	 * ValidationRulePanel
	 * 
	 * @return the cached actionList or null if it hasn't been loaded yet.
	 */
	public static boolean cacheLoaded() {
		return actions != null;
	}

	@Override
	public CriteriaPanelType getCriteriaPanelType() {
		return CriteriaPanelType.RULES;
	}
	
	private class ActionTextChangeHandler implements ChangeHandler {

		@Override
		public void onChange(ChangeEvent event) {
			clearWarningMessages();
			if (actionText.getText().equals("")) {
				addWarningMessage(myConstants
						.ruleValidationBlankMessage());
				actionText.addStyleName(TitleStyleNames.PINK_BOX_STYLE);
			} else if (actionText.getText().length() > 1024) {
				addWarningMessage(myConstants
						.ruleValidationTooLongMessage());
				actionText.addStyleName(TitleStyleNames.PINK_BOX_STYLE);
			} else {
				actionText.removeStyleName(TitleStyleNames.PINK_BOX_STYLE);
			}
		}
		
	}
}
