package com.ericsson.cms.scheduling.ui.client;

import java.util.ArrayList;
import java.util.HashSet;
import java.util.List;
import java.util.Set;

import com.ericsson.cms.scheduling.ui.client.common.Common;
import com.ericsson.cms.scheduling.ui.client.common.SchedulingConstants;
import com.ericsson.cms.scheduling.ui.client.data.UIWOJob;
import com.ericsson.cms.scheduling.ui.client.service.IUISchedulerService;
import com.ericsson.cms.scheduling.ui.client.service.IUISchedulerServiceAsync;
import com.ericsson.cms.scheduling.ui.client.widgets.SchedulingWidget;
import com.google.gwt.core.client.GWT;
import com.google.gwt.event.dom.client.ClickEvent;
import com.google.gwt.event.dom.client.ClickHandler;
import com.google.gwt.event.logical.shared.ValueChangeEvent;
import com.google.gwt.event.logical.shared.ValueChangeHandler;
import com.google.gwt.user.client.Element;
import com.google.gwt.user.client.History;
import com.google.gwt.user.client.Window;
import com.google.gwt.user.client.ui.Composite;
import com.google.gwt.user.client.ui.FlexTable;
import com.google.gwt.user.client.ui.ScrollPanel;
import com.tandbergtv.neptune.widgettoolkit.client.application.NeptuneApplication;
import com.tandbergtv.neptune.widgettoolkit.client.remote.NeptuneAsyncCallback;
import com.tandbergtv.neptune.widgettoolkit.client.security.NeptuneSecurity;
import com.tandbergtv.neptune.widgettoolkit.client.widget.basic.ButtonWidget;
import com.tandbergtv.neptune.widgettoolkit.client.widget.basic.CheckBoxWidget;
import com.tandbergtv.neptune.widgettoolkit.client.widget.basic.ImageWidget;
import com.tandbergtv.neptune.widgettoolkit.client.widget.basic.LabelWidget;
import com.tandbergtv.neptune.widgettoolkit.client.widget.composite.BusyIndicator;
import com.tandbergtv.neptune.widgettoolkit.client.widget.composite.messagearea.MessageArea;
import com.tandbergtv.neptune.widgettoolkit.client.widget.container.HorizontalContainer;
import com.tandbergtv.neptune.widgettoolkit.client.widget.container.SimpleContainer;
import com.tandbergtv.neptune.widgettoolkit.client.widget.container.VerticalContainer;
import com.tandbergtv.neptune.widgettoolkit.client.widget.style.StyleNames;

public class FilteredSchedules extends Composite {
	private static final String ENABLED_IMAGE = "cms_scheduling_ui/images/enabled.png";
	private static final String DISABLED_IMAGE = "cms_scheduling_ui/images/disabled.png";

	private static final String ERROR_MESSAGE_STYLE = "scheduler-errorMessage";
	private static final String INFO_MESSAGE_STYLE = "scheduler-infoMessage";
	private static final String TABLE_STYLE = "scheduler-schedulesList";
	private static final String HEADER_ROW_STYLE = "scheduler-schedulesList-header";
	private static final String DATA_ROW_STYLE = "scheduler-schedulesList-data";
	private static final String ENABLED_COL_STYLE = "scheduler-schedulesList-col-enabled";
	private static final String ID_COL_STYLE = "scheduler-schedulesList-col-id";
	private static final String ID_LINK_COL_STYLE = "scheduler-schedulesList-col-id-link";

	private static final String ANCHOR_PARAM_ASSOCIATIONTYPE = "associationType";
	private static final String ANCHOR_PARAM_ASSOCIATIONID = "associationId";
	
	private static final String STYLE_SCROLL_CONTAINER = "sites-jobsTab-scrollContainer";

	private IUISchedulerServiceAsync service = GWT.create(IUISchedulerService.class);
	private SchedulingConstants constants = GWT.create(SchedulingConstants.class);

	private BusyIndicator busyIndicator = new BusyIndicator(true);
	
	private VerticalContainer mainPanel = new VerticalContainer();
	private MessageArea messageArea;
	private SimpleContainer tableContainer = new SimpleContainer();
	private List<CheckBoxWidget> rowCheckBoxes;

	private Set<Long> selectedIds = new HashSet<Long>();
	
	private String entityType;
	private Long entityId;
	
	HorizontalContainer buttonPanel;
	ButtonWidget dissociateButton, createButton;
	
	private ScrollPanel scrollPanel;

	public FilteredSchedules() {		
	    scrollPanel = new ScrollPanel();
	    scrollPanel.addStyleName(STYLE_SCROLL_CONTAINER);
	    scrollPanel.add(tableContainer);
	    
	    mainPanel.setSpacing(10);
		mainPanel.add(scrollPanel);
		
		buttonPanel = new HorizontalContainer();
		mainPanel.add(buttonPanel);
		
		//create button		
		createButton = new ButtonWidget(constants.buttonLabelCreate());
		createButton.addStyleDependentName(StyleNames.ACT_TOWARDS_SAVE_BUTTON_STYLE);
		createButton.addClickHandler(new ClickHandler() {
			@Override
			public void onClick(ClickEvent event) {
				String anchor = "Scheduler.View.Create";
				if ((entityType != null) && (entityId != null)) {
					anchor += "?";
					anchor += ANCHOR_PARAM_ASSOCIATIONTYPE;
					anchor += "=";
					anchor += entityType;
					anchor += "&";
					anchor += ANCHOR_PARAM_ASSOCIATIONID;
					anchor += "=";
					anchor += entityId;
				}
				History.newItem(anchor);
			}});
		buttonPanel.add(createButton);		

		//dissociate button		
		dissociateButton = new ButtonWidget(constants.buttonLabelDissociate());
		dissociateButton.addStyleDependentName(StyleNames.DATALOSS_BUTTON_STYLE);
		dissociateButton.addClickHandler(new ClickHandler() {
			@Override
			public void onClick(ClickEvent event) {
				if (selectedIds.size() > 0) {
					service.dissociate(selectedIds, entityId,
							new NeptuneAsyncCallback<Void>() {
								@Override
								public void onNeptuneFailure(Throwable caught) {
									setErrorMessage("Failed to dissociate the selected jobs."
											+ Common.getFailureMessageSuffix(caught));
								}

								@Override
								public void onNeptuneSuccess(Void result) {
									setInfoMessage("Successfully dissociated the jobs.");
									refresh();
								}
							});
				}
				else {
					setErrorMessage(constants.noScheduleSelected());
				}
			}});
		buttonPanel.add(dissociateButton);

		initWidget(mainPanel);
	}

	public void setEntity(String entityType, Long entityId) {
		this.entityType = entityType;
		this.entityId = entityId;
		NeptuneSecurity security = NeptuneApplication.getApplication().getSecurity();
		if (!(security.isUserInRole(com.ericsson.cms.scheduling.ui.client.Permissions.CREATE) 
				&& Common.isUserAuthorized(entityType, Action.CREATE))) {
			this.createButton.setVisible(false);
		}
		if (!(security.isUserInRole(com.ericsson.cms.scheduling.ui.client.Permissions.DELETE) 
				&& Common.isUserAuthorized(entityType, Action.DELETE))) {
			this.dissociateButton.setVisible(false);
		}
		refresh();
	}
	
	/**
	 * @param messageArea the messageArea to set
	 */
	public void setMessageArea(MessageArea messageArea) {
		this.messageArea = messageArea;
	}

	public void clearMessages() {
		messageArea.reset();
		updateSizeForMessageArea();
	}

	private void refresh() {
		load(entityType, entityId);
	}
	
	private void load(String entityType, long entityId) {
		busyIndicator.center();
		service.getSchedulesByAssociatedEntity(entityType, entityId, new NeptuneAsyncCallback<List<UIWOJob>>() {
			@Override
			public void onNeptuneFailure(Throwable caught) {
				busyIndicator.hide();
				Window.alert("Failed to load schedules list." + Common.getFailureMessageSuffix(caught));
			}

			@Override
			public void onNeptuneSuccess(List<UIWOJob> uiWOJobs) {
				rowCheckBoxes = new ArrayList<CheckBoxWidget>(uiWOJobs.size());
				selectedIds.clear();
				buildTable(uiWOJobs);
				busyIndicator.hide();
			}
		});
	}
	
	private void buildTable(List<UIWOJob> jobs) {
		FlexTable table = new FlexTable();
		table.addStyleName(TABLE_STYLE);
		tableContainer.setWidget(table);

		table.getRowFormatter().addStyleName(0, HEADER_ROW_STYLE);

		final CheckBoxWidget allCB = new CheckBoxWidget();
		allCB.addValueChangeHandler(new ValueChangeHandler<Boolean>() {
			@Override
			public void onValueChange(ValueChangeEvent<Boolean> event) {
				if(rowCheckBoxes != null) {
					for(CheckBoxWidget cb : rowCheckBoxes) {
						cb.setValue(event.getValue(), true);
					}
				}
			}});
		table.setWidget(0, 0, allCB);
		table.getCellFormatter().addStyleName(0, 0, "sorttable_nosort");
		table.setText(0, 1, constants.columnLabelId());
		table.setText(0, 2, constants.columnLabelActive());
		table.setText(0, 3, constants.columnLabelName());
		table.setText(0, 4, constants.columnLabelTemplateName());
		table.setText(0, 5, constants.columnLabelJobScheduleRule());
		
		int rowIndex = 1;
		for(final UIWOJob job : jobs) {
			table.getRowFormatter().addStyleName(rowIndex, DATA_ROW_STYLE);
			CheckBoxWidget cb = new CheckBoxWidget();
			cb.addValueChangeHandler(new ValueChangeHandler<Boolean>() {
				@Override
				public void onValueChange(ValueChangeEvent<Boolean> event) {
					if(event.getValue()) {
						selectedIds.add(job.getId());
					} else {
						allCB.setValue(false, false);
						selectedIds.remove(job.getId());
					}
				}});
			table.setWidget(rowIndex, 0, cb);
			rowCheckBoxes.add(cb);
			
			LabelWidget idWidget = new LabelWidget(Long.toString(job.getJobId()));
			idWidget.addStyleName(ID_COL_STYLE);
			idWidget.addStyleName(ID_LINK_COL_STYLE);
			idWidget.addClickHandler(new ClickHandler() {
				@Override
				public void onClick(ClickEvent event) {
					History.newItem("Scheduler.View?id=" + job.getJobId());
				}
			});
			table.setWidget(rowIndex, 1, idWidget);
			
			ImageWidget enabledImageWidget = new ImageWidget((job.isActive() ? ENABLED_IMAGE : DISABLED_IMAGE));
			enabledImageWidget.addStyleName(ENABLED_COL_STYLE);
			table.setWidget(rowIndex, 2, enabledImageWidget);

			table.setWidget(rowIndex, 3, new LabelWidget(job.getName()));
			table.setWidget(rowIndex, 4, new LabelWidget(job.getTemplateName()));
			String scheduleInfoString = SchedulingWidget.getScheduleInfoString(job.getScheduleInfo());
			table.setWidget(rowIndex, 5, new LabelWidget(scheduleInfoString));
			rowIndex++;
		}
		
		enableSort(table.getElement());		
	}
	
	private void setInfoMessage(String message) {
		messageArea.setInfoMessage(message);
		updateSizeForMessageArea();
	}

	private void setErrorMessage(String message) {
		messageArea.setErrorMessage(message);
		updateSizeForMessageArea();
	}

	@Override
	protected void onUnload() {
		super.onUnload();
		busyIndicator.hide();
	}
	
	public static native void enableSort(Element element)/*-{

	    $wnd.sorttable.makeSortable(element);
	
	}-*/;

	/*
	 * Resize the scroll panel
	 */
	public void updateSize(int offsetWidth, int offsetHeight) {
		if (offsetWidth > 0) {
			int width = Window.getClientWidth() - offsetWidth;
			scrollPanel.setWidth(width + "px");
		}
		if (offsetHeight > 0) {
			int height = Window.getClientHeight() - offsetHeight
					- messageArea.getOffsetHeight() - buttonPanel.getOffsetHeight();
			scrollPanel.setHeight(height + "px");
		}
	}
	
	private void updateSizeForMessageArea() {
		scrollPanel.setHeight((scrollPanel.getOffsetHeight() - messageArea.getOffsetHeight()) + "px");
	}

}
