package com.ericsson.cms.scheduling.ui.client;

import java.util.ArrayList;
import java.util.Collection;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;

import com.ericsson.cms.scheduling.ui.client.common.Common;
import com.ericsson.cms.scheduling.ui.client.common.SchedulingConstants;
import com.ericsson.cms.scheduling.ui.client.data.UIExecutionRecord;
import com.ericsson.cms.scheduling.ui.client.service.IUISchedulerService;
import com.ericsson.cms.scheduling.ui.client.service.IUISchedulerServiceAsync;
import com.google.gwt.core.client.GWT;
import com.google.gwt.core.client.Scheduler;
import com.google.gwt.core.client.Scheduler.ScheduledCommand;
import com.google.gwt.event.dom.client.ClickEvent;
import com.google.gwt.event.dom.client.ClickHandler;
import com.google.gwt.event.logical.shared.ResizeEvent;
import com.google.gwt.event.logical.shared.ResizeHandler;
import com.google.gwt.event.logical.shared.ValueChangeEvent;
import com.google.gwt.event.logical.shared.ValueChangeHandler;
import com.google.gwt.event.shared.HandlerRegistration;
import com.google.gwt.user.client.History;
import com.google.gwt.user.client.Window;
import com.google.gwt.user.client.ui.Composite;
import com.google.gwt.user.client.ui.FlexTable;
import com.google.gwt.user.client.ui.HasHorizontalAlignment;
import com.google.gwt.user.client.ui.HasVerticalAlignment;
import com.google.gwt.user.client.ui.Widget;
import com.tandbergtv.neptune.widgettoolkit.client.remote.NeptuneAsyncCallback;
import com.tandbergtv.neptune.widgettoolkit.client.security.NeptuneSecurity;
import com.tandbergtv.neptune.widgettoolkit.client.widget.basic.ButtonWidget;
import com.tandbergtv.neptune.widgettoolkit.client.widget.basic.CheckBoxWidget;
import com.tandbergtv.neptune.widgettoolkit.client.widget.basic.HyperlinkWidget;
import com.tandbergtv.neptune.widgettoolkit.client.widget.basic.ImageWidget;
import com.tandbergtv.neptune.widgettoolkit.client.widget.basic.LabelWidget;
import com.tandbergtv.neptune.widgettoolkit.client.widget.composite.BusyIndicator;
import com.tandbergtv.neptune.widgettoolkit.client.widget.composite.messagearea.MessageArea;
import com.tandbergtv.neptune.widgettoolkit.client.widget.container.HorizontalContainer;
import com.tandbergtv.neptune.widgettoolkit.client.widget.container.ScrollContainer;
import com.tandbergtv.neptune.widgettoolkit.client.widget.container.VerticalContainer;
import com.tandbergtv.neptune.widgettoolkit.client.widget.style.StyleNames;

public class FutureExecutionsList extends Composite implements ISchedulerTab {
	private static final String ENABLED_IMAGE = "cms_scheduling_ui/images/enabled.png";
	private static final String DISABLED_IMAGE = "cms_scheduling_ui/images/disabled.png";

	private static final String TABLE_STYLE = "scheduler-schedulesList";
	private static final String HEADER_ROW_STYLE = "scheduler-schedulesList-header";
	private static final String DATA_ROW_STYLE = "scheduler-schedulesList-data";
	
	private static final int CMS_HEADER_MENU_SIZE_PX = 115;

	private IUISchedulerServiceAsync service = GWT.create(IUISchedulerService.class);
	private SchedulingConstants constants = GWT.create(SchedulingConstants.class);

	private BusyIndicator busyIndicator = new BusyIndicator(true);
	
	private VerticalContainer mainPanel = new VerticalContainer();
	private List<CheckBoxWidget> rowCheckBoxes;
	
	private MessageArea messageArea = new MessageArea();
	VerticalContainer tabPanel = new VerticalContainer();
	private ScrollContainer scrollPanel = new ScrollContainer();
	HorizontalContainer buttonPanel = new HorizontalContainer();
	private HandlerRegistration windowRegistration = null;

	private Set<UIExecutionRecord> selectedRecords = new HashSet<UIExecutionRecord>();
	
	public FutureExecutionsList(Map<String, String> info, NeptuneSecurity security) {
		mainPanel.setWidth("100%");

		mainPanel.add(scrollPanel);
		mainPanel.add(buttonPanel);

		//enable button
		if(security.isUserInRole(Permissions.EDIT)) {
			ButtonWidget enableButton = new ButtonWidget(constants.buttonLabelEnable());
			enableButton.addStyleDependentName(StyleNames.DO_BUTTON_STYLE);
			enableButton.addClickHandler(new ClickHandler() {
				@Override
				public void onClick(ClickEvent event) {
					if(selectedRecords.isEmpty()) {
						setErrorMessage("You have to select atleast one row to perform this action.");
						return;
					}
					for(UIExecutionRecord selectedRecord : selectedRecords) {
						if(selectedRecord.isEnabled()) {
							setErrorMessage("Only disabled executions can be enabled.");
							return;
						}
					}
					if(!isAuthorized(selectedRecords, Action.ENABLE_EXECUTION)) {
						setErrorMessage("You are not authorized to enable one or more of the selected executions.");
						return;
					}
					service.setFutureExecutionsEnabled(selectedRecords, true, new NeptuneAsyncCallback<Void>() {
						@Override
						public void onNeptuneFailure(Throwable caught) {
							setErrorMessage("Failed to enable the selected executions." + Common.getFailureMessageSuffix(caught));
						}
	
						@Override
						public void onNeptuneSuccess(Void result) {
							setInfoMessage("Successfully enabled the executions.");
							load();
						}
					});
				}});
			buttonPanel.add(enableButton);
		}
		
		//disable button
		if(security.isUserInRole(Permissions.EDIT)) {
			ButtonWidget disableButton = new ButtonWidget(constants.buttonLabelDisable());
			disableButton.addStyleDependentName(StyleNames.CAUTION_BUTTON_STYLE);
			disableButton.addClickHandler(new ClickHandler() {
				@Override
				public void onClick(ClickEvent event) {
					if(selectedRecords.isEmpty()) {
						setErrorMessage("You have to select atleast one row to perform this action.");
						return;
					}
					for(UIExecutionRecord selectedRecord : selectedRecords) {
						if(!selectedRecord.isEnabled()) {
							setErrorMessage("Only enabled executions can be disabled.");
							return;
						}
					}
					if(!isAuthorized(selectedRecords, Action.DISABLE_EXECUTION)) {
						setErrorMessage("You are not authorized to disable one or more of the selected executions.");
						return;
					}
					service.setFutureExecutionsEnabled(selectedRecords, false, new NeptuneAsyncCallback<Void>() {
						@Override
						public void onNeptuneFailure(Throwable caught) {
							setErrorMessage("Failed to disable the selected executions." + Common.getFailureMessageSuffix(caught));
						}
	
						@Override
						public void onNeptuneSuccess(Void result) {
							setInfoMessage("Successfully disabled the executions.");
							load();
						}
					});
				}});
			buttonPanel.add(disableButton);
		}
		
		initWidget(mainPanel);

		Scheduler.get().scheduleDeferred(new ScheduledCommand() {
			@Override
			public void execute() {
				updateScrollPanelSize();
			}
		});
	}

	private void load() {
		busyIndicator.center();
		service.getFutureExecutions(new NeptuneAsyncCallback<List<UIExecutionRecord>>() {
			@Override
			public void onNeptuneFailure(Throwable caught) {
				busyIndicator.hide();
				Window.alert("Failed to load future executions list." + Common.getFailureMessageSuffix(caught));
			}

			@Override
			public void onNeptuneSuccess(List<UIExecutionRecord> records) {
				rowCheckBoxes = new ArrayList<CheckBoxWidget>(records.size());
				selectedRecords.clear();
				buildTable(records);
				updateScrollPanelSize();
				busyIndicator.hide();
			}
		});
	}
	
	private void buildTable(List<UIExecutionRecord> records) {
		FlexTable table = new FlexTable();
		table.addStyleName(TABLE_STYLE);
		VerticalContainer tablePanel = new VerticalContainer();
		tablePanel.setWidth("100%");
		tablePanel.add(table);
		scrollPanel.setWidget(tablePanel);
		
		table.getRowFormatter().addStyleName(0, HEADER_ROW_STYLE);

		final CheckBoxWidget allCB = new CheckBoxWidget();
		allCB.addValueChangeHandler(new ValueChangeHandler<Boolean>() {
			@Override
			public void onValueChange(ValueChangeEvent<Boolean> event) {
				if(rowCheckBoxes != null) {
					for(CheckBoxWidget cb : rowCheckBoxes) {
						cb.setValue(event.getValue(), true);
					}
				}
			}});
		table.setWidget(0, 0, allCB);
		table.setWidget(0, 1, new LabelWidget(constants.columnLabelEnabled()));
		table.setWidget(0, 2, new LabelWidget(constants.columnLabelExecutionTime()));
		table.setWidget(0, 3, new LabelWidget(constants.columnLabelId()));
		table.setWidget(0, 4, new LabelWidget(constants.columnLabelName()));
		table.setWidget(0, 5, new LabelWidget(constants.columnLabelAssociation()));
		
		table.getCellFormatter().setWidth(0, 4, "25%");
		table.getCellFormatter().setWidth(0, 5, "35%");
		
		int associationDescriptionCellWidth = Window.getClientWidth()*35/100;

		int rowIndex = 1;
		for(final UIExecutionRecord r : records) {
			table.getRowFormatter().addStyleName(rowIndex, DATA_ROW_STYLE);

			CheckBoxWidget cb = new CheckBoxWidget();
			cb.addValueChangeHandler(new ValueChangeHandler<Boolean>() {
				@Override
				public void onValueChange(ValueChangeEvent<Boolean> event) {
					if(event.getValue()) {
						selectedRecords.add(r);
					} else {
						allCB.setValue(false, false);
						selectedRecords.remove(r);
					}
				}});
			table.setWidget(rowIndex, 0, cb);
			table.getCellFormatter().setAlignment(rowIndex, 0,
					HasHorizontalAlignment.ALIGN_CENTER,
					HasVerticalAlignment.ALIGN_MIDDLE);
			rowCheckBoxes.add(cb);

			ImageWidget enabledImageWidget = new ImageWidget((r.isEnabled() ? ENABLED_IMAGE : DISABLED_IMAGE));
			table.setWidget(rowIndex, 1, enabledImageWidget);
			table.getCellFormatter().setAlignment(rowIndex, 1,
					HasHorizontalAlignment.ALIGN_CENTER,
					HasVerticalAlignment.ALIGN_MIDDLE);

			table.setWidget(rowIndex, 2, new LabelWidget(r.getExecutionTimeString()));

			Widget idWidget = null;
			if (r.getHistoryToken() != null) {
				idWidget = new HyperlinkWidget(String.valueOf(r.getJobId()),
						r.getHistoryToken());
			} else {
				idWidget = new LabelWidget(String.valueOf(r.getJobId()));
			}
			table.setWidget(rowIndex, 3, idWidget);
			table.getCellFormatter().setAlignment(rowIndex, 3,
					HasHorizontalAlignment.ALIGN_CENTER,
					HasVerticalAlignment.ALIGN_MIDDLE);

			table.setWidget(rowIndex, 4, new LabelWidget(r.getJobName()));

			String truncatedAssociatedDescription = SchedulesListUtils
					.getTruncatedAssociationDescription(r.getAssociation(),
							associationDescriptionCellWidth);
			table.setWidget(rowIndex, 5, new LabelWidget(truncatedAssociatedDescription));

			rowIndex++;
		}
	}
	
	private boolean isAuthorized(Collection<UIExecutionRecord> records, Action action) {
		for(UIExecutionRecord r : records) {
			if(!Common.isUserAuthorized(r.getAssociatedEntityType(), action))
				return false;
		}
		return true;
	}
	
	private void setInfoMessage(String message) {
		messageArea.setInfoMessage(message);
		updateScrollPanelSize();
	}

	private void setErrorMessage(String message) {
		messageArea.setErrorMessage(message);
		updateScrollPanelSize();
	}
	
	private void updateScrollPanelSize() {
		int height = Window.getClientHeight()
			- CMS_HEADER_MENU_SIZE_PX
			- messageArea.getOffsetHeight()
			- buttonPanel.getOffsetHeight()
			- 92;	//adjustment for borders and spacing at the bottom and header panel at the top
		scrollPanel.setHeight(height + "px");
	}
	
	// ==============================================================
	// ===================== WIDGET OVERRIDES
	// ==============================================================

	@Override
	protected void onLoad() {
		super.onLoad();

		windowRegistration = Window.addResizeHandler(new ResizeHandler() {
			public void onResize(ResizeEvent event) {
				updateScrollPanelSize();
			}
		});
	};

	@Override
	protected void onUnload() {
		windowRegistration.removeHandler();
		windowRegistration = null;

		if (busyIndicator != null)
			busyIndicator.hide();

		super.onUnload();
	}

	@Override
	public void setMessageArea(MessageArea messageArea) {
		this.messageArea = messageArea;
	}

	@Override
	public void setSelected(boolean selected) {
		if(!selected)
			return;
		
		messageArea.reset();
		
		History.newItem("Scheduler.Executions", false);

		load();
	}
}
