package com.ericsson.cms.scheduling.ui.client;

import java.util.List;
import java.util.Map;

import com.ericsson.cms.scheduling.ui.client.common.Common;
import com.ericsson.cms.scheduling.ui.client.common.SchedulingConstants;
import com.ericsson.cms.scheduling.ui.client.data.UIExecutionRecord;
import com.ericsson.cms.scheduling.ui.client.service.IUISchedulerService;
import com.ericsson.cms.scheduling.ui.client.service.IUISchedulerServiceAsync;
import com.google.gwt.core.client.GWT;
import com.google.gwt.core.client.Scheduler;
import com.google.gwt.core.client.Scheduler.ScheduledCommand;
import com.google.gwt.event.logical.shared.ResizeEvent;
import com.google.gwt.event.logical.shared.ResizeHandler;
import com.google.gwt.event.shared.HandlerRegistration;
import com.google.gwt.user.client.History;
import com.google.gwt.user.client.Window;
import com.google.gwt.user.client.ui.Composite;
import com.google.gwt.user.client.ui.FlexTable;
import com.google.gwt.user.client.ui.HasHorizontalAlignment;
import com.google.gwt.user.client.ui.HasVerticalAlignment;
import com.google.gwt.user.client.ui.Widget;
import com.tandbergtv.neptune.widgettoolkit.client.remote.NeptuneAsyncCallback;
import com.tandbergtv.neptune.widgettoolkit.client.widget.basic.HTMLWidget;
import com.tandbergtv.neptune.widgettoolkit.client.widget.basic.HyperlinkWidget;
import com.tandbergtv.neptune.widgettoolkit.client.widget.basic.ImageWidget;
import com.tandbergtv.neptune.widgettoolkit.client.widget.basic.LabelWidget;
import com.tandbergtv.neptune.widgettoolkit.client.widget.composite.BusyIndicator;
import com.tandbergtv.neptune.widgettoolkit.client.widget.composite.messagearea.MessageArea;
import com.tandbergtv.neptune.widgettoolkit.client.widget.container.ScrollContainer;
import com.tandbergtv.neptune.widgettoolkit.client.widget.container.VerticalContainer;

public class PastExecutionsList extends Composite  implements ISchedulerTab {
	private static final String ENABLED_IMAGE = "cms_scheduling_ui/images/enabled.png";
	private static final String DISABLED_IMAGE = "cms_scheduling_ui/images/disabled.png";

	private static final String TABLE_STYLE = "scheduler-schedulesList";
	private static final String HEADER_ROW_STYLE = "scheduler-schedulesList-header";
	private static final String DATA_ROW_STYLE = "scheduler-schedulesList-data";
	private static final String STALEDATA_ROW_STYLE = "scheduler-schedulesList-staledata";
	private static final String NOTES_MARKER_STYLE = "scheduler-notesMarker";
	
	private static final int CMS_HEADER_MENU_SIZE_PX = 115;

	private IUISchedulerServiceAsync service = GWT.create(IUISchedulerService.class);
	private SchedulingConstants constants = GWT.create(SchedulingConstants.class);

	private BusyIndicator busyIndicator = new BusyIndicator(true);
	
	private VerticalContainer mainPanel = new VerticalContainer();
	private ScrollContainer scrollPanel = new ScrollContainer();
	HTMLWidget legendWidget = new HTMLWidget("<span class='" + NOTES_MARKER_STYLE + "'>*</span> "
			+ constants.staleRecordsLegend());
	private HandlerRegistration windowRegistration = null;

	private MessageArea messageArea = new MessageArea();

	public PastExecutionsList(Map<String, String> info) {
		mainPanel.setWidth("100%");

		mainPanel.add(scrollPanel);
		mainPanel.add(legendWidget);
		
		initWidget(mainPanel);
		
		Scheduler.get().scheduleDeferred(new ScheduledCommand() {
			@Override
			public void execute() {
				updateScrollPanelSize();
			}
		});
	}

	private void load() {
		busyIndicator.center();
		service.getPastExecutions(new NeptuneAsyncCallback<List<UIExecutionRecord>>() {
			@Override
			public void onNeptuneFailure(Throwable caught) {
				busyIndicator.hide();
				Window.alert("Failed to load past execution records." + Common.getFailureMessageSuffix(caught));
			}

			@Override
			public void onNeptuneSuccess(List<UIExecutionRecord> records) {
				buildTable(records);
				updateScrollPanelSize();
				busyIndicator.hide();
			}
		});
	}
	
	private void buildTable(List<UIExecutionRecord> records) {
		FlexTable table = new FlexTable();
		table.addStyleName(TABLE_STYLE);
		VerticalContainer tablePanel = new VerticalContainer();
		tablePanel.setWidth("100%");
		tablePanel.add(table);
		scrollPanel.setWidget(tablePanel);
		
		table.getRowFormatter().addStyleName(0, HEADER_ROW_STYLE);

		table.setWidget(0, 0, new LabelWidget(constants.columnLabelEnabled()));
		table.setWidget(0, 1, new LabelWidget(constants.columnLabelExecutionTime()));
		table.setWidget(0, 2, new LabelWidget(constants.columnLabelId()));
		table.setWidget(0, 3, new LabelWidget(constants.columnLabelName()));
		table.setWidget(0, 4, new LabelWidget(constants.columnLabelAssociation()));

		table.getCellFormatter().setWidth(0, 3, "25%");
		table.getCellFormatter().setWidth(0, 4, "40%");
		
		int associationDescriptionCellWidth = Window.getClientWidth()*40/100;

		int rowIndex = 1;
		for(final UIExecutionRecord r : records) {
			table.getRowFormatter().addStyleName(rowIndex, r.isStale() ? STALEDATA_ROW_STYLE : DATA_ROW_STYLE);

			ImageWidget enabledImageWidget = new ImageWidget((r.isEnabled() ? ENABLED_IMAGE : DISABLED_IMAGE));
			table.setWidget(rowIndex, 0, enabledImageWidget);
			table.getCellFormatter().setAlignment(rowIndex, 0,
					HasHorizontalAlignment.ALIGN_CENTER,
					HasVerticalAlignment.ALIGN_MIDDLE);

			table.setWidget(rowIndex, 1, new LabelWidget(r.getExecutionTimeString()));

			Widget idWidget = null;
			if (r.getHistoryToken() != null) {
				idWidget = new HyperlinkWidget(String.valueOf(r.getJobId()),
						r.getHistoryToken());
			} else {
				idWidget = new LabelWidget(String.valueOf(r.getJobId()));
			}
			table.setWidget(rowIndex, 2, idWidget);
			table.getCellFormatter().setAlignment(rowIndex, 2,
					HasHorizontalAlignment.ALIGN_CENTER,
					HasVerticalAlignment.ALIGN_MIDDLE);

			table.setWidget(rowIndex, 3, new LabelWidget(r.getJobName()));
			
			String truncatedAssociatedDescription = SchedulesListUtils
					.getTruncatedAssociationDescription(
							r.getAssociation(),
							associationDescriptionCellWidth);
			table.setWidget(rowIndex, 4, new LabelWidget(truncatedAssociatedDescription));
			rowIndex++;
		}
	}
	
	private void updateScrollPanelSize() {
		int height = Window.getClientHeight()
			- CMS_HEADER_MENU_SIZE_PX
			- messageArea.getOffsetHeight()
			- 115;	//adjustment for borders, spacing and legend at the bottom and header panel at the top
		scrollPanel.setHeight(height + "px");
	}
	
	// ==============================================================
	// ===================== WIDGET OVERRIDES
	// ==============================================================

	@Override
	protected void onLoad() {
		super.onLoad();

		windowRegistration = Window.addResizeHandler(new ResizeHandler() {
			public void onResize(ResizeEvent event) {
				updateScrollPanelSize();
			}
		});
	};

	@Override
	protected void onUnload() {
		windowRegistration.removeHandler();
		windowRegistration = null;

		if (busyIndicator != null)
			busyIndicator.hide();

		super.onUnload();
	}

	@Override
	public void setMessageArea(MessageArea messageArea) {
		this.messageArea = messageArea;
	}

	@Override
	public void setSelected(boolean selected) {
		if(!selected)
			return;

		messageArea.reset();
		
		History.newItem("Scheduler.Executions.Past", false);
		load();
	}
}
