package com.ericsson.cms.scheduling.ui.client;

import java.util.ArrayList;
import java.util.HashSet;
import java.util.List;
import java.util.Set;

import com.ericsson.cms.scheduling.ui.client.common.Common;
import com.ericsson.cms.scheduling.ui.client.common.SchedulingConstants;
import com.ericsson.cms.scheduling.ui.client.data.SchedulesListRow;
import com.ericsson.cms.scheduling.ui.client.service.IUISchedulerService;
import com.ericsson.cms.scheduling.ui.client.service.IUISchedulerServiceAsync;
import com.ericsson.cms.scheduling.ui.client.widgets.SchedulingWidget;
import com.google.gwt.core.client.GWT;
import com.google.gwt.core.client.Scheduler;
import com.google.gwt.core.client.Scheduler.ScheduledCommand;
import com.google.gwt.dom.client.Document;
import com.google.gwt.event.dom.client.ClickEvent;
import com.google.gwt.event.dom.client.ClickHandler;
import com.google.gwt.event.logical.shared.ResizeEvent;
import com.google.gwt.event.logical.shared.ResizeHandler;
import com.google.gwt.event.logical.shared.ValueChangeEvent;
import com.google.gwt.event.logical.shared.ValueChangeHandler;
import com.google.gwt.event.shared.HandlerRegistration;
import com.google.gwt.user.client.Command;
import com.google.gwt.user.client.Element;
import com.google.gwt.user.client.History;
import com.google.gwt.user.client.Window;
import com.google.gwt.user.client.ui.Composite;
import com.google.gwt.user.client.ui.FlexTable;
import com.google.gwt.user.client.ui.HasHorizontalAlignment;
import com.google.gwt.user.client.ui.HasVerticalAlignment;
import com.google.gwt.user.client.ui.HorizontalPanel;
import com.google.gwt.user.client.ui.Widget;
import com.tandbergtv.neptune.widgettoolkit.client.remote.NeptuneAsyncCallback;
import com.tandbergtv.neptune.widgettoolkit.client.security.NeptuneSecurity;
import com.tandbergtv.neptune.widgettoolkit.client.widget.basic.ButtonWidget;
import com.tandbergtv.neptune.widgettoolkit.client.widget.basic.CheckBoxWidget;
import com.tandbergtv.neptune.widgettoolkit.client.widget.basic.HyperlinkWidget;
import com.tandbergtv.neptune.widgettoolkit.client.widget.basic.ImageWidget;
import com.tandbergtv.neptune.widgettoolkit.client.widget.basic.LabelWidget;
import com.tandbergtv.neptune.widgettoolkit.client.widget.composite.BusyIndicator;
import com.tandbergtv.neptune.widgettoolkit.client.widget.composite.HeaderPanel;
import com.tandbergtv.neptune.widgettoolkit.client.widget.composite.messagearea.MessageArea;
import com.tandbergtv.neptune.widgettoolkit.client.widget.container.DialogBoxContainer;
import com.tandbergtv.neptune.widgettoolkit.client.widget.container.HorizontalContainer;
import com.tandbergtv.neptune.widgettoolkit.client.widget.container.ScrollContainer;
import com.tandbergtv.neptune.widgettoolkit.client.widget.container.TabContainer;
import com.tandbergtv.neptune.widgettoolkit.client.widget.container.VerticalContainer;
import com.tandbergtv.neptune.widgettoolkit.client.widget.style.StyleNames;

public class SchedulesListWidget extends Composite {
	private static class CheckBox extends CheckBoxWidget {
		private SchedulesListRow data;
		
		public CheckBox() {
			super();
		}
		
		public CheckBox(SchedulesListRow data) {
			this();
			this.data = data;
		}

		/**
		 * @return the data
		 */
		public SchedulesListRow getData() {
			return data;
		}

		/**
		 * @param data the data to set
		 */
		public void setData(SchedulesListRow data) {
			this.data = data;
		}
	}
	
	private static final String ENABLED_IMAGE = "cms_scheduling_ui/images/enabled.png";
	private static final String DISABLED_IMAGE = "cms_scheduling_ui/images/disabled.png";

	private static final String TAB_CONTAINER_STYLE = "scheduler-tabContainer";
	private static final String TABLE_STYLE = "scheduler-schedulesList";
	private static final String HEADER_ROW_STYLE = "scheduler-schedulesList-header";
	private static final String DATA_ROW_STYLE = "scheduler-schedulesList-data";
	private static int CMS_HEADER_MENU_SIZE_PX = 115;
	
	private static final String ID_COLUMN_HEADER_CELL_ID = "schedulesListIdHeader";

	private IUISchedulerServiceAsync service = GWT.create(IUISchedulerService.class);
	private SchedulingConstants constants = GWT.create(SchedulingConstants.class);

	private BusyIndicator busyIndicator = new BusyIndicator(true);
	
	private VerticalContainer mainPanel = new VerticalContainer();
	private HeaderPanel headerPanel = new HeaderPanel("View Schedules");
	private MessageArea messageArea = new MessageArea();
	private TabContainer tabContainer = new TabContainer();
	private ScrollContainer tableContainer = new ScrollContainer();
	private HorizontalContainer buttonPanel = new HorizontalContainer();
	private HandlerRegistration resizeHandlerRegistration;
	
	private List<CheckBox> rowCheckBoxes;
	
	public SchedulesListWidget(NeptuneSecurity security) {
		if(!security.isUserInRole(Permissions.VIEW)) {
			Window.alert("You do not have permission to view schedules.");
			return;
		}
		
		mainPanel.setWidth("100%");
		mainPanel.add(headerPanel);
		
		messageArea.reset();
		mainPanel.add(messageArea);
		
		tabContainer.addStyleName(TAB_CONTAINER_STYLE);
		mainPanel.add(tabContainer);
		
		VerticalContainer tabPanel = new VerticalContainer();
		tabContainer.add(tabPanel, "Schedules");
		tabContainer.selectTab(0);

		tableContainer.setHeight("0px");
		tableContainer.setWidth("100%");
		tabPanel.add(tableContainer);
		
		tabPanel.add(buttonPanel);
		
		//activate button
		if(security.isUserInRole(Permissions.EDIT)) {
			ButtonWidget activateButton = new ButtonWidget(constants.buttonLabelActivate());
			activateButton.addStyleDependentName("do-button");
			activateButton.addClickHandler(new ClickHandler() {
				@Override
				public void onClick(ClickEvent event) {
						Set<Long> selectedIds = new HashSet<Long>(rowCheckBoxes.size());
						
						List<SchedulesListRow> selected = getSelected();
						for (SchedulesListRow schedule : selected) {
							if (schedule.isActive()) {
								setErrorMessage("Only inactive schedules can be activated.");
								return;
							}
							selectedIds.add(schedule.getWoJobId());
						}
						
						if (selectedIds.isEmpty()) {
							setErrorMessage(constants.noScheduleSelected());
							return;
						}
	
						if(!isUserAuthorized(selected, Action.ACTIVATE)) {
							setErrorMessage("You are not authorized to activate one or more of the selected schedules.");
							return;
						}
	
						service.setActive(selectedIds, true,
							new NeptuneAsyncCallback<Void>() {
								@Override
								public void onNeptuneFailure(Throwable caught) {
									setErrorMessage("Failed to activate the selected schedules."
											+ Common.getFailureMessageSuffix(caught));
								}
	
								@Override
								public void onNeptuneSuccess(Void result) {
									setInfoMessage("Successfully activated the schedules.");
									load();
								}
							});
				}});
			buttonPanel.add(activateButton);
		}
		
		//run-now button
		if(security.isUserInRole(Permissions.EXECUTE)) {
			ButtonWidget runNowButton = new ButtonWidget(constants.buttonLabelRunNow());
			runNowButton.addStyleDependentName("do-button");
			runNowButton.addClickHandler(new ClickHandler() {
				@Override
				public void onClick(ClickEvent event) {
						Set<Long> selectedIds = new HashSet<Long>(rowCheckBoxes.size());
						
						List<SchedulesListRow> selected = getSelected();
						for (SchedulesListRow schedule : selected) {
							if (!schedule.isActive()) {
								setErrorMessage("Only active schedules can be run.");
								return;
							}
							selectedIds.add(schedule.getId());
						}
						
						if (selectedIds.isEmpty()) {
							setErrorMessage(constants.noScheduleSelected());
							return;
						}
	
						if(!isUserAuthorized(selected, Action.EXECUTE)) {
							setErrorMessage("You are not authorized to run one or more of the selected schedules.");
							return;
						}
	
						service.runNow(selectedIds, new NeptuneAsyncCallback<Void>() {
								@Override
								public void onNeptuneFailure(Throwable caught) {
									setErrorMessage("Failed to run the selected schedules."
											+ Common.getFailureMessageSuffix(caught));
								}
	
								@Override
								public void onNeptuneSuccess(Void result) {
									setInfoMessage("Successfully triggered to run the schedules.");
									//No need to reload as the table does not yet show any data that change in each run.
									//Just unselect.
									for(CheckBox cb : rowCheckBoxes) {
										cb.setValue(false, false);
									}
								}
							});
				}});
			buttonPanel.add(runNowButton);
		}

		//create button
		if(security.isUserInRole(Permissions.CREATE)) {
			ButtonWidget createButton = new ButtonWidget(constants.buttonLabelCreate());
			createButton.addStyleDependentName("act-towards-save-button");
			createButton.addClickHandler(new ClickHandler() {
				@Override
				public void onClick(ClickEvent event) {
					History.newItem("Scheduler.View.Create");
				}});
			buttonPanel.add(createButton);
		}
		
		//inactivate button
		if(security.isUserInRole(Permissions.EDIT)) {
			ButtonWidget inactivateButton = new ButtonWidget(constants.buttonLabelInactivate());
			inactivateButton.addStyleDependentName("caution-button");
			inactivateButton.addClickHandler(new ClickHandler() {
				@Override
				public void onClick(ClickEvent event) {
						Set<Long> selectedIds = new HashSet<Long>(rowCheckBoxes.size());
						
						List<SchedulesListRow> selected = getSelected();
						for (SchedulesListRow schedule : selected) {
							if (!schedule.isActive()) {
								setErrorMessage("Only active schedules can be inactivated.");
								return;
							}
							selectedIds.add(schedule.getWoJobId());
						}
						
						if (selectedIds.isEmpty()) {
							setErrorMessage(constants.noScheduleSelected());
							return;
						}
	
						if(!isUserAuthorized(selected, Action.INACTIVATE)) {
							setErrorMessage("You are not authorized to inactivate one or more of the selected schedules.");
							return;
						}
	
						service.setActive(selectedIds, false,
							new NeptuneAsyncCallback<Void>() {
								@Override
								public void onNeptuneFailure(Throwable caught) {
									setErrorMessage("Failed to inactivate the selected schedules."
											+ Common.getFailureMessageSuffix(caught));
								}
	
								@Override
								public void onNeptuneSuccess(Void result) {
									setInfoMessage("Successfully inactivated the schedules.");
									load();
								}
							});
				}});
			buttonPanel.add(inactivateButton);
		}

		//delete button
		if(security.isUserInRole(Permissions.DELETE)) {
			ButtonWidget deleteButton = new ButtonWidget(constants.buttonLabelDelete());
			deleteButton.addStyleDependentName("dataloss-button");
			deleteButton.addClickHandler(new ClickHandler() {
				@Override
				public void onClick(ClickEvent event) {
					final Set<Long> selectedIds = new HashSet<Long>(rowCheckBoxes.size());
					
					List<SchedulesListRow> selected = getSelected();
					for (SchedulesListRow schedule : selected) {
						selectedIds.add(schedule.getWoJobId());
					}
					
					if (selectedIds.isEmpty()) {
						setErrorMessage(constants.noScheduleSelected());
						return;
					}
					
					if(!isUserAuthorized(selected, Action.DELETE)) {
						setErrorMessage("You are not authorized to delete one or more of the selected schedules.");
						return;
					}
					
					confirm(constants.confirmMultipleDelete(), new Command() {
						@Override
						public void execute() {
							delete(selectedIds);
						}
					}, null);
				}});
			buttonPanel.add(deleteButton);
		}

		load();
		initWidget(mainPanel);
		
		Scheduler.get().scheduleDeferred(new ScheduledCommand() {
			@Override
			public void execute() {
				updateTableContainerSize();
			}
		});
	}
	
	private void updateTableContainerSize() {
		int height = Window.getClientHeight()
			- CMS_HEADER_MENU_SIZE_PX
			- headerPanel.getOffsetHeight()
			- messageArea.getOffsetHeight()
			- tabContainer.getTabBar().getOffsetHeight()
			- buttonPanel.getOffsetHeight()
			- 14;	//adjustment for borders and spacing at the bottom
		tableContainer.setHeight(height + "px");
	}

	private List<SchedulesListRow> getSelected() {
		List<SchedulesListRow> selected = new ArrayList<SchedulesListRow>(rowCheckBoxes.size());
		for (CheckBox rowCB : rowCheckBoxes)
			if(rowCB.getValue())
				selected.add(rowCB.getData());
		return selected;
	}
	
	private boolean isUserAuthorized(List<SchedulesListRow> selected, Action action) {
		for(SchedulesListRow schedule : selected)
			if(!Common.isUserAuthorized(schedule.getAssociatedEntityType(), action))
				return false;
		return true;
	}
	
	private void load() {
		busyIndicator.center();
		service.getAllJobs(new NeptuneAsyncCallback<List<SchedulesListRow>>() {
			@Override
			public void onNeptuneFailure(Throwable caught) {
				busyIndicator.hide();
				Window.alert("Failed to load schedules list." + Common.getFailureMessageSuffix(caught));
			}

			@Override
			public void onNeptuneSuccess(List<SchedulesListRow> schedules) {
				rowCheckBoxes = new ArrayList<CheckBox>(schedules.size());
				buildTable(schedules);
				busyIndicator.hide();

				Scheduler.get().scheduleDeferred(new ScheduledCommand() {

					@Override
					public void execute() {
						Document.get().getElementById(ID_COLUMN_HEADER_CELL_ID).dispatchEvent(Document.get().createClickEvent(0, 0, 0, 0, 0, false, false, false, false));
					}
					
				}) ;
			}
		});
	}
	
	private void buildTable(List<SchedulesListRow> jobs) {
		FlexTable table = new FlexTable();
		table.addStyleName(TABLE_STYLE);
		tableContainer.setWidget(table);
		
		table.getRowFormatter().addStyleName(0, HEADER_ROW_STYLE);

		final CheckBoxWidget allCB = new CheckBoxWidget();
		allCB.addValueChangeHandler(new ValueChangeHandler<Boolean>() {
			@Override
			public void onValueChange(ValueChangeEvent<Boolean> event) {
				if(rowCheckBoxes != null) {
					for(CheckBoxWidget cb : rowCheckBoxes) {
						cb.setValue(event.getValue(), true);
					}
				}
			}});
		table.setWidget(0, 0, allCB);
		table.setText(0, 1, constants.columnLabelId());
		table.setText(0, 2, constants.columnLabelActive());
		table.setText(0, 3, constants.columnLabelName());
		table.setText(0, 4, constants.columnLabelAssociation());
		table.setText(0, 5, constants.columnLabelTemplateName());
		table.setText(0, 6, constants.columnLabelJobScheduleRule());
		
		table.getCellFormatter().addStyleName(0, 0, "sorttable_nosort");

		table.getCellFormatter().getElement(0, 1).setId(ID_COLUMN_HEADER_CELL_ID);
		
		table.getCellFormatter().setWidth(0, 0, "1%");
		table.getCellFormatter().setWidth(0, 1, "5%");
		table.getCellFormatter().setWidth(0, 2, "4%");
		table.getCellFormatter().setWidth(0, 3, "15%");
		table.getCellFormatter().setWidth(0, 4, "30%");
		table.getCellFormatter().setWidth(0, 5, "15%");
		table.getCellFormatter().setWidth(0, 6, "30%");
		
		int tableWidth = table.getOffsetWidth();
		int associationDescriptionCellWidth = tableWidth*4/10;

		int rowIndex = 1;
		for(final SchedulesListRow job : jobs) {
			table.getRowFormatter().addStyleName(rowIndex, DATA_ROW_STYLE);
			if(job.isEditAllowed()) {
				CheckBox cb = new CheckBox(job);
				cb.addValueChangeHandler(new ValueChangeHandler<Boolean>() {
					@Override
					public void onValueChange(ValueChangeEvent<Boolean> event) {
						if(!event.getValue()) {
							allCB.setValue(false, false);
						}
					}});
				table.setWidget(rowIndex, 0, cb);
				rowCheckBoxes.add(cb);
			}
			
			Widget idWidget = new LabelWidget(Long.toString(job.getId()));
			if (job.getHistoryToken() != null) {
				idWidget = new HyperlinkWidget(Long.toString(job.getId()),
						job.getHistoryToken());
			}
			table.setWidget(rowIndex, 1, idWidget);
			table.getCellFormatter().setAlignment(rowIndex, 1,
					HasHorizontalAlignment.ALIGN_CENTER,
					HasVerticalAlignment.ALIGN_MIDDLE);
			
			ImageWidget enabledImageWidget = new ImageWidget((job.isActive() ? ENABLED_IMAGE : DISABLED_IMAGE));
			LabelWidget hiddenLabelWidget = new LabelWidget(job.isActive() ? "1" : "0");
			hiddenLabelWidget.setVisible(false);
			HorizontalPanel imagePanel = new HorizontalPanel();
			imagePanel.add(hiddenLabelWidget);
			imagePanel.add(enabledImageWidget);
			table.setWidget(rowIndex, 2, imagePanel);
			table.getCellFormatter().setAlignment(rowIndex, 2,
					HasHorizontalAlignment.ALIGN_CENTER,
					HasVerticalAlignment.ALIGN_MIDDLE);
			
			table.setWidget(rowIndex, 3, new LabelWidget(job.getName()));
			
			String truncatedAssociatedDescription = SchedulesListUtils
					.getTruncatedAssociationDescription(
							job.getAssociationDescription(),
							associationDescriptionCellWidth);
			table.setWidget(rowIndex, 4, new LabelWidget(truncatedAssociatedDescription));
			table.setWidget(rowIndex, 5, new LabelWidget(job.getTemplateName()));
			table.setWidget(rowIndex, 6, new LabelWidget(
					SchedulingWidget.getScheduleInfoString(job.getScheduleInfo())));
			rowIndex++;
		}
		enableSort(table.getElement());
	}
	
	private void delete(Set<Long> ids) {
		service.delete(ids, new NeptuneAsyncCallback<Void>() {
				@Override
				public void onNeptuneFailure(Throwable caught) {
					setErrorMessage("Failed to delete the selected schedules."
							+ Common.getFailureMessageSuffix(caught));
				}

				@Override
				public void onNeptuneSuccess(Void result) {
					setInfoMessage("Successfully deleted the schedules.");
					load();
				}
			});
	}
	
	private void confirm(String questionText, final Command yesCommand,
			final Command noCommand) {
		final DialogBoxContainer dialogBox = new DialogBoxContainer(false, true);
		dialogBox.setText(questionText);

		// yes button
		ButtonWidget yesButton = new ButtonWidget(constants.yes());
		yesButton.addStyleDependentName(StyleNames.COMMIT_BUTTON_STYLE);
		yesButton.addClickHandler(new ClickHandler() {
			@Override
			public void onClick(ClickEvent event) {
				dialogBox.hide();
				if (yesCommand != null)
					yesCommand.execute();
			}
		});

		// no button
		ButtonWidget noButton = new ButtonWidget(constants.no());
		noButton.addStyleDependentName(StyleNames.DATALOSS_BUTTON_STYLE);
		noButton.addClickHandler(new ClickHandler() {
			@Override
			public void onClick(ClickEvent event) {
				dialogBox.hide();
				if (noCommand != null)
					noCommand.execute();
			}
		});

		// panel which contains the buttons
		HorizontalContainer buttonPanel = new HorizontalContainer();
		dialogBox.setWidget(buttonPanel);
		buttonPanel.add(yesButton);
		buttonPanel.setCellHorizontalAlignment(yesButton,
				HorizontalPanel.ALIGN_RIGHT);
		buttonPanel.add(noButton);
		buttonPanel.setCellHorizontalAlignment(noButton,
				HorizontalPanel.ALIGN_LEFT);

		dialogBox.center();
		dialogBox.show();
	}
	
	private void setInfoMessage(String message) {
		messageArea.setInfoMessage(message);
		updateTableContainerSize();
	}

	private void setErrorMessage(String message) {
		messageArea.setErrorMessage(message);
		updateTableContainerSize();
	}
	
	private static native void enableSort(Element element)/*-{
    	$wnd.sorttable.makeSortable(element);
	}-*/;

	@Override
	protected void onLoad() {
		super.onLoad();

		resizeHandlerRegistration = Window.addResizeHandler(new ResizeHandler() {
			public void onResize(ResizeEvent event) {
				updateTableContainerSize();
			}
		});
	};

	@Override
	protected void onUnload() {
		resizeHandlerRegistration.removeHandler();
		resizeHandlerRegistration = null;
		busyIndicator.hide();
		super.onUnload();
	}
}
