package com.ericsson.cms.scheduling.ui.client.detailsPage;

import java.util.ArrayList;
import java.util.Collection;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;

import com.ericsson.cms.scheduling.ui.client.common.Common;
import com.ericsson.cms.scheduling.ui.client.common.SchedulingConstants;
import com.ericsson.cms.scheduling.ui.client.service.IUISchedulerService;
import com.ericsson.cms.scheduling.ui.client.service.IUISchedulerServiceAsync;
import com.google.gwt.core.client.GWT;
import com.google.gwt.event.dom.client.ChangeEvent;
import com.google.gwt.event.dom.client.ChangeHandler;
import com.google.gwt.user.client.Window;
import com.google.gwt.user.client.ui.Composite;
import com.tandbergtv.neptune.widgettoolkit.client.remote.NeptuneAsyncCallback;
import com.tandbergtv.neptune.widgettoolkit.client.widget.basic.LabelWidget;
import com.tandbergtv.neptune.widgettoolkit.client.widget.basic.ListBoxWidget;
import com.tandbergtv.neptune.widgettoolkit.client.widget.composite.MultiTaskBusyIndicator;
import com.tandbergtv.neptune.widgettoolkit.client.widget.container.DisclosureContainer;
import com.tandbergtv.neptune.widgettoolkit.client.widget.container.HorizontalContainer;
import com.tandbergtv.neptune.widgettoolkit.client.widget.container.RoundedDisclosureContainer;
import com.tandbergtv.neptune.widgettoolkit.client.widget.container.VerticalContainer;

public class EntityAssociationWidget extends Composite {
	public static interface ValueChangeListener {
		public void valueChanged(String entityType);
	}
	
	private IUISchedulerServiceAsync service = GWT.create(IUISchedulerService.class);
	private SchedulingConstants constants = GWT.create(SchedulingConstants.class);
	
	private MultiTaskBusyIndicator busyIndicator = new MultiTaskBusyIndicator();

	private ValueChangeListener listener;
	private ListBoxWidget<String> associatedEntityTypeListBox;
	private ListBoxWidget<Long> associatedEntitiesListBox;
	
	public EntityAssociationWidget(String selectedEntityType, Set<Long> selectedEntities) {
		DisclosureContainer disclosurePanel = new RoundedDisclosureContainer(constants.association(),true);
		disclosurePanel.addStyleName(Common.STYLE_DISCLOSURE_CONTAINER);
		
		VerticalContainer associationPanel = new VerticalContainer();
		disclosurePanel.add(associationPanel);
		
		associationPanel.add(Common.getRequiredFieldName(constants.associatedWith()));
		
		HorizontalContainer associationSelectionPanel = new HorizontalContainer();
		associationPanel.add(associationSelectionPanel);
		
		associatedEntityTypeListBox = new ListBoxWidget<String>(false);
		associationSelectionPanel.add(associatedEntityTypeListBox);
		
		LabelWidget spacerWidget = new LabelWidget();
		associationSelectionPanel.add(spacerWidget);
		associationSelectionPanel.setCellWidth(spacerWidget, "10");

		associatedEntitiesListBox = new ListBoxWidget<Long>(true);
		associatedEntitiesListBox.setVisibleItemCount(5);
		associatedEntitiesListBox.setVisible(false);
		associationSelectionPanel.add(associatedEntitiesListBox);

		associatedEntityTypeListBox.addChangeHandler(new ChangeHandler() {
			@Override
			public void onChange(ChangeEvent event) {
				updateEntities(associatedEntityTypeListBox.getSelectedItem(), null);
				if(listener != null)
					listener.valueChanged(associatedEntityTypeListBox.getSelectedItem());
			}});
		
		initialize(selectedEntityType, selectedEntities);

		initWidget(disclosurePanel);
	}
	
	private void initialize(final String selectedEntityType, final Set<Long> selectedEntities) {
		busyIndicator.center();
		service.getAllEntityTypes(new NeptuneAsyncCallback<Map<String, String>>() {
			@Override
			public void onNeptuneFailure(Throwable caught) {
				busyIndicator.hide();
				Window.alert("Failed to get list of entity types." + Common.getFailureMessageSuffix(caught));
			}

			@Override
			public void onNeptuneSuccess(Map<String, String> entityTypes) {
				for(Map.Entry<String, String> entry : entityTypes.entrySet()) {
					associatedEntityTypeListBox.addItem(entry.getValue(), entry.getKey());
				}
				if(selectedEntityType != null) {
					associatedEntityTypeListBox.setSelectedItem(selectedEntityType);
				}
				else {
					associatedEntityTypeListBox.setSelectedIndex(0);
				}
				updateEntities(associatedEntityTypeListBox.getSelectedItem(), selectedEntities);
				busyIndicator.hide();
			}
		});
	}
	
	private void updateEntities(final String selectedEntityType, final Set<Long> selectedEntities) {
		if(selectedEntityType == null)	//this should not happen, as default is "Self"
			return;
		
		busyIndicator.center();
		service.getEntitiesOfType(selectedEntityType, new NeptuneAsyncCallback<Map<Long, String>>() {
			@Override
			public void onNeptuneFailure(Throwable caught) {
				busyIndicator.hide();
				Window.alert("Failed to get list of entities of type: " + selectedEntityType
						+ Common.getFailureMessageSuffix(caught));
			}

			@Override
			public void onNeptuneSuccess(Map<Long, String> entities) {
				//section: update the entities list box
				associatedEntitiesListBox.clear();
				if(entities == null) {
					associatedEntitiesListBox.setVisible(false);
				} else {
					//add entities to list box
					for(Map.Entry<Long, String> entry : entities.entrySet()) {
						associatedEntitiesListBox.addItem(entry.getValue(), entry.getKey());
					}
					//select in the list box all the selectedEntities
					if(selectedEntities != null) {
						for(Long selectedEntity : selectedEntities) {
							int index = associatedEntitiesListBox.getIndex(selectedEntity);
							//skip if selected entity is not available anymore
							if(index == -1)
								continue;
							associatedEntitiesListBox.setItemSelected(index, true);
						}
					}
					associatedEntitiesListBox.setVisible(true);
				}
				
				busyIndicator.hide();
			}
		});
	}
	
	public void setValueChangeListener(ValueChangeListener listener) {
		this.listener = listener;
	}
	
	public String getEntityType() {
		return associatedEntityTypeListBox.getSelectedItem();
	}
	
	public Set<Long> getEntities() {
		Set<Long> entities = new HashSet<Long>();
		for(int i = 0; i < associatedEntitiesListBox.getItemCount(); ++i) {
			if(associatedEntitiesListBox.isItemSelected(i)) {
				entities.add(associatedEntitiesListBox.getItem(i));
			}
		}
		return entities;
	}
	
	@Override
	protected void onUnload() {
		super.onUnload();
		busyIndicator.hide();
	}

	public Collection<? extends String> validate() {
		List<String> messages = new ArrayList<String>();
		if(associatedEntitiesListBox.isVisible() && getEntities().isEmpty())
			messages.add("Association is required");
		return messages;
	}
	
	/**
	 * This function may not function correctly if this method is called
	 * before or during initialization of this widget.
	 */
	public String getEntityTypeDisplayName(String entityType) {
		for(int i = 0; i < associatedEntityTypeListBox.getItemCount(); ++i) {
			if(associatedEntityTypeListBox.getItem(i).equals(entityType))
				return associatedEntityTypeListBox.getItemText(i);
		}
		return null;
	}

}
