package com.ericsson.cms.scheduling.ui.client.detailsPage;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import com.ericsson.cms.scheduling.ui.client.common.Common;
import com.ericsson.cms.scheduling.ui.client.common.SchedulingConstants;
import com.ericsson.cms.scheduling.ui.client.data.UITemplateVariable;
import com.ericsson.cms.scheduling.ui.client.data.UiMetadataTree;
import com.ericsson.cms.scheduling.ui.client.service.IUISchedulerService;
import com.ericsson.cms.scheduling.ui.client.service.IUISchedulerServiceAsync;
import com.ericsson.cms.scheduling.ui.client.widgets.MetadataFieldWidget;
import com.google.gwt.core.client.GWT;
import com.google.gwt.event.dom.client.ChangeEvent;
import com.google.gwt.event.dom.client.ChangeHandler;
import com.google.gwt.user.client.Command;
import com.google.gwt.user.client.Window;
import com.google.gwt.user.client.ui.Composite;
import com.tandbergtv.neptune.widgettoolkit.client.remote.NeptuneAsyncCallback;
import com.tandbergtv.neptune.widgettoolkit.client.widget.basic.LabelWidget;
import com.tandbergtv.neptune.widgettoolkit.client.widget.basic.ListBoxWidget;
import com.tandbergtv.neptune.widgettoolkit.client.widget.composite.FormContainer;
import com.tandbergtv.neptune.widgettoolkit.client.widget.composite.MultiTaskBusyIndicator;
import com.tandbergtv.neptune.widgettoolkit.client.widget.container.DisclosureContainer;
import com.tandbergtv.neptune.widgettoolkit.client.widget.container.HorizontalContainer;
import com.tandbergtv.neptune.widgettoolkit.client.widget.container.RoundedDisclosureContainer;
import com.tandbergtv.neptune.widgettoolkit.client.widget.container.SimpleContainer;
import com.tandbergtv.neptune.widgettoolkit.client.widget.container.VerticalContainer;

public class ExecutionInfoWidget extends Composite {
	private IUISchedulerServiceAsync service = GWT.create(IUISchedulerService.class);
	private SchedulingConstants constants = GWT.create(SchedulingConstants.class);

	private String entityType;
	private boolean associatedWithTitles;
	private UiMetadataTree metadataTree;
	
	private MultiTaskBusyIndicator busyIndicator = new MultiTaskBusyIndicator();
	
	private ListBoxWidget<String> templateListBox;
	private ListBoxWidget<String> priorityListBox;
	private SimpleContainer paramsContainer;
	
	private Map<String, MetadataFieldWidget> paramWidgetsMap = new HashMap<String, MetadataFieldWidget>();
	private List<UITemplateVariable> params;
	
	public ExecutionInfoWidget(String entityType, boolean associatedWithTitles,
			final String templateName, final String priority, final Map<String, String> paramValues) {
		this.entityType = entityType;
		this.associatedWithTitles = associatedWithTitles;
		
		DisclosureContainer disclosurePanel = new RoundedDisclosureContainer(constants.executeWorkOrder(),true);
		disclosurePanel.addStyleName(Common.STYLE_DISCLOSURE_CONTAINER);
		
		VerticalContainer sectionContainer = new VerticalContainer();
		disclosurePanel.add(sectionContainer);
		
		FormContainer templatePriorityForm = new FormContainer(HorizontalContainer.ALIGN_CENTER);
		templateListBox = new ListBoxWidget<String>();
		templatePriorityForm.addRow(constants.template(), templateListBox, true);

		priorityListBox = new ListBoxWidget<String>();
		//TODO get it from the app backend
		priorityListBox.addItem(constants.priorityLowest(), "LOWEST");
		priorityListBox.addItem(constants.priorityLow(), "LOW");
		priorityListBox.addItem(constants.priorityNormal(), "NORMAL");
		priorityListBox.addItem(constants.priorityHigh(), "HIGH");
		priorityListBox.addItem(constants.priorityHighest(), "HIGHEST");
		priorityListBox.addItem(constants.priorityAssignByRules(), "_rules");
		priorityListBox.setSelectedItem(priority);
		templatePriorityForm.addRow(constants.priority(), priorityListBox, true);
		sectionContainer.add(templatePriorityForm);
		
		LabelWidget jobParamsHeading = new LabelWidget(constants.jobParameters());
		sectionContainer.add(jobParamsHeading);
		
		paramsContainer = new SimpleContainer();
		sectionContainer.add(paramsContainer);

		templateListBox.addChangeHandler(new ChangeHandler() {
			@Override
			public void onChange(ChangeEvent event) {
				showParams(templateListBox.getSelectedItem(), null);
			}
		});
		
		loadTemplates(templateName, associatedWithTitles);
		loadMetadataTree(new Command() {
			@Override
			public void execute() {
				showParams(templateName, paramValues);
			}
		});
		
		initWidget(disclosurePanel);

	}
	
	private void loadTemplates(final String selectedTemplate, boolean associatedWithTitles) {
		busyIndicator.center();
		service.getTemplates(associatedWithTitles, new NeptuneAsyncCallback<List<String>>() {
			@Override
			public void onNeptuneFailure(Throwable caught) {
				busyIndicator.hide();
				Window.alert("Failed to get list of templates." + Common.getFailureMessageSuffix(caught));
			}
			@Override
			public void onNeptuneSuccess(List<String> templates) {
				templateListBox.clear();
				for(String template : templates) {
					templateListBox.addItem(template, template);
				}
				templateListBox.setSelectedItem(selectedTemplate);
				busyIndicator.hide();
			}
		});
	}
	
	private void showParams(final String templateName, final Map<String, String> paramValues) {
		paramsContainer.clear();
		paramWidgetsMap.clear();
		if(params != null)
			params.clear();

		if(templateName == null)
			return;

		busyIndicator.center();
		service.getStartVariablesForTemplate(templateName, associatedWithTitles,
				new NeptuneAsyncCallback<List<UITemplateVariable>>() {
			@Override
			public void onNeptuneFailure(Throwable caught) {
				busyIndicator.hide();
				Window.alert("Failed to get list of params for template: " + templateName
						+ Common.getFailureMessageSuffix(caught));
			}

			@Override
			public void onNeptuneSuccess(List<UITemplateVariable> variables) {
				params = variables;
				if(variables != null) {
					FormContainer paramsForm = new FormContainer(HorizontalContainer.ALIGN_CENTER);
					paramsContainer.setWidget(paramsForm);
					for(UITemplateVariable var : variables) {
						MetadataFieldWidget treeWidget = new MetadataFieldWidget();
						treeWidget.setEditable(true);
						treeWidget.setVisibleLength(50);
						treeWidget.setMetadataTree(metadataTree);
						paramWidgetsMap.put(var.getName(), treeWidget);
						if(var.isAutoMapped())
							treeWidget.setValue("_autoMapped");
						if(paramValues != null)
							treeWidget.setValue(paramValues.get(var.getName()));
						//if it is an automapped variable, don't show the field
						if(!var.isAutoMapped())
							paramsForm.addRow(var.getDisplayName() + ":", treeWidget, var.isRequired());
					}
				}
				busyIndicator.hide();
			}
		});
	}

	public void setEntityType(String entityType) {
		this.entityType = entityType;
		loadMetadataTree(new Command() {
			@Override
			public void execute() {
				showParams(templateListBox.getSelectedItem(), null);
			}
		});
	}
	
	public void setAssociatedWithTitles(boolean associatedWithTitles) {
		this.associatedWithTitles = associatedWithTitles;
		loadTemplates(templateListBox.getSelectedItem(), associatedWithTitles);
		loadMetadataTree(new Command() {
			@Override
			public void execute() {
				showParams(templateListBox.getSelectedItem(), null);
			}
		});
	}
	
	private void loadMetadataTree(final Command command) {
		busyIndicator.center();
		service.getWOMappingMetadataTree(entityType, associatedWithTitles, new NeptuneAsyncCallback<UiMetadataTree>() {
			@Override
			public void onNeptuneFailure(Throwable caught) {
				busyIndicator.hide();
				Window.alert("Failed to get workorder mapping metadata tree for entity type: " + entityType
						+ " | associatedWithTitles: " + associatedWithTitles + Common.getFailureMessageSuffix(caught));
			};
			@Override
			public void onNeptuneSuccess(UiMetadataTree result) {
				metadataTree = result;
				command.execute();
				busyIndicator.hide();
			};
		});
	}
	
	public List<String> validate() {
		List<String> messages = new ArrayList<String>();
		if(Common.isBlank(getTemplateName()))
			messages.add("Template Name is a required field.");
		if(Common.isBlank(getPriority()))
			messages.add("Priority is a required field.");
		if(params != null) {
			for(UITemplateVariable var : params) {
				if(var.isRequired()) {
					if(Common.isBlank(paramWidgetsMap.get(var.getName()).getValue())) {
						messages.add("Job parameter " + var.getDisplayName() + " is required.");
					}
				}
			}
		}
		return messages;
	}

	public String getTemplateName() {
		return templateListBox.getSelectedItem();
	}

	public String getPriority() {
		return priorityListBox.getSelectedItem();
	}

	public Map<String, String> getWoMappingParams() {
		Map<String, String> params = new HashMap<String, String>();
		for(Map.Entry<String, MetadataFieldWidget> entry : paramWidgetsMap.entrySet()) {
			String value = entry.getValue().getValue();
			if(!Common.isBlank(value))
				params.put(entry.getKey(), value);
		}
		return params;
	}

	@Override
	protected void onUnload() {
		super.onUnload();
		busyIndicator.hideAll();
	}
}
