package com.ericsson.cms.scheduling.ui.client.detailsPage;

import java.util.ArrayList;
import java.util.HashSet;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.Set;

import com.ericsson.cms.scheduling.ui.client.Action;
import com.ericsson.cms.scheduling.ui.client.Permissions;
import com.ericsson.cms.scheduling.ui.client.common.Common;
import com.ericsson.cms.scheduling.ui.client.common.SchedulingConstants;
import com.ericsson.cms.scheduling.ui.client.common.UiInvalidScheduleException;
import com.ericsson.cms.scheduling.ui.client.data.ScheduleType;
import com.ericsson.cms.scheduling.ui.client.data.UIWOJob;
import com.ericsson.cms.scheduling.ui.client.service.IUISchedulerService;
import com.ericsson.cms.scheduling.ui.client.service.IUISchedulerServiceAsync;
import com.google.gwt.core.client.GWT;
import com.google.gwt.core.client.Scheduler;
import com.google.gwt.core.client.Scheduler.ScheduledCommand;
import com.google.gwt.event.dom.client.ClickEvent;
import com.google.gwt.event.dom.client.ClickHandler;
import com.google.gwt.event.logical.shared.ResizeEvent;
import com.google.gwt.event.logical.shared.ResizeHandler;
import com.google.gwt.event.shared.HandlerRegistration;
import com.google.gwt.user.client.History;
import com.google.gwt.user.client.Window;
import com.google.gwt.user.client.ui.Composite;
import com.tandbergtv.neptune.widgettoolkit.client.remote.NeptuneAsyncCallback;
import com.tandbergtv.neptune.widgettoolkit.client.security.NeptuneSecurity;
import com.tandbergtv.neptune.widgettoolkit.client.widget.basic.ButtonWidget;
import com.tandbergtv.neptune.widgettoolkit.client.widget.basic.LabelWidget;
import com.tandbergtv.neptune.widgettoolkit.client.widget.basic.TextBoxWidget;
import com.tandbergtv.neptune.widgettoolkit.client.widget.composite.HeaderPanel;
import com.tandbergtv.neptune.widgettoolkit.client.widget.composite.MultiTaskBusyIndicator;
import com.tandbergtv.neptune.widgettoolkit.client.widget.composite.messagearea.MessageArea;
import com.tandbergtv.neptune.widgettoolkit.client.widget.container.HorizontalContainer;
import com.tandbergtv.neptune.widgettoolkit.client.widget.container.ScrollContainer;
import com.tandbergtv.neptune.widgettoolkit.client.widget.container.TabContainer;
import com.tandbergtv.neptune.widgettoolkit.client.widget.container.VerticalContainer;

/*
 * This widget allows for association, dissociation and modification of a schedule.
 * 
 * A user can associate if the user has schedule create permission
 *  and if the newly associated entity allows for a schedule to be created for it.
 * 
 * A user can dissociate if the user has schedule delete permission
 *  and if the dissociated entity allows for a schedule to be deleted from it.
 * 
 * For simplicity in implementation, this widget, if used for edit,
 *  will not show Save button if the user did not have Scheduler Edit permission
 *  or is not allowed by the associated entity.
 * So, in effect, the user cannot associate/dissociate just by having authorization
 *  to associate/dissociate (create/delete). He should also have authorization to modify.
 */
public class ScheduleDetailsWidget extends Composite {
	static enum ViewMode {
		/**
		 * User intended to show create page.
		 * Has scheduler create permission.
		 * Save action will verify if associated entity allows create.
		 */
		CREATE,
		/** 
		 * User intended to show details page.
		 * Does not have scheduler edit permission or is not allowed to edit by the associated entity,
		 *  but has scheduler view permission and is allowed to view by the associated entity.
		 */
		VIEW,
		/**
		 * User intended to show details page.
		 * Has scheduler edit permission and is allowed to edit by the associated entity.
		 */
		EDIT;
	}
	
	private static final String TAB_CONTAINER_STYLE = "scheduler-tabContainer";
	private static int CMS_HEADER_MENU_SIZE_PX = 115;
	
	private IUISchedulerServiceAsync service = GWT.create(IUISchedulerService.class);
	private SchedulingConstants constants = GWT.create(SchedulingConstants.class);

	private MultiTaskBusyIndicator busyIndicator = new MultiTaskBusyIndicator();
	
	private HeaderPanel headerWidget;
	private MessageArea messageArea;
	private TabContainer tabContainer;
	private VerticalContainer tabPanel;
	private ScrollContainer scrollPanel;
	private VerticalContainer contentPanel;
	private TextBoxWidget nameTextBox;
	private EntityAssociationWidget ea;
	private ScheduleInfoWidget si;
	private ExecutionInfoWidget ei;
	private HorizontalContainer buttonPanel;
	private ButtonWidget saveButton, activateButton, inactivateButton, runNowButton, cancelButton;
	
	private HandlerRegistration resizeHandlerRegistration;
	
	private NeptuneSecurity security;
	private UIWOJob uiWOJob;
	private ViewMode viewMode;
	
	protected ScheduleDetailsWidget(NeptuneSecurity security) {
		this.security = security;
		
		VerticalContainer pagePanel = new VerticalContainer();
		pagePanel.setWidth("100%");
		
		headerWidget = new HeaderPanel(null);
		pagePanel.add(headerWidget);
		
		messageArea = new MessageArea();
		messageArea.reset();
		pagePanel.add(messageArea);
		
		tabContainer = new TabContainer();
		tabContainer.addStyleName(TAB_CONTAINER_STYLE);
		pagePanel.add(tabContainer);
		
		tabPanel = new VerticalContainer();
		
		scrollPanel = new ScrollContainer();
		tabPanel.add(scrollPanel);
		
		contentPanel = new VerticalContainer();
		contentPanel.setSpacing(10);
		scrollPanel.add(contentPanel);
		
		buttonPanel = new HorizontalContainer();
		tabPanel.add(buttonPanel);

		initWidget(pagePanel);
	}
	
	//called to show create page
	public ScheduleDetailsWidget(final String associatedEntityType, Long associatedEntityId, NeptuneSecurity security) {
		this(security);
		
		if(!security.isUserInRole(Permissions.CREATE)) {
			Window.alert("You do not have permission to create a schedule.");
			return;
		}
		
		viewMode = ViewMode.CREATE;
		
		if(associatedEntityType != null && !Common.isUserAuthorized(associatedEntityType, Action.CREATE)) {
			busyIndicator.center();
			//get display name for associated entity and then show unauthorized access message
			//TODO optimize
			service.getAllEntityTypes(new NeptuneAsyncCallback<Map<String, String>>() {
				@Override
				public void onNeptuneFailure(Throwable caught) {
					busyIndicator.hide();
					Window.alert("You do not have permission to create a schedule for this entity.");
				}

				@Override
				public void onNeptuneSuccess(Map<String, String> entityTypes) {
					String entityTypeDisplayName = "this entity";
					for(Map.Entry<String, String> entry : entityTypes.entrySet()) {
						if(entry.getKey().equals(associatedEntityType)) {
							entityTypeDisplayName = entry.getValue();
							break;
						}
					}
					busyIndicator.hide();
					Window.alert("You do not have permission to create a schedule for "
							+ entityTypeDisplayName + ".");
				}
			});
			return;
		}
		
		uiWOJob = new UIWOJob();
		
		uiWOJob.setAssociatedEntityType(associatedEntityType);
		if(associatedEntityId != null) {
			Set<Long> associatedEntities = new HashSet<Long>();
			associatedEntities.add(associatedEntityId);
			uiWOJob.setAssociatedEntities(associatedEntities);
		}
		
		uiWOJob.setPriority("_rules");

		busyIndicator.center();
		if(Common.isBlank(associatedEntityType)) {
			service.isSelfPartnerAssociated(new NeptuneAsyncCallback<Boolean>() {

				@Override
				public void onNeptuneFailure(Throwable caught) {
					busyIndicator.hide();
					Window.alert("Checking SELF Partner association with user failed: "
							+ Common.getFailureMessageSuffix(caught));
				}

				@Override
				public void onNeptuneSuccess(Boolean result) {
					if (Boolean.TRUE.equals(result)) {
						uiWOJob.setAssociatedEntityType("SELF");
					}
					else {
						uiWOJob.setAssociatedEntityType("SOURCE");
					}
					show(uiWOJob);
					busyIndicator.hide();
				}
				
			});
		}
		else {
			show(uiWOJob);
			busyIndicator.hide();
		}
	}
	
	//called to show view or edit page
	public ScheduleDetailsWidget(final long jobId, final NeptuneSecurity security) {
		this(security);
		
		//preliminary security check
		if(!security.isUserInRole(Permissions.EDIT) && !security.isUserInRole(Permissions.VIEW)) {
			//you have no business here, get the hell out
			Window.alert("You do not have permission to view or edit a schedule.");
			return;
		}
		
		busyIndicator.center();
		service.getWOJobByJobId(jobId, new NeptuneAsyncCallback<UIWOJob>() {
			@Override
			public void onNeptuneFailure(Throwable caught) {
				busyIndicator.hide();
				Window.alert("Getting workorder job failed for jobId: " + jobId
						+ Common.getFailureMessageSuffix(caught));
			}

			@Override
			public void onNeptuneSuccess(UIWOJob result) {
				uiWOJob = result;
				if(security.isUserInRole(Permissions.EDIT) && 
						Common.isUserAuthorized(uiWOJob.getAssociatedEntityType(), Action.MODIFY)) {
					viewMode = ViewMode.EDIT;
				} else if(security.isUserInRole(Permissions.VIEW) && 
						Common.isUserAuthorized(uiWOJob.getAssociatedEntityType(), Action.VIEW)) {
					viewMode = ViewMode.VIEW;
				} else {
					busyIndicator.hide();
					Window.alert("You do not have permission to view or edit this schedule.");
					return;
				}
				show(uiWOJob);
				busyIndicator.hide();
			}
		});
	}
	
	private void show(final UIWOJob woJob) {
		setHeaderLabelAndTab(woJob);
		
		HorizontalContainer namePanel = new HorizontalContainer();
		namePanel.add(Common.getRequiredFieldName(constants.name()));
		
		LabelWidget spacerWidget = new LabelWidget();
		namePanel.add(spacerWidget);
		namePanel.setCellWidth(spacerWidget, "10");
		
		nameTextBox = new TextBoxWidget();
		nameTextBox.setVisibleLength(constants.maxVisibleLengthJobName());
		nameTextBox.setText(woJob.getName());
		namePanel.add(nameTextBox);
		
		contentPanel.add(namePanel);
		
		ea = new EntityAssociationWidget(woJob.getAssociatedEntityType(), woJob.getAssociatedEntities());
		contentPanel.add(ea);
		
		si = new ScheduleInfoWidget(woJob.getAssociatedEntityType(), woJob.getScheduleInfo());
		contentPanel.add(si);
		
		boolean associatedWithTitles = true;
		if(woJob.getScheduleInfo() != null)
			associatedWithTitles = (woJob.getScheduleInfo().getScheduleType() == ScheduleType.N_DAYS_FROM_METADATA);
		
		ei = new ExecutionInfoWidget(woJob.getAssociatedEntityType(), associatedWithTitles,
				woJob.getTemplateName(), woJob.getPriority(), woJob.getWoMappingParams());
		contentPanel.add(ei);
		
		ea.setValueChangeListener(new EntityAssociationWidget.ValueChangeListener() {
			@Override
			public void valueChanged(String entityType) {
				si.setEntityType(entityType);
				ei.setEntityType(entityType);
			}
		});
		
		si.setValueChangeListener(new ScheduleInfoWidget.ValueChangeListener() {
			@Override
			public void valueChanged(boolean associatedWithTitles) {
				ei.setAssociatedWithTitles(associatedWithTitles);
			}
		});
		
		//save button
		saveButton = new ButtonWidget(constants.buttonLabelSave());
		saveButton.addStyleDependentName("do-button");
		saveButton.setVisible(false);
		saveButton.addClickHandler(new ClickHandler() {
			@Override
			public void onClick(ClickEvent event) {
				busyIndicator.center();
				List<String> validationMessages = validate();
				if(!validationMessages.isEmpty()) {
					setErrorMessages(validationMessages);
					busyIndicator.hide();
					return;
				}
				else {
					// Check if job name is already in use.
					final String jobName = nameTextBox.getText().trim();
					service.isJobNameAlreadyUsed(jobName, new NeptuneAsyncCallback<Boolean>() {

						@Override
						public void onNeptuneFailure(Throwable caught) {
							setErrorMessage("Failed to check if Job Name is unique." 
									+ Common.getFailureMessageSuffix(caught));
							busyIndicator.hide();
						}

						@Override
						public void onNeptuneSuccess(Boolean result) {
							if ((Boolean.FALSE.equals(result)) || (jobName.equals(uiWOJob.getName()))) {
								busyIndicator.center();
								service.save(getData(), new NeptuneAsyncCallback<UIWOJob>() {
									@Override
									public void onNeptuneFailure(Throwable caught) {
										if (caught instanceof UiInvalidScheduleException) {
											UiInvalidScheduleException uise = (UiInvalidScheduleException) caught;
											List<String> errorMessages = new ArrayList<String>();
											for (String reasonCode : uise.getReasons().keySet()) {
												if ("errInvalidSchedule".equals(reasonCode)) {
													errorMessages.add(constants.errInvalidSchedule());
												}
												else {
													errorMessages.add(constants.errDefault());
												}
											}
											setErrorMessages(errorMessages);
										}
										else {
											setErrorMessage("Failed to save the schedule."
													+ Common.getFailureMessageSuffix(caught));
										}
										busyIndicator.hide();
									}

									@Override
									public void onNeptuneSuccess(UIWOJob result) {
										uiWOJob = result;
										setInfoMessage("Successfully saved the schedule.");
										if(viewMode == ViewMode.CREATE) {
											if(security.isUserInRole(Permissions.EDIT) &&
													Common.isUserAuthorized(uiWOJob.getAssociatedEntityType(), Action.MODIFY))
												viewMode = ViewMode.EDIT;
											else
												viewMode = ViewMode.VIEW;
											/* The user may not have view permission either.
											 * He just created it, So, I'll let him view it. :)
											 */
										}
										//update header label and tab text
										setHeaderLabelAndTab(uiWOJob);
										showAppropriateButtons();
										busyIndicator.hide();
									}
								});
							}
							else {
								setErrorMessage(constants.jobNameIsAlredayInUse());
							}
							busyIndicator.hide();
						}
						
					});
				}
				// End else
			}});
		buttonPanel.add(saveButton);

		//activate button
		activateButton = new ButtonWidget(constants.buttonLabelActivate());
		activateButton.addStyleDependentName("do-button");
		activateButton.setVisible(false);
		activateButton.addClickHandler(new ClickHandler() {
			@Override
			public void onClick(ClickEvent event) {
				Set<Long> ids = new HashSet<Long>();
				ids.add(uiWOJob.getId());
				service.setActive(ids, true, new NeptuneAsyncCallback<Void>() {
					@Override
					public void onNeptuneFailure(Throwable caught) {
						setErrorMessage("Failed to activate the schedule." + Common.getFailureMessageSuffix(caught));
					}

					@Override
					public void onNeptuneSuccess(Void result) {
						setInfoMessage("Successfully activated the schedule.");
						uiWOJob.setActive(true);
						showAppropriateButtons();
					}
				});
			}});
		buttonPanel.add(activateButton);

		//run-now button
		runNowButton = new ButtonWidget(constants.buttonLabelRunNow());
		runNowButton.addStyleDependentName("do-button");
		runNowButton.setVisible(false);
		runNowButton.addClickHandler(new ClickHandler() {
			@Override
			public void onClick(ClickEvent event) {
				Set<Long> ids = new HashSet<Long>();
				ids.add(uiWOJob.getJobId());
				service.runNow(ids, new NeptuneAsyncCallback<Void>() {
					@Override
					public void onNeptuneFailure(Throwable caught) {
						setErrorMessage("Failed to run schedule." + Common.getFailureMessageSuffix(caught));
					}

					@Override
					public void onNeptuneSuccess(Void result) {
						setInfoMessage("Requested to run the schedule.");
					}
				});
			}});
		buttonPanel.add(runNowButton);
	
		//inactivate button
		inactivateButton = new ButtonWidget(constants.buttonLabelInactivate());
		inactivateButton.addStyleDependentName("caution-button");
		inactivateButton.setVisible(false);
		inactivateButton.addClickHandler(new ClickHandler() {
			@Override
			public void onClick(ClickEvent event) {
				Set<Long> ids = new HashSet<Long>();
				ids.add(uiWOJob.getId());
				service.setActive(ids, false, new NeptuneAsyncCallback<Void>() {
					@Override
					public void onNeptuneFailure(Throwable caught) {
						setErrorMessage("Failed to inactivate the schedule." + Common.getFailureMessageSuffix(caught));
					}

					@Override
					public void onNeptuneSuccess(Void result) {
						setInfoMessage("Successfully inactivated the schedule.");
						uiWOJob.setActive(false);
						showAppropriateButtons();
					}
				});
			}});
		buttonPanel.add(inactivateButton);
		
		//cancel button
		cancelButton = new ButtonWidget(constants.buttonLabelCancel());
		cancelButton.addStyleDependentName("caution-button");
		cancelButton.setVisible(false);
		cancelButton.addClickHandler(new ClickHandler() {
			@Override
			public void onClick(ClickEvent event) {
				History.back();
			}});
		buttonPanel.add(cancelButton);

		showAppropriateButtons();
		Scheduler.get().scheduleDeferred(new ScheduledCommand() {
			@Override
			public void execute() {
				updateScrollPanelSize();
				nameTextBox.setFocus(true);
			}
		});
	}
	
	private void setHeaderLabelAndTab(UIWOJob woJob) {
		switch (viewMode) {
		case CREATE:
			headerWidget.setLabel(constants.headingCreate());
			tabContainer.add(tabPanel, "Create");
			break;
		case VIEW:
			headerWidget.setLabel(constants.headingView() + woJob.getName());
			tabContainer.add(tabPanel, "View");
			break;
		case EDIT:
			headerWidget.setLabel(constants.headingEdit() + woJob.getName());
			tabContainer.add(tabPanel, "Edit");
			break;
		}
		tabContainer.selectTab(0);
	}

	private void showAppropriateButtons() {
		saveButton.setVisible(false);
		activateButton.setVisible(false);
		inactivateButton.setVisible(false);
		runNowButton.setVisible(false);
		cancelButton.setVisible(true);
		
		if(viewMode == ViewMode.CREATE || viewMode == ViewMode.EDIT)
			saveButton.setVisible(true);
		
		if(viewMode == ViewMode.EDIT) {
			if(uiWOJob.isActive()) {
				if(Common.isUserAuthorized(uiWOJob.getAssociatedEntityType(), Action.INACTIVATE))
					inactivateButton.setVisible(true);
			} else {
				if(Common.isUserAuthorized(uiWOJob.getAssociatedEntityType(), Action.ACTIVATE))
					activateButton.setVisible(true);
			}
		}
		
		if(viewMode != ViewMode.CREATE && 
				uiWOJob.isActive() && 
				security.isUserInRole(Permissions.EXECUTE) &&
				Common.isUserAuthorized(uiWOJob.getAssociatedEntityType(), Action.EXECUTE)) {
			runNowButton.setVisible(true);
		}
	}
	
	private void updateScrollPanelSize() {
		int height = Window.getClientHeight()
			- CMS_HEADER_MENU_SIZE_PX
			- headerWidget.getOffsetHeight()
			- messageArea.getOffsetHeight()
			- tabContainer.getTabBar().getOffsetHeight()
			- buttonPanel.getOffsetHeight()
			- 14;	//adjustment for borders and spacing at the bottom
		scrollPanel.setHeight(height + "px");
	}
	
	private void setInfoMessage(String message) {
		messageArea.setInfoMessage(message);
		updateScrollPanelSize();
	}

	private void setErrorMessage(String message) {
		messageArea.setErrorMessage(message);
		updateScrollPanelSize();
	}
	
	private void setErrorMessages(List<String> messages) {
		StringBuilder sb = new StringBuilder();
		
		sb.append("<ul>");
		for(String message : messages)
			sb.append("<li>").append(message).append("</li>");
		sb.append("</ul>");
		
		messageArea.setErrorHTML(sb.toString());
		updateScrollPanelSize();
	}
	
	public List<String> validate() {
		List<String> messages = new ArrayList<String>();
		String jobName = nameTextBox.getText();
		if (Common.isBlank(jobName)) {
			messages.add(constants.jobNameIsRequired());
		}
		else {
			jobName = jobName.trim();
			int jobNameMaxLength = constants.maxLengthJobName();
			if ( jobNameMaxLength < jobName.length()) {
				messages.add(constants.jobNameCannotBeLonger());
			}
		}
		messages.addAll(ea.validate());
		messages.addAll(si.validate());
		messages.addAll(ei.validate());
		
		messages.addAll(verifyAssociationPermissions(uiWOJob, getData()));

		return messages;
	}
	
	public UIWOJob getData() {
		UIWOJob uiWOJob = new UIWOJob();
		uiWOJob.setId(this.uiWOJob.getId());
		uiWOJob.setName(nameTextBox.getText().trim());
		uiWOJob.setAssociatedEntityType(ea.getEntityType());
		uiWOJob.setAssociatedEntities(ea.getEntities());
		try {
			uiWOJob.setScheduleInfo(si.getScheduleInfo());
		} catch (Exception e) {
		}
		uiWOJob.setTemplateName(ei.getTemplateName());
		uiWOJob.setPriority(ei.getPriority());
		uiWOJob.setWoMappingParams(ei.getWoMappingParams());
		return uiWOJob;
	}
	
	private List<String> verifyAssociationPermissions(UIWOJob oldJob, UIWOJob newJob) {
		List<String> errorMessages = new LinkedList<String>();
		if(viewMode == ViewMode.CREATE) {
			//just verify in the new job
			if(!authorizedToAssociate(newJob.getAssociatedEntityType())) {
				errorMessages.add("You do not have permission to create a schedule for "
						+ ea.getEntityTypeDisplayName(newJob.getAssociatedEntityType()));
			}
		} else if(viewMode == ViewMode.EDIT) {
			//if the entity type has not been changed
			if(newJob.getAssociatedEntityType().equals(oldJob.getAssociatedEntityType())) {
				Set<Long> oldEntities = new HashSet<Long>(oldJob.getAssociatedEntities());
				Set<Long> newEntities = new HashSet<Long>(newJob.getAssociatedEntities());
				
				oldEntities.removeAll(newEntities);
				Set<Long> removedEntities = oldEntities;
				
				oldEntities = new HashSet<Long>(oldJob.getAssociatedEntities());
				newEntities.removeAll(oldEntities);
				Set<Long> addedEntities = newEntities;
				
				if(!removedEntities.isEmpty()) {
					//does the user have permission to delete schedule for this entity type?
					if(!authorizedToDissociate(newJob.getAssociatedEntityType())) {
						errorMessages.add("You do not have permission to dissociate this schedule from "
								+ ea.getEntityTypeDisplayName(newJob.getAssociatedEntityType()));
					}
				}
				
				if(!addedEntities.isEmpty()) {
					//does the user have permission to create schedule for this entity type?
					if(!authorizedToAssociate(newJob.getAssociatedEntityType())) {
						errorMessages.add("You do not have permission to associate this schedule with "
								+ ea.getEntityTypeDisplayName(newJob.getAssociatedEntityType()));
					}
				}
			} else {	//if the entity type has been changed
				//does the user have permission to delete schedule for old entity type?
				if(!authorizedToDissociate(oldJob.getAssociatedEntityType())) {
					errorMessages.add("You do not have permission to dissociate this schedule from "
							+ ea.getEntityTypeDisplayName(oldJob.getAssociatedEntityType()));
				}
				
				//does the user have permission to create schedule for new entity type?
				if(!authorizedToAssociate(newJob.getAssociatedEntityType())) {
					errorMessages.add("You do not have permission to associate this schedule with "
							+ ea.getEntityTypeDisplayName(newJob.getAssociatedEntityType()));
				}
			}
		}
		return errorMessages;
	}
	
	private boolean authorizedToAssociate(String entityType) {
		return security.isUserInRole(Permissions.CREATE) && Common.isUserAuthorized(entityType, Action.CREATE);
	}
	
	private boolean authorizedToDissociate(String entityType) {
		return security.isUserInRole(Permissions.DELETE) && Common.isUserAuthorized(entityType, Action.DELETE);
	}
	
	@Override
	protected void onLoad() {
		super.onLoad();
		resizeHandlerRegistration = Window.addResizeHandler(new ResizeHandler() {
			public void onResize(ResizeEvent event) {
				updateScrollPanelSize();
			}
		});
	};

	@Override
	protected void onUnload() {
		resizeHandlerRegistration.removeHandler();
		resizeHandlerRegistration = null;
		busyIndicator.hide();
		super.onUnload();
	}
}
