package com.ericsson.cms.sites.ui.client;

import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import com.ericsson.cms.sites.ui.client.entities.UITemplateInfo;
import com.ericsson.cms.sites.ui.client.entities.UITemplateVariable;
import com.ericsson.cms.sites.ui.client.entities.UITemplateVariableDataType;
import com.ericsson.cms.sites.ui.client.entities.UIResourceGroup;
import com.ericsson.cms.sites.ui.client.i18n.SitesConstants;
import com.ericsson.cms.sites.ui.client.i18n.SitesMessages;
import com.ericsson.cms.sites.ui.client.images.ISiteClientBundle;
import com.google.gwt.core.client.GWT;
import com.google.gwt.event.dom.client.ChangeEvent;
import com.google.gwt.event.dom.client.ChangeHandler;
import com.google.gwt.i18n.client.DateTimeFormat;
import com.google.gwt.user.client.rpc.AsyncCallback;
import com.google.gwt.user.client.ui.Composite;
import com.google.gwt.user.client.ui.HorizontalPanel;
import com.tandbergtv.neptune.widgettoolkit.client.widget.basic.ImageWidget;
import com.tandbergtv.neptune.widgettoolkit.client.widget.basic.LabelWidget;
import com.tandbergtv.neptune.widgettoolkit.client.widget.basic.ListBoxWidget;
import com.tandbergtv.neptune.widgettoolkit.client.widget.basic.TextBoxWidget;
import com.tandbergtv.neptune.widgettoolkit.client.widget.composite.BusyIndicator;
import com.tandbergtv.neptune.widgettoolkit.client.widget.composite.FormContainer;
import com.tandbergtv.neptune.widgettoolkit.client.widget.container.HorizontalContainer;
import com.tandbergtv.neptune.widgettoolkit.client.widget.container.VerticalContainer;

public class DistConfPanel extends Composite {
	private VerticalContainer mainPanel;
	private HorizontalContainer listBoxContainer;
	private ListBoxWidget<UITemplateInfo> confListBox;
	private FormContainer form;
	private Map<String, TextBoxWidget> paramWidgets = new HashMap<String, TextBoxWidget>();
	private ListBoxWidget<Long> resourcesListBox;
	private ListBoxWidget<Long> groupsListBox;
	private ImageWidget isNotLoadedImage;
	Collection<UIResourceGroup> groupsResult;
	private BusyIndicator busyIndicator = new BusyIndicator();
	
	private IUISiteServiceAsync service = GWT.create(IUISiteService.class);
	private SitesConstants constants = GWT.create(SitesConstants.class);
	private SitesMessages messages = GWT.create(SitesMessages.class);
	private ISiteClientBundle clientBundle = GWT.create(ISiteClientBundle.class);

	private boolean isAssociatedResourceRequiredField = false;
	
	private static final String PARAM_VALUE_DATE_FORMAT = "yyyy-MM-dd";

	public DistConfPanel() {
		initWidgets();
		initWidget(mainPanel);
	}
	
	private void initWidgets() {
		mainPanel = new VerticalContainer();
		
		listBoxContainer = new HorizontalContainer();
		listBoxContainer.setSpacing(1);
		isNotLoadedImage = new ImageWidget();
		isNotLoadedImage.setResource(clientBundle.sitePluginConfigurationError());
		isNotLoadedImage.setVisible(false);
		listBoxContainer.add(isNotLoadedImage);
		
		confListBox = new ListBoxWidget<UITemplateInfo>();
		confListBox.addChangeHandler(new ChangeHandler() {
			@Override
			public void onChange(ChangeEvent event) {
				updateForm(confListBox.getSelectedItem(), null, null);
			}});

		listBoxContainer.add(confListBox);
		
		mainPanel.add(listBoxContainer);
		form = new FormContainer(HorizontalPanel.ALIGN_LEFT);
		resourcesListBox = new ListBoxWidget<Long>();
		mainPanel.add(form);
	}
	
	private void updateForm(final UITemplateInfo ti, Map<String, String> paramValues, final Long resourceId) {
		busyIndicator.center();
		if(ti == null) {
			isNotLoadedImage.setVisible(false);
			setSelectedTemplateInfo(null);
			setParams(null);
			setParamValues(null);
			resourcesListBox = new ListBoxWidget<Long>();
			groupsListBox = new ListBoxWidget<Long>();
			busyIndicator.hide();
		} else {
			isNotLoadedImage.setVisible(!ti.isLoaded());
			setSelectedTemplateInfo(ti);
			setParams(ti.getRequiredVariables());
			setParamValues(paramValues);
			resourcesListBox = new ListBoxWidget<Long>();
			groupsListBox = new ListBoxWidget<Long>();
			final String rgName = ti.getAssociatedResourceGroupName();
			isAssociatedResourceRequiredField = false;
			
			form.addRow("Associated Group", groupsListBox, false);
			service.getAllGroupsByTemplateName(ti.getTemplateName(), new AsyncCallback<Collection<UIResourceGroup>>() {

				@Override
				public void onFailure(Throwable caught) {
					busyIndicator.hide();
					
				}

				@Override
				public void onSuccess(Collection<UIResourceGroup> result) {
					if(result != null) {
						groupsResult = result;
						groupsListBox.addItem("", null); //default
					for(UIResourceGroup group: result) {
						groupsListBox.addItem(group.getGroupName(), group.getGroupId());
					}
					groupsListBox.setSelectedItem(getGroupId(rgName, result));
					if(rgName != null) {
						showResources(rgName, resourceId);
					} else {
						busyIndicator.hide();
					}
					}
				}

				
				
			});
			
			groupsListBox.addChangeHandler(new ChangeHandler() {
				@Override
				public void onChange(ChangeEvent event) {
					updateFormWithResources(getGroupName(groupsListBox.getSelectedItem(), groupsResult), ti, resourceId);
				}

				

				});
			
		
		}
	}
	private void showResources(String groupName, final Long resourceId) {
		int x = form.getFieldRowIndex(resourcesListBox);
		if(x < 0) {//row does not exist
			form.addRow("Associated Resource", resourcesListBox, false);
		}
		resourcesListBox.clear();
//		resourcesListBox = new ListBoxWidget<Long>();
		service.getResourcesInGroup(groupName, new AsyncCallback<Map<Long, String>>() {
			@Override
			public void onFailure(Throwable caught) {
				busyIndicator.hide();
			}

			@Override
			public void onSuccess(Map<Long, String> resources) {
				resourcesListBox.addItem("", null); //default
				for(Map.Entry<Long, String> resource : resources.entrySet()) {
					resourcesListBox.addItem(resource.getValue(), resource.getKey());
				}
				if(resourceId != null) {
					resourcesListBox.setSelectedItem(resourceId);
				}
				busyIndicator.hide();
			}});
	}
	
		
	

	private void updateFormWithResources(String groupName, UITemplateInfo ti, final Long resourceId) {
		if(groupName == null) {
			isAssociatedResourceRequiredField = false;
			//if there was resource column - remove it
			int x = form.getFieldRowIndex(resourcesListBox);
			if(x>=0) {
				resourcesListBox.setSelectedItem(null);
				form.removeRow(x);
				
			}
			busyIndicator.hide();
		} else {
			if (groupName.trim().length() > 0) {
				isAssociatedResourceRequiredField = true;
				ti.setAssociatedResourceGroupName(groupName);
			}
			showResources(groupName, resourceId);
		}
			
	}
	private String getGroupName(Long id, Collection<UIResourceGroup>  groups) {
		if(groups != null) {
			for(UIResourceGroup group: groups) {
				if(group.getGroupId() != null && id != null) {
					if(group.getGroupId().longValue() == id.longValue()) {
						return group.getGroupName();
					}
				}
			}
		}
		return null;
	}
	private Long getGroupId(String associatedResourceGroupName, Collection<UIResourceGroup> result) {
		for(UIResourceGroup group: result) {
			if(group.getGroupName().equalsIgnoreCase(associatedResourceGroupName)) {
				return group.getGroupId();
			}
		}
		return null;
	}

	public void setData(Collection<UITemplateInfo> allTemplatesInfo,
			String selectedTemplateName, String groupName, Map<String, String> paramValues, Long associatedResource) {
		UITemplateInfo selectedTemplateInfo = null;
		if(allTemplatesInfo != null) {
			for(UITemplateInfo ti : allTemplatesInfo) {
				if(ti.getTemplateName().equals(selectedTemplateName)) {
					selectedTemplateInfo = ti;
					break;
				}
			}
		}
		if (selectedTemplateName != null && !selectedTemplateName.isEmpty()
				&& selectedTemplateInfo == null) {
			selectedTemplateInfo = new UITemplateInfo();
			selectedTemplateInfo.setTemplateName(selectedTemplateName);
			selectedTemplateInfo.setAssociatedResourceGroupName("");
			selectedTemplateInfo.setRequiredVariables(new HashMap<String, UITemplateVariable>());
			selectedTemplateInfo.setLoaded(false);
			allTemplatesInfo.add(selectedTemplateInfo);
		}
		setAllTemplatesInfo(allTemplatesInfo);
		if(groupName != null) {
			selectedTemplateInfo.setAssociatedResourceGroupName(groupName);
		}
		updateForm(selectedTemplateInfo, paramValues, associatedResource);
	}
	
	private void setAllTemplatesInfo(Collection<UITemplateInfo> templatesInfo) {
		confListBox.clear();
		if(templatesInfo != null) {
			for(UITemplateInfo ti : templatesInfo) {
				confListBox.addItem(ti.getTemplateName(), ti);
			}
		}
	}
	
	private void setSelectedTemplateInfo(UITemplateInfo ti) {
		if (ti == null) {
			ti = new UITemplateInfo();
			confListBox.insertItem("", ti, 0);
		}
		confListBox.setSelectedItem(ti);
	}
	
	private UITemplateInfo getSelectedTemplateInfo() {
		return confListBox.getSelectedItem();
	}
	
	private void setParams(Map<String, UITemplateVariable> params) {
		mainPanel.remove(form);
		form = new FormContainer(HorizontalPanel.ALIGN_LEFT);
		mainPanel.add(form);
		paramWidgets.clear();

		if(params != null) {
			for(Map.Entry<String, UITemplateVariable> param : params.entrySet()) {
				TextBoxWidget textBox = new TextBoxWidget();
				form.addRow(new LabelWidget(param.getValue().getMappedName()), textBox, true);
				paramWidgets.put(param.getKey(), textBox);
			}
		}
	}
	
	private void setParamValues(Map<String, String> paramValues) {
		if(paramValues != null) {
			for(Map.Entry<String, String> paramValue : paramValues.entrySet()) {
				TextBoxWidget textBox = paramWidgets.get(paramValue.getKey());
				if(textBox != null) {
					textBox.setValue(paramValue.getValue());
				} else {
					//TODO handle the case when a variable is removed from template
				}
			}
		}
	}
	
	public String getSelectedTemplateName() {
		UITemplateInfo selectedTemplateInfo = confListBox.getSelectedItem();
		return (selectedTemplateInfo != null) ? selectedTemplateInfo.getTemplateName() : null;
	}
	
	public Map<String, String> getParamValues() {
		Map<String, String> paramValues = new HashMap<String, String>();
		for(Map.Entry<String, TextBoxWidget> paramWidget : paramWidgets.entrySet()) {
			paramValues.put(paramWidget.getKey(), paramWidget.getValue().getValue());
		}
		return paramValues;
	}
	
	public Long getAssociatedResource() {
		return resourcesListBox.getSelectedItem();
	}
	
	public String getAssociatedGroupName() {
		return getGroupName(groupsListBox.getSelectedItem(), groupsResult);
	}
	public List<String> validate() {
		List<String> validationErrors = new ArrayList<String>();
		
		if (isAssociatedResourceRequiredField) {
			Long associatedResource = getAssociatedResource();
			if (associatedResource == null) {
				validationErrors.add(constants.associatedResourceIsRequired());
			}
		}
		else {
			Long associatedResource = getAssociatedResource();
			String groupName = getAssociatedGroupName();
			if (associatedResource == null && groupName != null) {
				validationErrors.add(constants.associatedResourceIsRequired());
			}
		}
		
		UITemplateInfo uiTemplateInfo = getSelectedTemplateInfo();
		Map<String, String> paramKeyValuesMap = getParamValues();
		String paramKey = null;
		String paramValue = null;
		for (Map.Entry<String, String> paramKeyValueEntry : paramKeyValuesMap.entrySet()) {
			paramKey = paramKeyValueEntry.getKey();
			paramValue = paramKeyValueEntry.getValue();
			if ((paramValue == null) || (paramValue.trim().length() < 1)) {
				validationErrors.add(messages.paramValueIsRequired(paramKey));
			}
			else {
				UITemplateVariable uiTemplateVariable = uiTemplateInfo.getRequiredVariables().get(paramKey);
				UITemplateVariableDataType uiTemplateVariableDataType = uiTemplateVariable.getDataType();
				if (UITemplateVariableDataType.BOOLEAN.equals(uiTemplateVariableDataType)) {
					Boolean booleanValue = Boolean.valueOf(paramValue);
					if (!((Boolean.TRUE.equals(booleanValue)) || (Boolean.FALSE.equals(booleanValue)))) {
						validationErrors.add(messages.paramValueRequiredBoolean(paramKey));
					}
				}
				else if (UITemplateVariableDataType.INTEGER.equals(uiTemplateVariableDataType)) {
					try {
						int intValue = Integer.valueOf(paramValue);
						if ((intValue < Integer.MIN_VALUE) || (intValue > Integer.MAX_VALUE)) {
							validationErrors.add(messages.paramValueRequiredInteger(paramKey));
						}
					} catch (Exception e) {
						validationErrors.add(messages.paramValueRequiredInteger(paramKey));
					}
				}
				else if (UITemplateVariableDataType.DATE.equals(uiTemplateVariableDataType)) {
					try {
						DateTimeFormat dateTimeFormat = DateTimeFormat.getFormat(PARAM_VALUE_DATE_FORMAT);
						dateTimeFormat.parse(paramValue);
					} catch (Exception e) {
						validationErrors.add(messages.paramValueRequiredDate(paramKey));
					}
				}

			}
		}
		
		return validationErrors;
	}
	
	protected void onUnload() {
		super.onUnload();
		busyIndicator.hide();
	}
}
