/**
 * File Name	:	PartnerInfoPanel.java
 * Created By	:	Sanjay Khattar (sanjay.khattar@ericsson.com)
 * Date Created	:	Jul 17, 2012 5:49:06 PM
 * Purpose		:	
 *
 * (c) Ericsson Television Inc. 
 */
package com.ericsson.cms.sites.ui.client;

import java.util.ArrayList;
import java.util.List;
import java.util.Set;

import com.ericsson.cms.sites.ui.client.entities.UIContact;
import com.ericsson.cms.sites.ui.client.entities.UISite;
import com.ericsson.cms.sites.ui.client.entities.UISitePartner;
import com.ericsson.cms.sites.ui.client.i18n.SitePartnerConstants;
import com.ericsson.cms.sites.ui.client.i18n.SitePartnerMessages;
import com.google.gwt.core.client.GWT;
import com.google.gwt.user.client.rpc.AsyncCallback;
import com.google.gwt.user.client.ui.Composite;
import com.tandbergtv.neptune.widgettoolkit.client.application.NeptuneApplication;
import com.tandbergtv.neptune.widgettoolkit.client.remote.NeptuneAsyncCallback;
import com.tandbergtv.neptune.widgettoolkit.client.security.NeptuneSecurity;
import com.tandbergtv.neptune.widgettoolkit.client.widget.basic.LabelWidget;
import com.tandbergtv.neptune.widgettoolkit.client.widget.basic.TextBoxWidget;
import com.tandbergtv.neptune.widgettoolkit.client.widget.composite.FormContainer;
import com.tandbergtv.neptune.widgettoolkit.client.widget.composite.MultiTaskBusyIndicator;
import com.tandbergtv.neptune.widgettoolkit.client.widget.composite.messagearea.MessageArea;
import com.tandbergtv.neptune.widgettoolkit.client.widget.container.HorizontalContainer;
import com.tandbergtv.neptune.widgettoolkit.client.widget.container.VerticalContainer;

/**
 * @author esakhat (sanjay.khattar@ericsson.com)
 *
 */
public class PartnerInfoPanel extends Composite {

	private static final String STYLE_TEXTBOX = "sites-partnerTab-textbox";
	
	private VerticalContainer mainPanel;
	private FormContainer formContainer;
	private LabelWidget contextIdValueWidget = new LabelWidget();
	private TextBoxWidget providerIdValueWidget;
	private TextBoxWidget lookupKeyValueWidget;
	private TextBoxWidget firstNameValueWidget;
	private TextBoxWidget lastNameValueWidget;
	private TextBoxWidget emailValueWidget;
	private PropertiesForm propertiesForm = new PropertiesForm();

	private MultiTaskBusyIndicator busyIndicator = new MultiTaskBusyIndicator();
	
	private IUISitePartnerServiceAsync service = GWT.create(IUISitePartnerService.class);
	private SitePartnerConstants constants = GWT.create(SitePartnerConstants.class);
	private SitePartnerMessages messages = GWT.create(SitePartnerMessages.class);

	private UISitePartner partner;
	
	private NeptuneSecurity security;

	private MessageArea messageArea;

	public PartnerInfoPanel() {
		security = NeptuneApplication.getApplication().getSecurity();
		
		buildUI();
		initWidget(mainPanel);
	}
	
	public void showSite(UISite site) {
		if (site != null) {
			if (site.getId() != null) {
				busyIndicator.center();
				service.getSitePartner(site.getId(),
						new NeptuneAsyncCallback<UISitePartner>() {
							@Override
							public void onNeptuneFailure(Throwable caught) {
								showError("Unable to get partner information for this site. | Reason: "
										+ caught != null ? caught
										.getLocalizedMessage() : null);
								formContainer.setVisible(false);
								busyIndicator.hide();
							}

							@Override
							public void onNeptuneSuccess(UISitePartner result) {
								PartnerInfoPanel.this.partner = result;
								updateForm(result);
								formContainer.setVisible(true);
								busyIndicator.hide();
							}
						});
			}
			else {
				UISitePartner sitePartner = new UISitePartner();
				PartnerInfoPanel.this.partner = sitePartner;
				updateForm(sitePartner);
				formContainer.setVisible(true);
				busyIndicator.hide();
			}
		}
		
	}
	
	public UISitePartner getPartnerFromForm() {
		updatePartnerFromForm();

		return partner;
	}
	
	public void setMessageArea(MessageArea messageArea) {
		this.messageArea = messageArea;
	}

	public void validate(final AsyncCallback<List<String>> callback) {
		final List<String> errors = new ArrayList<String>();
		
		String providerId = providerIdValueWidget.getText();
		if(providerId != null) {
			providerId = providerId.trim();
			if(providerId.length() > 25)
				errors.add(messages.errorLongProviderId(25));
			if(providerId.indexOf('/') != -1)
				errors.add(messages.errorProviderIdRestrictedChar('/'));
		}
		
		
		String lookupKey = lookupKeyValueWidget.getText();
		if(lookupKey != null) {
			lookupKey = lookupKey.trim();
			if(lookupKey.length() > 25)
				errors.add(messages.errorLongLookupKey(25));
		}
			
		String firstName = firstNameValueWidget.getText();
		if(firstName != null) {
			firstName = firstName.trim();
			if(!firstName.isEmpty()) {
				if(firstName.length() > 25)
					errors.add(messages.errorLongFirstName(25));
				if(firstName.matches(".*\\d.*"))
					errors.add(constants.errorFirstNameNumericChars());
			}
		}
			
		String lastName = lastNameValueWidget.getText();
		if(lastName != null) {
			lastName = lastName.trim();
			if(!lastName.isEmpty()) {
				if(lastName.length() > 25)
					errors.add(messages.errorLongLastName(25));
				if(lastName.matches(".*\\d.*"))
					errors.add(constants.errorLastNameNumericChars());
			}
		}
			
		String email = emailValueWidget.getText();
		if(email != null) {
			email = email.trim();
			if(!email.isEmpty()) {
				if (email.length() > 50)
					errors.add(messages.errorLongEmail(50));
				//TODO do it better
				if(email.indexOf('@') == -1 || email.indexOf('.') == -1)
					errors.add(constants.errorInvalidEmail());
			}
		}
		
		errors.addAll(propertiesForm.validate());

		updatePartnerFromForm();
		busyIndicator.center();
		service.validate(partner, new AsyncCallback<Set<String>>() {
			@Override
			public void onFailure(Throwable caught) {
				callback.onFailure(caught);
				busyIndicator.hide();
			}

			@Override
			public void onSuccess(Set<String> serverErrors) {
				if(serverErrors != null) {
					for(String se : serverErrors) {
						errors.add(constants.getString(se));
					}
				}
				callback.onSuccess(errors);
				busyIndicator.hide();
			}
		});
	}

	private void updatePartnerFromForm() {
		partner.setProviderId(trim(providerIdValueWidget.getText()));
		partner.setLookupKey(trim(lookupKeyValueWidget.getText()));
		partner.setContact(new UIContact());
		partner.getContact().setFirstName(trim(firstNameValueWidget.getText()));
		partner.getContact().setLastName(trim(lastNameValueWidget.getText()));
		partner.getContact().setEmail(trim(emailValueWidget.getText()));
		partner.setProperties(propertiesForm.getProperties());
	}
	
	private void buildUI() {
		mainPanel = new VerticalContainer();
		
		providerIdValueWidget = new TextBoxWidget();
		providerIdValueWidget.addStyleName(STYLE_TEXTBOX);
		lookupKeyValueWidget = new TextBoxWidget();
		lookupKeyValueWidget.addStyleName(STYLE_TEXTBOX);
		firstNameValueWidget = new TextBoxWidget();
		firstNameValueWidget.addStyleName(STYLE_TEXTBOX);
		lastNameValueWidget = new TextBoxWidget();
		lastNameValueWidget.addStyleName(STYLE_TEXTBOX);
		emailValueWidget = new TextBoxWidget();
		emailValueWidget.addStyleName(STYLE_TEXTBOX);
		
		formContainer = new FormContainer(HorizontalContainer.ALIGN_LEFT);
		formContainer.addRow(constants.fieldLabelContextId(), contextIdValueWidget);
		formContainer.addRow(constants.fieldLabelProviderId(), providerIdValueWidget);
		formContainer.addRow(constants.fieldLabelLookupKey(), lookupKeyValueWidget);
		formContainer.addRow(constants.fieldLabelFirstName(), firstNameValueWidget);
		formContainer.addRow(constants.fieldLabelLastName(), lastNameValueWidget);
		formContainer.addRow(constants.fieldLabelEmail(), emailValueWidget);
		formContainer.addRow("Properties", propertiesForm);
		
		mainPanel.add(formContainer);
	}

	private void updateForm(UISitePartner partner) {
		if (partner != null) {
			long contextId = partner.getContextId();
			if (contextId > 0) {
				contextIdValueWidget.setText(Long.toString(contextId));
			}
			else {
				contextIdValueWidget.setText(null);
			}
			providerIdValueWidget.setText(partner.getProviderId());
			lookupKeyValueWidget.setText(partner.getLookupKey());
			UIContact contact = partner.getContact();
			if (contact != null) {
				firstNameValueWidget.setText(contact.getFirstName());
				lastNameValueWidget.setText(contact.getLastName());
				emailValueWidget.setText(contact.getEmail());
			}
			else {
				firstNameValueWidget.setText(null);
				lastNameValueWidget.setText(null);
				emailValueWidget.setText(null);
			}
			boolean isFormEditable = security
					.isUserInRole(Permissions.EDIT) || ((security
							.isUserInRole(Permissions.CREATE) && (contextId <= 0)));
			
			setUpdateFormEditable(isFormEditable);
			propertiesForm.setProperties(partner.getProperties(),
					isFormEditable);
		}
	}

	private void setUpdateFormEditable(boolean isFormEditable) {
		providerIdValueWidget.setEnabled(isFormEditable);
		lookupKeyValueWidget.setEnabled(isFormEditable);
		firstNameValueWidget.setEnabled(isFormEditable);
		lastNameValueWidget.setEnabled(isFormEditable);
		emailValueWidget.setEnabled(isFormEditable);		
	}

	private void showError(String message) {
		messageArea.setErrorMessage(message);
	}

	private void showInfo(String message) {
		messageArea.setInfoMessage(message);
	}

	private String trim(String s) {
		if(s == null)
			return s;
		else
			return s.trim();
	}

}
