package com.ericsson.cms.sites.ui.client;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;

import com.ericsson.cms.sites.ui.client.entities.UIProperty;
import com.ericsson.cms.sites.ui.client.i18n.SitePartnerConstants;
import com.ericsson.cms.sites.ui.client.i18n.SitePartnerMessages;
import com.ericsson.cms.sites.ui.client.images.ISiteClientBundle;
import com.google.gwt.core.client.GWT;
import com.google.gwt.event.dom.client.ClickEvent;
import com.google.gwt.event.dom.client.ClickHandler;
import com.google.gwt.event.dom.client.KeyUpEvent;
import com.google.gwt.event.dom.client.KeyUpHandler;
import com.google.gwt.event.shared.HandlerRegistration;
import com.google.gwt.user.client.ui.Composite;
import com.google.gwt.user.client.ui.Widget;
import com.tandbergtv.neptune.widgettoolkit.client.widget.basic.ImageWidget;
import com.tandbergtv.neptune.widgettoolkit.client.widget.basic.LabelWidget;
import com.tandbergtv.neptune.widgettoolkit.client.widget.basic.TextBoxWidget;
import com.tandbergtv.neptune.widgettoolkit.client.widget.container.FlexTableContainer;

/**
 * @author Raj Prakash
 *
 */
public class PropertiesForm extends Composite {
	private static final String STYLE_HEAD = "sites-partnerTab-properties-head";
	private static final String STYLE_BODY = "sites-partnerTab-properties-body";
	private static final String STYLE_TEXTBOX = "sites-partnerTab-properties-textbox";
	
	private FlexTableContainer table = new FlexTableContainer();
	private Map<ImageWidget, Set<HandlerRegistration>> handlersMap =
		new HashMap<ImageWidget, Set<HandlerRegistration>>();
	
	private SitePartnerConstants constants = GWT.create(SitePartnerConstants.class);
	private SitePartnerMessages messages = GWT.create(SitePartnerMessages.class);
	private ISiteClientBundle imagesBundle = GWT.create(ISiteClientBundle.class);

	public PropertiesForm() {
		initWidget(table);
	}
	
	public void setProperties(List<UIProperty> props, boolean editable) {
		table.removeAllRows();
		table.getRowFormatter().addStyleName(0, STYLE_HEAD);
		table.setWidget(0, 0, new LabelWidget("Name"));
		table.setWidget(0, 1, new LabelWidget("Value"));
		for(UIProperty p : props)
			addRow(p.getName(), p.getValue(), editable);
		if (editable) {
			addNewNonDeletableRow();
		}
	}
	
	private void addNewNonDeletableRow() {
		addRow(null, null, true);
	}
	
	private void addRow(String name, String value, boolean editable) {
		int rowIndex = table.getRowCount();
		table.getRowFormatter().addStyleName(rowIndex, STYLE_BODY);

		Set<HandlerRegistration> hrSet = new HashSet<HandlerRegistration>();
		
		TextBoxWidget nameWidget = new TextBoxWidget(name);
		nameWidget.addStyleName(STYLE_TEXTBOX);
		nameWidget.setEnabled(editable);
		HandlerRegistration hr;
		if (editable) {
			hr = nameWidget.addKeyUpHandler(new KeyUpHandler() {
				@Override
				public void onKeyUp(KeyUpEvent event) {
					makeSureOfOneBlankRow();
				}
			});
			hrSet.add(hr);
		}
		table.setWidget(rowIndex, 0, nameWidget);
		
		TextBoxWidget valueWidget = new TextBoxWidget(value);
		valueWidget.addStyleName(STYLE_TEXTBOX);
		valueWidget.setEnabled(editable);
		if (editable) {
			hr = valueWidget.addKeyUpHandler(new KeyUpHandler() {
				@Override
				public void onKeyUp(KeyUpEvent event) {
					makeSureOfOneBlankRow();
				}
			});
			hrSet.add(hr);
		}
		table.setWidget(rowIndex, 1, valueWidget);
		
		final ImageWidget deleteIcon = new ImageWidget();
		deleteIcon.setResource(imagesBundle.deleteIcon());
		hr = deleteIcon.addClickHandler(new ClickHandler(){
			@Override
			public void onClick(ClickEvent event) {
				int rowIndex = findRowIndex(deleteIcon);
				table.removeRow(rowIndex);
				Set<HandlerRegistration> hrSet = handlersMap.remove(deleteIcon);
				for(HandlerRegistration hr : hrSet) {
					hr.removeHandler();
				}
				makeSureOfOneBlankRow();
			}});
		hrSet.add(hr);
		deleteIcon.setVisible(editable);
		table.setWidget(rowIndex, 2, deleteIcon);
		
		handlersMap.put(deleteIcon, hrSet);
	}
	
	public Set<String> validate() {
		Set<String> names = new HashSet<String>();
		Set<String> errors = new HashSet<String>();
		for(int i=1; i<table.getRowCount(); ++i) {
			String name = getName(i);
			String value = getValue(i);
			if(isBlank(name) && isBlank(value))
				continue;
			
			if(isBlank(name))
				errors.add(constants.errorEmptyPropertyName());
			else if(name.length() > 100)
				errors.add(messages.errorLongPropertyName(name, 100));
				
			if(isBlank(value))
				errors.add(constants.errorEmptyPropertyValue());
			else if(value.length() > 250)
				errors.add(messages.errorLongPropertyValue(value, 250));
				
			if(names.contains(name))
				errors.add(constants.errorDuplicateProperty());
			else
				names.add(name);
		}
		return errors;
	}
	
	/**
	 * Name and values are trimmed.
	 * If both name and value for a property is blank, that property is ignored.
	 * Note: Validation will fail if name xor value is blank.
	 */
	public List<UIProperty> getProperties() {
		List<UIProperty> props = new ArrayList<UIProperty>();
		
		for(int i=1; i<table.getRowCount(); ++i) {
			String name = getName(i);
			String value = getValue(i);
			if(isBlank(name) && isBlank(value))
				continue;
			props.add(new UIProperty(name, value));
		}
		
		return props;
	}
		
	private int findRowIndex(Widget w) {
		for(int i=1; i<table.getRowCount(); ++i) {
			if(table.getWidget(i, 2).equals(w)) {
				return i;
			}
		}
		return -1;
	}
	
	private void makeSureOfOneBlankRow() {
		int numBlankRows = getNumBlankRows();
		if(numBlankRows == 0) {
			addNewNonDeletableRow();
		} else {
			for(int i=1; i<table.getRowCount(); ++i) {
				ImageWidget deleteIcon = (ImageWidget) table.getWidget(i, 2);
				if(isBlank(i)) {
					deleteIcon.setVisible(numBlankRows > 1);
				} else {
					deleteIcon.setVisible(true);
				}
			}
		}
	}
	
	private int getNumBlankRows() {
		int count = 0;
		for(int i=1; i<table.getRowCount(); ++i) {
			if(isBlank(i)) {
				count++;
			}
		}
		return count;
	}
	
	private boolean isBlank(int rowIndex) {
		return isBlank(getName(rowIndex)) && isBlank(getValue(rowIndex));
	}

	private String getName(int rowIndex) {
		TextBoxWidget nameWidget = (TextBoxWidget) table.getWidget(rowIndex, 0);
		String name = nameWidget.getText();
		if(name != null)
			name = name.trim(); 
		return name;
	}
	
	private String getValue(int rowIndex) {
		TextBoxWidget valueWidget = (TextBoxWidget) table.getWidget(rowIndex, 1);
		String value = valueWidget.getText();
		if(value != null)
			value = value.trim(); 
		return value;
	}
	
	private boolean isBlank(String s) {
		return s == null || s.isEmpty();
	}
}
