package com.ericsson.cms.sites.ui.client;

import java.util.ArrayList;
import java.util.List;
import java.util.Map;

import com.ericsson.cms.sites.ui.client.aggregatedview.AggregatedViewTab;
import com.ericsson.cms.sites.ui.client.entities.UISite;
import com.ericsson.cms.sites.ui.client.entities.UISiteType;
import com.ericsson.cms.sites.ui.client.i18n.SitesConstants;
import com.ericsson.cms.sites.ui.client.i18n.SitesMessages;
import com.ericsson.cms.sites.ui.client.msooverrides.MSOOverridesTab;
import com.ericsson.cms.sites.ui.client.tabs.schedules.SchedulesTab;
import com.google.gwt.core.client.GWT;
import com.google.gwt.event.logical.shared.SelectionEvent;
import com.google.gwt.event.logical.shared.SelectionHandler;
import com.google.gwt.user.client.Command;
import com.google.gwt.user.client.History;
import com.google.gwt.user.client.Window;
import com.google.gwt.user.client.rpc.AsyncCallback;
import com.google.gwt.user.client.ui.Composite;
import com.tandbergtv.neptune.widgettoolkit.client.application.NeptuneApplication;
import com.tandbergtv.neptune.widgettoolkit.client.remote.NeptuneAsyncCallback;
import com.tandbergtv.neptune.widgettoolkit.client.security.NeptuneSecurity;
import com.tandbergtv.neptune.widgettoolkit.client.widget.composite.BusyIndicator;
import com.tandbergtv.neptune.widgettoolkit.client.widget.composite.messagearea.MessageArea;
import com.tandbergtv.neptune.widgettoolkit.client.widget.container.TabContainer;
import com.tandbergtv.neptune.widgettoolkit.client.widget.container.VerticalContainer;

public class SitesTabContainer extends Composite {
	private static final String STYLE_TABCONTAINER = "sites-tabContainer";
	
	private static final String SCHEDULER_LICENSE_KEY = "SchedulerLicense";

	private static final String PMM_LICENSE_KEY = "PMMLicense";

	private VerticalContainer mainPanel;
	private MessageArea messageArea;;
	private TabContainer tabContainer;
	private List<ISiteTab> tabs = new ArrayList<ISiteTab>();
	private BasicTab basicTab;
	private RulesTab rulesTab;
	private JobsTab jobsTab;
	private TargetedTitlesListPanel targetedTitlesTab;	
	private AggregatedViewTab aggViewTab;
	private MSOOverridesTab msoOverridesTab;
	private SchedulesTab schedulesTab;
	
	private BusyIndicator busyIndicator = new BusyIndicator(); 
	
	private SiteCRUDNotifier crudNotifier;

	private SitesTree sitesTree;
	private NeptuneSecurity security;
	private Map<String, String> info;
	
	private IUISiteServiceAsync service = GWT.create(IUISiteService.class);
	private SitesConstants constants = GWT.create(SitesConstants.class);
	private SitesMessages messages = GWT.create(SitesMessages.class);
	
	public static interface SiteCRUDNotifier  {
		void created(UISite site, ISiteTab caller);
		void updated(UISite site, ISiteTab caller);
		void deleted(UISite site, ISiteTab caller);
	}
	
	public SitesTabContainer(SitesTree sitesTree, MessageArea messageArea, 
			Map<String, String> info) {
		this.sitesTree = sitesTree;
		this.info = info;
		this.messageArea = messageArea;
		
		security = NeptuneApplication.getApplication().getSecurity();
		crudNotifier = new SiteCRUDNotifier() {
			@Override
			public void created(UISite site, ISiteTab caller) {
				setSite(site, caller);
				if(site.getType() == UISiteType.TRACKING)
					History.newItem(Anchors.getAnchor(site.getId(), site.getParentId()), false);
				else
					History.newItem(Anchors.getAnchor(site.getId(), null), false);
				SitesTabContainer.this.sitesTree.reload(site.getId(), site.getParentId());
			}

			@Override
			public void deleted(UISite site, ISiteTab caller) {
				setSite(null, caller);
				History.newItem(Anchors.VIEW_ANCHOR, false);
				SitesTabContainer.this.sitesTree.reload(null, null);
			}

			@Override
			public void updated(UISite site, ISiteTab caller) {
				setSite(site, caller);
				SitesTabContainer.this.sitesTree.reload(site.getId(), site.getParentId());
			}
			
			/**
			 * Except for the given tab,
			 * Updates all tabs with the given site.
			 * Enables/disables tab as indicated by the tab.
			 */
			private void setSite(UISite site, ISiteTab ignoreTab) {
				for(int i=0; i<tabs.size(); ++i) {
					ISiteTab tab = tabs.get(i);
					if(tab == ignoreTab)
						continue;
					boolean enabled = tab.setSite(site, null);
					tabContainer.getTabBar().setTabEnabled(i, enabled);
				}
			}
		};
		mainPanel = new VerticalContainer();
		mainPanel.setWidth("100%");
		tabContainer = new TabContainer();
		mainPanel.add(tabContainer);
		tabContainer.addSelectionHandler(new SelectionHandler<Integer>() {
			@Override
			public void onSelection(SelectionEvent<Integer> event) {
				//TODO notify only the two tabs that got selected and unselected
				for(int i=0; i<tabs.size(); ++i) {
					tabs.get(i).setSelected(i == event.getSelectedItem());
				}
			}});
		tabContainer.addStyleName(STYLE_TABCONTAINER);
		addTabs();
		initWidget(mainPanel);

	}
	
	private void addTabs() {
		basicTab = new BasicTab(info);
		basicTab.setCRUDNotifier(crudNotifier);
		basicTab.setMessageArea(messageArea);
		basicTab.setParentTabContainer(this);
		basicTab.setTreeWidget(this.sitesTree);
		tabs.add(basicTab);
		tabContainer.add(basicTab, basicTab.getDisplayName());
		
		rulesTab = new RulesTab();
		rulesTab.setCRUDNotifier(crudNotifier);
		rulesTab.setMessageArea(messageArea);
		tabs.add(rulesTab);
		tabContainer.add(rulesTab, rulesTab.getDisplayName());
		
		boolean isSchedulerLicensed = Boolean.parseBoolean(info.get(SCHEDULER_LICENSE_KEY));
		if (isSchedulerLicensed) {
			//if the user has scheduler view permission
			if(security.isUserInRole(com.ericsson.cms.scheduling.ui.client.Permissions.VIEW)) {
				jobsTab = new JobsTab();
				jobsTab.setCRUDNotifier(crudNotifier);
				jobsTab.setMessageArea(messageArea);
				tabs.add(jobsTab);
				tabContainer.add(jobsTab, jobsTab.getDisplayName());
			}
		}
		
		// Add Targeted Titles Tab
		targetedTitlesTab = new TargetedTitlesListPanel(info);
		targetedTitlesTab.setCRUDNotifier(crudNotifier);
		targetedTitlesTab.setMessageArea(messageArea);
		tabs.add(targetedTitlesTab);
		tabContainer.add(targetedTitlesTab, targetedTitlesTab.getDisplayName());

		aggViewTab = new AggregatedViewTab(info);
		aggViewTab.setCRUDNotifier(crudNotifier);
		aggViewTab.setMessageArea(messageArea);
		tabs.add(aggViewTab);
		tabContainer.add(aggViewTab, aggViewTab.getDisplayName());		

		msoOverridesTab = new MSOOverridesTab(info);
		msoOverridesTab.setCRUDNotifier(crudNotifier);
		//!! add tab later for sprint12
	//	tabs.add(msoOverridesTab);
		msoOverridesTab.setMessageArea(messageArea);
	//	tabContainer.add(msoOverridesTab, msoOverridesTab.getDisplayName());
		
		boolean isPMMLicensed = Boolean.parseBoolean(info.get(PMM_LICENSE_KEY));
		if (isPMMLicensed && isSchedulerLicensed 
				&& security.isUserInRole(Permissions.SCHEDULES_VIEW)
				&& security.isUserInRole(Permissions.SCHEDULER_VIEW)) {
			schedulesTab = new SchedulesTab();
			schedulesTab.setCRUDNotifier(crudNotifier);
			schedulesTab.setMessageArea(messageArea);
			tabs.add(schedulesTab);
			tabContainer.add(schedulesTab, schedulesTab.getDisplayName());
		}
	}

	private int findIndexOfTab(String tabName) {
		for(int i=0; i<tabs.size(); ++i)
			if(tabs.get(i).getName().equals(tabName))
				return i;
		return -1;
	}
	
	public void anchorChanged(final String anchor) {
		if(isBlank(anchor)) {
			sitesTree.reload(null, null);
			reset();
			showSite(null, null, 0);
		} else if(anchor.startsWith("?")) {
			if(!security.isUserInRole(Permissions.VIEW) &&
					!security.isUserInRole(Permissions.EDIT)) {
				handleUnauthorizedAccess();
				return;
			}
			
			String paramsStr = anchor.substring(1);
			Map<String, String> params = Util.getParams(paramsStr);
			//find id
			String idStr = params.get(Anchors.ID);
			int id = -1;
			try {
				id = Integer.parseInt(idStr);
			} catch(Exception e) {
				Window.alert(constants.invalidSiteId());
				return;
			}
			//find parent id
			Integer parentId = null;
			if(params.containsKey(Anchors.PARENT_ID)) {
				String parentIdStr = params.get(Anchors.PARENT_ID);
				try {
					parentId = Integer.valueOf(parentIdStr);
				} catch(Exception e) {
					Window.alert(constants.invalidParentSiteId());
					return;
				}
			}
			//find tab to select
			String tabName = params.get(Anchors.TAB);
			int tabIndexToSelect = findIndexOfTab(tabName);
			if(tabIndexToSelect == -1)
				tabIndexToSelect = 0;
			showSite(id, parentId, params, tabIndexToSelect);
			sitesTree.select(id, parentId);
		} else if(anchor.equals(Anchors.CREATE_LOGICAL_SITE)) {
			if(!security.isUserInRole(Permissions.CREATE)) {
				handleUnauthorizedAccess();
				return;
			}
			internalShowCreateLogicalSite();
			sitesTree.select(null, null);
		} else if(anchor.equals(Anchors.CREATE_TOP_DIST_SITE)) {
			if(!security.isUserInRole(Permissions.CREATE)) {
				handleUnauthorizedAccess();
				return;
			}
			verifyCreateDistSiteLicense(new Command() {
				@Override
				public void execute() {
					internalShowCreateDistributionSite(null);
					sitesTree.select(null, null);
				}});
		} else if(anchor.startsWith(Anchors.CREATE_CHILD_DIST_SITE)) {
			if(!security.isUserInRole(Permissions.CREATE)) {
				handleUnauthorizedAccess();
				return;
			}
			verifyCreateDistSiteLicense(new Command() {
				@Override
				public void execute() {
					String parentSiteIdStr = anchor.substring(Anchors.CREATE_CHILD_DIST_SITE.length());
					int parentSiteId = -1;
					try {
						parentSiteId = Integer.parseInt(parentSiteIdStr);
					} catch(Exception e) {
						Window.alert(constants.invalidParentSiteId());
						return;
					}
					internalShowCreateDistributionSite(parentSiteId);
					sitesTree.select(parentSiteId, null);
				}});
		} else if(anchor.equals(Anchors.CREATE_TOP_MEDIAPATH_DIST_SITE)) {
			if(!security.isUserInRole(Permissions.CREATE)) {
				handleUnauthorizedAccess();
				return;
			}
			verifyCreateDistSiteLicense(new Command() {
				@Override
				public void execute() {
					internalShowCreateMediaPathDistributionSite(null);
					sitesTree.select(null, null);
				}});
		} else if(anchor.startsWith(Anchors.CREATE_CHILD_MEDIAPATH_DIST_SITE)) {
			if(!security.isUserInRole(Permissions.CREATE)) {
				handleUnauthorizedAccess();
				return;
			}
			verifyCreateDistSiteLicense(new Command() {
				@Override
				public void execute() {
					String parentSiteIdStr = anchor.substring(Anchors.CREATE_CHILD_MEDIAPATH_DIST_SITE.length());
					int parentSiteId = -1;
					try {
						parentSiteId = Integer.parseInt(parentSiteIdStr);
					} catch(Exception e) {
						Window.alert(constants.invalidParentSiteId());
						return;
					}
					internalShowCreateMediaPathDistributionSite(parentSiteId);
					sitesTree.select(parentSiteId, null);
				}});
		} else if(anchor.startsWith(Anchors.CREATE_TRACK_SITE)) {
			if(!security.isUserInRole(Permissions.CREATE)) {
				handleUnauthorizedAccess();
				return;
			}
			verifyCreateTrackSiteLicense(new Command() {
				@Override
				public void execute() {
					String parentSiteIdStr = anchor.substring(Anchors.CREATE_TRACK_SITE.length());
					int parentSiteId = -1; 
					try {
						parentSiteId = Integer.parseInt(parentSiteIdStr);
					} catch(Exception e) {
						Window.alert(constants.invalidParentSiteId());
						return;
					}
					internalShowCreateTrackingSite(parentSiteId);
					sitesTree.select(parentSiteId, null);
				}});
		} else if(anchor.startsWith(Anchors.ADD_EXISTING_TRACK_SITE)) {
			if(!security.isUserInRole(Permissions.CREATE)) {
				handleUnauthorizedAccess();
				return;
			}
			String parentSiteIdStr = anchor.substring(Anchors.ADD_EXISTING_TRACK_SITE.length());
			int parentSiteId = -1; 
			try {
				parentSiteId = Integer.parseInt(parentSiteIdStr);
			} catch(Exception e) {
				Window.alert(constants.invalidParentSiteId());
				return;
			}
			internalShowAddExistingTrackingSite(parentSiteId);
			sitesTree.select(parentSiteId, null);
		} else {
			Window.alert(constants.invalidURL());
		}
	}
	
	public void showSite(int siteId, Integer parentSiteId) {
		History.newItem(Anchors.getAnchor(siteId, parentSiteId), false);
		showSite(siteId, parentSiteId, null, 0);
	}
	
	public void showCreateLogicalSite() {
		History.newItem(Anchors.VIEW_ANCHOR + "." + Anchors.CREATE_LOGICAL_SITE, false);
		internalShowCreateLogicalSite();
	}
	
	public void showCreateDistributionSite(final Integer parentSiteId) {
		verifyCreateDistSiteLicense(new Command() {
			@Override
			public void execute() {
				if(parentSiteId == null)
					History.newItem(Anchors.VIEW_ANCHOR + "." + Anchors.CREATE_TOP_DIST_SITE, false);
				else
					History.newItem(Anchors.VIEW_ANCHOR + "." + Anchors.CREATE_CHILD_DIST_SITE + parentSiteId, false);
				internalShowCreateDistributionSite(parentSiteId);
			}});
	}
	
	public void showCreateMediaPathDistributionSite(final Integer parentSiteId) {
		verifyCreateDistSiteLicense(new Command() {
			@Override
			public void execute() {
				if(parentSiteId == null)
					History.newItem(Anchors.VIEW_ANCHOR + "." + Anchors.CREATE_TOP_MEDIAPATH_DIST_SITE, false);
				else
					History.newItem(Anchors.VIEW_ANCHOR + "." + Anchors.CREATE_CHILD_MEDIAPATH_DIST_SITE + parentSiteId, false);
				internalShowCreateMediaPathDistributionSite(parentSiteId);
			}});
	}
	
	private void verifyCreateDistSiteLicense(final Command command) {
		busyIndicator.center();
		service.hasReachedMaxLicensedDistributionSites(new NeptuneAsyncCallback<Boolean>() {
			@Override
			public void onNeptuneFailure(Throwable caught) {
				busyIndicator.hide();
				Window.alert(constants.unableToVerifyLicense() + getFailureMessageSuffix(caught));
			}

			@Override
			public void onNeptuneSuccess(Boolean result) {
				if(result) {
					busyIndicator.hide();
					Window.alert(constants.reachedLicensedMaxDistSites());
					return;
				}
				command.execute();
				busyIndicator.hide();
			}
		});
	}
	
	public void showCreateTrackingSite(final int parentSiteId) {
		verifyCreateTrackSiteLicense(new Command() {
			@Override
			public void execute() {
				History.newItem(Anchors.VIEW_ANCHOR + "." + Anchors.CREATE_TRACK_SITE + parentSiteId, false);
				internalShowCreateTrackingSite(parentSiteId);
			}});
	}
	
	private void verifyCreateTrackSiteLicense(final Command command) {
		busyIndicator.center();
		service.hasReachedMaxLicensedTrackingSites(new NeptuneAsyncCallback<Boolean>() {
			@Override
			public void onNeptuneFailure(Throwable caught) {
				busyIndicator.hide();
				Window.alert(constants.unableToVerifyLicense() + getFailureMessageSuffix(caught));
			}

			@Override
			public void onNeptuneSuccess(Boolean result) {
				if(result) {
					busyIndicator.hide();
					Window.alert(constants.reachedLicensedMaxTrackingSites());
					return;
				}
				command.execute();
				busyIndicator.hide();
			}
		});
	}
	
	public void showAddExistingTrackingSite(int parentSiteId) {
		History.newItem(Anchors.VIEW_ANCHOR + "." + Anchors.ADD_EXISTING_TRACK_SITE + parentSiteId, false);
		internalShowAddExistingTrackingSite(parentSiteId);
	}
	
	private void reset() {
		messageArea.reset();
	}
	
	private void internalShowCreateLogicalSite() {
		reset();
		disableAllTabsOtherThanBasic();
		basicTab.showCreateLogicalSite();
		tabContainer.selectTab(0);
	}
	
	private void internalShowCreateDistributionSite(Integer parentSiteId) {
		reset();
		disableAllTabsOtherThanBasic();
		basicTab.showCreateDistributionSite(parentSiteId);
		tabContainer.selectTab(0);
	}
	
	private void internalShowCreateMediaPathDistributionSite(Integer parentSiteId) {
		reset();
		disableAllTabsOtherThanBasic();
		basicTab.showCreateMediaPathDistributionSite(parentSiteId);
		tabContainer.selectTab(0);
	}
	
	private void internalShowCreateTrackingSite(int parentSiteId) {
		reset();
		disableAllTabsOtherThanBasic();
		basicTab.showCreateTrackingSite(parentSiteId);
		tabContainer.selectTab(0);
	}
	
	private void internalShowAddExistingTrackingSite(final int parentSiteId) {
		reset();
		disableAllTabsOtherThanBasic();
		basicTab.showAddExistingTrackingSite(parentSiteId, new AsyncCallback<Void>() {
			@Override
			public void onFailure(Throwable caught) {
			}

			@Override
			public void onSuccess(Void result) {
				showSite(parentSiteId, null);
				SitesTabContainer.this.sitesTree.reload(parentSiteId, null);
			}
			
		});
		tabContainer.selectTab(0);
	}
	
	/**
	 * Loads the site.
	 * Calls {@link #showSite(UISite, params, int)}.
	 */
	private void showSite(final int siteId, final Integer parentSiteId, final Map<String, String> params, final int tabToSelect) {
		reset();
		busyIndicator.center();
		service.get(siteId, parentSiteId, new NeptuneAsyncCallback<UISite>() {
			@Override
			public void onNeptuneFailure(Throwable caught) {
				messageArea.setErrorMessage(messages.loadSiteFailed(siteId) + getFailureMessageSuffix(caught));
				showSite(null, null, 0);
				sitesTree.select(null, null);
				busyIndicator.hide();
			}

			@Override
			public void onNeptuneSuccess(UISite site) {
				showSite(site, params, tabToSelect);
				busyIndicator.hide();
			}});
	}

	/**
	 * Resets the error messages.
	 * Updates all tabs with the given site.
	 * Enables/disables tab as indicated by the tab.
	 */
	private void showSite(UISite site, Map<String, String> params, int tabToSelect) {
		for(int i=0; i<tabs.size(); ++i) {
			boolean enabled = tabs.get(i).setSite(site, params);
			tabContainer.getTabBar().setTabEnabled(i, enabled);
		}
		tabContainer.selectTab(tabToSelect);
	}
	
	private void disableAllTabsOtherThanBasic() {
		for(int i=1; i<tabs.size(); ++i) {
			tabs.get(i).setSite(null, null);
			tabContainer.getTabBar().setTabEnabled(i, false);
		}
	}

	private boolean isBlank(String s) {
		return (s == null || s.isEmpty());
	}

	private String getFailureMessageSuffix(Throwable caught) {
		return " | " + constants.reason() + ": " + (caught!= null ? caught.getLocalizedMessage() : "");
	}
	
	private void handleUnauthorizedAccess() {
		Window.alert(constants.unAuthorizedAction());
	}
	
	// ==============================================================
	// ===================== WIDGET OVERRIDES
	// ==============================================================

	@Override
	protected void onLoad() {
		super.onLoad();
	};

	@Override
	protected void onUnload() {
		if (busyIndicator != null)
			busyIndicator.hide();

		super.onUnload();
	}


}
