package com.ericsson.cms.sites.ui.client;

import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Set;

import com.ericsson.cms.sites.ui.client.bundle.SiteClientBundle;
import com.ericsson.cms.sites.ui.client.entities.UISiteTreeNode;
import com.ericsson.cms.sites.ui.client.entities.UISiteType;
import com.ericsson.cms.sites.ui.client.i18n.SitesConstants;
import com.google.gwt.core.client.GWT;
import com.google.gwt.event.dom.client.ContextMenuEvent;
import com.google.gwt.event.dom.client.ContextMenuHandler;
import com.google.gwt.event.logical.shared.CloseEvent;
import com.google.gwt.event.logical.shared.CloseHandler;
import com.google.gwt.event.logical.shared.OpenEvent;
import com.google.gwt.event.logical.shared.OpenHandler;
import com.google.gwt.event.logical.shared.SelectionEvent;
import com.google.gwt.event.logical.shared.SelectionHandler;
import com.google.gwt.user.client.Command;
import com.google.gwt.user.client.Window;
import com.google.gwt.user.client.ui.TreeItem;
import com.google.gwt.user.client.ui.Widget;
import com.tandbergtv.neptune.widgettoolkit.client.remote.NeptuneAsyncCallback;
import com.tandbergtv.neptune.widgettoolkit.client.security.NeptuneSecurity;
import com.tandbergtv.neptune.widgettoolkit.client.widget.basic.ImageWidget;
import com.tandbergtv.neptune.widgettoolkit.client.widget.basic.LabelWidget;
import com.tandbergtv.neptune.widgettoolkit.client.widget.basic.MenuBarWidget;
import com.tandbergtv.neptune.widgettoolkit.client.widget.basic.TreeWidget;
import com.tandbergtv.neptune.widgettoolkit.client.widget.composite.BusyIndicator;
import com.tandbergtv.neptune.widgettoolkit.client.widget.container.HorizontalContainer;
import com.tandbergtv.neptune.widgettoolkit.client.widget.container.PopupContainer;

public class SitesTree extends TreeWidget  {
	private NeptuneSecurity security;
	private boolean loaded;
	private TreeItem sitesRoot;
	private PopupContainer contextMenuContainer = new PopupContainer(true);
	private Set<UISiteTreeNode> openedSiteNodes = new HashSet<UISiteTreeNode>();
	private SitesTabContainer detailView; 
	private SitesTreeButtonsPanel buttonsPanel;

	private static final SiteClientBundle bundle = new SiteClientBundle();

	private BusyIndicator busyIndicator = new BusyIndicator();
	
	private IUISiteServiceAsync service = GWT.create(IUISiteService.class);
	private SitesConstants constants = GWT.create(SitesConstants.class);

	public SitesTree(NeptuneSecurity security) {
		this.security = security;
		addOpenHandler(new OpenHandler<TreeItem>() {
			@Override
			public void onOpen(OpenEvent<TreeItem> event) {
				UISiteTreeNode nodeData = (UISiteTreeNode) event.getTarget().getUserObject();
				//if sitesRoot, do nothing
				if(nodeData == null)
					return;
				openedSiteNodes.add(nodeData);
				TreeItem openedItem = event.getTarget();
				//If the node is opened, it should only show the error status the node itself, 
				//i.e isLocalError parameter set to true
				openedItem.setWidget(SitesTreeItem.getWidget(nodeData, true));
			}});
		
		addCloseHandler(new CloseHandler<TreeItem>() {
			@Override
			public void onClose(CloseEvent<TreeItem> event) {
				UISiteTreeNode nodeData = (UISiteTreeNode) event.getTarget().getUserObject();
				//if sitesRoot, do nothing
				if(nodeData == null)
					return;
				openedSiteNodes.remove(nodeData);
				TreeItem closedItem = event.getTarget();
				//If the node is collapsed it should also reflect the error status of any of its descendant nodes
				//i.e isLocalError parameter set to false
				closedItem.setWidget(SitesTreeItem.getWidget(nodeData, false));
			}});
		addSelectionHandler(new SelectionHandler<TreeItem>() {
			@Override
			public void onSelection(SelectionEvent<TreeItem> event) {
				int id;
				TreeItem selectedItem = event.getSelectedItem();
				UISiteTreeNode nodeData = (UISiteTreeNode) selectedItem.getUserObject();
				//if sitesRoot, do nothing
				if(nodeData == null)
					return;
				id = nodeData.getId();
				
				Integer parentId = null;
				if(nodeData.getType() == UISiteType.TRACKING) {
					TreeItem parentItem = selectedItem.getParentItem();
					UISiteTreeNode parentNodeData = (UISiteTreeNode) parentItem.getUserObject();
					parentId = parentNodeData.getId();
				}
				
				detailView.showSite(id, parentId);
				buttonsPanel.setSiteInfo(nodeData);

			}});
	}

	public void setDetailView(SitesTabContainer detailView) {
		this.detailView = detailView;
	}

	/**
	 * @param buttonsPanel the buttonsPanel to set
	 */
	public void setButtonsPanel(SitesTreeButtonsPanel buttonsPanel) {
		this.buttonsPanel = buttonsPanel;
	}

	public void reload(final Integer siteIdToSelect, final Integer parentSiteIdToSelect) {
		busyIndicator.center();
		service.getSiteTree(new NeptuneAsyncCallback<List<UISiteTreeNode>>() {
			@Override
			public void onNeptuneFailure(Throwable caught) {
				busyIndicator.hide();
				Window.alert(constants.loadingSitesFailed() + " | " + constants.reason() + ": " + (caught != null ? caught.getLocalizedMessage() : ""));
			}

			@Override
			public void onNeptuneSuccess(List<UISiteTreeNode> result) {
				loaded = true;
				renderTree(result);
				buttonsPanel.reset();
				if(siteIdToSelect != null)
					selectTreeItem(siteIdToSelect, parentSiteIdToSelect);
				busyIndicator.hide();
			}});
	}
	
	public void select(Integer siteIdToSelect, Integer parentSiteIdToSelect) {
		if(!loaded) {
			reload(siteIdToSelect, parentSiteIdToSelect);
		} else {
			if(siteIdToSelect == null)
				clearSelection();
			else
				selectTreeItem(siteIdToSelect, parentSiteIdToSelect);
		}
	}
	
	private void renderTree(List<UISiteTreeNode> treeData) {
		removeItems();
		propagateError(treeData);		
		addToTreeItem(null, treeData);

		//maintain open state
		Iterator<UISiteTreeNode> openedSiteNodeIter = openedSiteNodes.iterator();
		while(openedSiteNodeIter.hasNext()) {
			UISiteTreeNode openedSiteNode = openedSiteNodeIter.next();
			
			//find the tree item in the tree that matches the opened node
			boolean foundAndHasChildren = false;
			Iterator<TreeItem> treeItemIterator = treeItemIterator(); 
			while(treeItemIterator.hasNext()) {
				TreeItem treeItem = treeItemIterator.next();
					if(openedSiteNode.equals(treeItem.getUserObject())) {
					if(treeItem.getChildCount() > 0) {
						//set the tree item to be open
						treeItem.setState(true, false);
						foundAndHasChildren = true;
					}
					break;
				}
			}
			if(!foundAndHasChildren) {
				/*
				 * the site node is not there anymore or
				 * it does not have children anymore, so, remove from opened set
				 */
				openedSiteNodeIter.remove();
			}
		}
	}
	
	//adds children recursively
	private void addToTreeItem(TreeItem parentTreeItem, List<UISiteTreeNode> treeData) {
		for(UISiteTreeNode nodeData : treeData) {
			SitesTreeItem treeItem = new SitesTreeItem(nodeData);
			if (parentTreeItem != null) {
				parentTreeItem.addItem(treeItem);
			}
			else {
				addItem(treeItem);
			}
			if(nodeData.hasChildren()) {
				addToTreeItem(treeItem, nodeData.getChildren());
			}
		}
	}
    
	//Propagate error recursively from the descendant nodes
	private boolean propagateError(List<UISiteTreeNode> treeData) {
		boolean isError = false; //flag to indicate if current node or any of the descendant node has error 
		for(UISiteTreeNode nodeData : treeData) {
			if(nodeData.hasChildren()) {
				nodeData.setDescendantHasError( propagateError(nodeData.getChildren()));
			}
			else // for leaf node, i.e Tracking Site set descendantHasError property equal to error  
				nodeData.setDescendantHasError(nodeData.isError());  				
			if (nodeData.isError() == true ||  nodeData.isDescendantHasError()==true) 
				isError = true;
		}
		return isError;
	}
	
	private void setupContextMenu() {
		addDomHandler(new ContextMenuHandler() {
			@Override
			public void onContextMenu(ContextMenuEvent event) {
				showContextMenu();
				// Prevent the browser context menu from showing 
				event.preventDefault();
				event.stopPropagation();
			}
		}, ContextMenuEvent.getType());
	}

	private void showContextMenu() {
		TreeItem selectedItem = getSelectedItem();
		if(selectedItem == null)
			return;

		MenuBarWidget menu = new MenuBarWidget(true);
		contextMenuContainer.setWidget(menu);
		
		boolean hasMenuItems = false;
		//build the menu
		//if root
		if(selectedItem == getItem(0)) {
			if(security.isUserInRole(Permissions.CREATE)) {
				hasMenuItems = true;
				//MENU: Create Logical Site
				menu.addItem(constants.menuCreateTopLevelLogicalSite(), new Command() {
					@Override
					public void execute() {
						hideContextMenu();
						detailView.showCreateLogicalSite();
					}});
				//MENU: Create Distribution Site 
				menu.addItem(constants.menuCreateTopLevelDistributionSite(), new Command() {
					@Override
					public void execute() {
						hideContextMenu();
						detailView.showCreateDistributionSite(null);
					}});
				//MENU: Create MediaPath Distribution Site 
				menu.addItem(constants.menuCreateTopLevelMediapathDistributionSite(), new Command() {
					@Override
					public void execute() {
						hideContextMenu();
						detailView.showCreateMediaPathDistributionSite(null);
					}});
			}
		} else {	//if not a root
			final UISiteTreeNode siteNode = (UISiteTreeNode) selectedItem.getUserObject();
			//based on type
			switch(siteNode.getType()) {
			case LOGICAL:
				if(security.isUserInRole(Permissions.CREATE)) {
					hasMenuItems = true;
					//MENU: Create Child Distribution Site
					menu.addItem(constants.menuCreateChildDistributionSite(), new Command() {
						@Override
						public void execute() {
							hideContextMenu();
							detailView.showCreateDistributionSite(siteNode.getId());
						}});
					menu.addItem(constants.menuCreateChildMediapathDistributionSite(), new Command() {
						@Override
						public void execute() {
							hideContextMenu();
							detailView.showCreateMediaPathDistributionSite(siteNode.getId());
						}});
				}
				break;
			case DISTRIBUTION: 
			case MEDIAPATH_DISTRIBUTION:
				if(security.isUserInRole(Permissions.VIEW) && security.isUserInRole(Permissions.CREATE)) {
					hasMenuItems = true;
					//MENU: Create Tracking Site
					menu.addItem(constants.menuCreateChildTrackingSite(), new Command() {
						@Override
						public void execute() {
							hideContextMenu();
							detailView.showCreateTrackingSite(siteNode.getId());
						}});
					//MENU: Add Existing Tracking Site
					menu.addItem(constants.menuAddExistingTrackingSite(), new Command() {
						@Override
						public void execute() {
							hideContextMenu();
							detailView.showAddExistingTrackingSite(siteNode.getId());
						}});
				}
				break;
			case TRACKING:
				if(siteNode.isShared() && security.isUserInRole(Permissions.EDIT)) {
					hasMenuItems = true;
					//MENU: Dissociate From Parent
					menu.addItem(constants.menuDissociateFromParent(), new Command() {
						@Override
						public void execute() {
							hideContextMenu();
							dissociateFromParent(siteNode.getId(), siteNode.getParent().getId());
						}});
				}
				break;
			}
		}
		
		//display
		if(hasMenuItems) {
			Widget relativeWidget = (selectedItem != null) ? selectedItem.getWidget() : getItem(0).getWidget();
			contextMenuContainer.setPopupPosition(relativeWidget.getAbsoluteLeft(),
					relativeWidget.getAbsoluteTop() + relativeWidget.getOffsetHeight());
			contextMenuContainer.show();
		}

	}

	private void hideContextMenu() {
		contextMenuContainer.hide();
	}
	
	private void dissociateFromParent(int trackingSiteId, final int parentDistSiteId) {
		busyIndicator.center();
		service.dissociateTrackingSite(trackingSiteId, parentDistSiteId, new NeptuneAsyncCallback<Void>() {
			@Override
			public void onNeptuneFailure(Throwable caught) {
				busyIndicator.hide();
				Window.alert(constants.dissociatingTrackingSiteFailed() + " | " + constants.reason() + ": " + (caught != null ? caught.getLocalizedMessage() : ""));
			}

			@Override
			public void onNeptuneSuccess(Void result) {
				reload(parentDistSiteId, null);
				detailView.showSite(parentDistSiteId, null);
			}});
	}

	private void selectTreeItem(int id, Integer parentSiteId) {
		SitesTreeItem item = null;
		if (parentSiteId != null) {
			TreeItem parentItem = findFirstTreeItemById(parentSiteId, null);
			if (parentItem != null) {
				item = findFirstTreeItemById(id, parentItem);
			}
		}
		else {
			item = findFirstTreeItemById(id, null);
		}
		if(item != null) {
			// Select item
			item.getTree().setSelectedItem(item, false);
			// Make sure it is expanded and visible
			item.getTree().ensureSelectedItemVisible();
			// Update buttons panel at the bottom
			UISiteTreeNode selectedUiSiteTreeNode = (UISiteTreeNode) item.getUserObject();
			buttonsPanel.setSiteInfo(selectedUiSiteTreeNode);
		}
	}
	
	private SitesTreeItem findFirstTreeItemById(int id, TreeItem parentItem) {
		
		SitesTreeItem retItem = null;
		
		if (parentItem != null) {
			int numChildItems = parentItem.getChildCount();
			for (int i=0; i<numChildItems; i++) {
				TreeItem childItem = parentItem.getChild(i);
				if (childItem != null) {
					if (childItem instanceof SitesTreeItem) {
						UISiteTreeNode uiNode = (UISiteTreeNode) childItem.getUserObject();
						if (uiNode != null) {
							int candidateId = uiNode.getId();
							if (id == candidateId) {
								retItem = (SitesTreeItem) childItem;
							}
							else {
								retItem = findFirstTreeItemById(id, childItem);
							}
							if (retItem != null) {
								break;
							}
						}
					}
				}
			}
		}
		else  {
			int numRootItems = SitesTree.this.getItemCount();
			for (int i=0; i<numRootItems; i++) {
				TreeItem rootItem = SitesTree.this.getItem(i);
				if (rootItem != null) {
					if (rootItem instanceof SitesTreeItem) {
						UISiteTreeNode uiNode = (UISiteTreeNode) rootItem.getUserObject();
						if (uiNode != null) {
							int candidateId = uiNode.getId();
							if (id == candidateId) {
								retItem = (SitesTreeItem) rootItem;
							}
							else {
								retItem = findFirstTreeItemById(id, rootItem);
							}
							if (retItem != null) {
								break;
							}
						}
					}
				}
			}
		}
		
		return retItem;
	}

	private void clearSelection() {
		clearSelection(sitesRoot);
		buttonsPanel.reset();
	}
	
	private boolean clearSelection(TreeItem item) {
		if(item == null)
			return false;
		
		if(item.isSelected()) {
			item.setSelected(false);
			return true;
		}
		
		//recurse
		if(item.getChildCount() > 0) {
			for(int i = 0; i < item.getChildCount(); i++) {
				boolean cleared = clearSelection(item.getChild(i));
				if(cleared)
					return true;
			}
		}
		
		return false;
	}
	
	private static class SitesTreeItem extends TreeItem {
		
		private static final String STYLE_TREEITEM_SELECTED = "sites-treeItem-selected";
		private static final String STYLE_TREEITEM = "sites-treeItem";

		public SitesTreeItem(UISiteTreeNode nodeData) {
			super(getWidget(nodeData,false));
			setUserObject(nodeData);
//			setWidth("100%");
		}

		private static Widget getWidget(UISiteTreeNode nodeData, boolean isLocalError) {
			HorizontalContainer container = new HorizontalContainer();
			container.setSpacing(2);
//			container.addStyleName(STYLE_TREEITEM);
			ImageWidget icon;
			icon = getImage(nodeData, isLocalError);	
			container.add(icon);
			LabelWidget label = new LabelWidget(nodeData.getName());
			container.add(label);
//			container.setCellWidth(label, "100%");
			return container;
		}
		
		private static ImageWidget getImage(UISiteTreeNode siteNode, boolean isLocalError) {
			ImageWidget icon = new ImageWidget();
			bundle.updateSiteIcon(icon, siteNode, isLocalError);			
			return icon;
		}

		@Override
		public void setSelected(boolean selected) {
			super.setSelected(selected);
			if(selected) {
				getWidget().addStyleName(STYLE_TREEITEM_SELECTED);
			} else {
				getWidget().removeStyleName(STYLE_TREEITEM_SELECTED);
			}
		}
	}

}
