package com.ericsson.cms.sites.ui.client;

import java.util.ArrayList;
import java.util.List;
import java.util.Map;

import com.ericsson.cms.sites.ui.client.SitesTabContainer.SiteCRUDNotifier;
import com.ericsson.cms.sites.ui.client.entities.UISite;
import com.ericsson.cms.sites.ui.client.entities.UISiteType;
import com.ericsson.cms.sites.ui.client.i18n.SitesConstants;
import com.google.gwt.core.client.GWT;
import com.google.gwt.core.client.Scheduler;
import com.google.gwt.core.client.Scheduler.ScheduledCommand;
import com.google.gwt.event.dom.client.ChangeEvent;
import com.google.gwt.event.dom.client.ChangeHandler;
import com.google.gwt.event.dom.client.ClickEvent;
import com.google.gwt.event.dom.client.ClickHandler;
import com.google.gwt.event.dom.client.KeyDownEvent;
import com.google.gwt.event.dom.client.KeyDownHandler;
import com.google.gwt.event.logical.shared.ResizeEvent;
import com.google.gwt.event.logical.shared.ResizeHandler;
import com.google.gwt.event.shared.HandlerRegistration;
import com.google.gwt.user.client.History;
import com.google.gwt.user.client.Timer;
import com.google.gwt.user.client.Window;
import com.google.gwt.user.client.ui.Button;
import com.google.gwt.user.client.ui.Composite;
import com.google.gwt.user.client.ui.HorizontalPanel;
import com.google.gwt.user.client.ui.Label;
import com.google.gwt.user.client.ui.TextBox;
import com.google.gwt.user.client.ui.VerticalPanel;
import com.google.gwt.user.client.ui.Widget;
import com.tandbergtv.cms.portal.content.client.PreferenceKeys;
import com.tandbergtv.cms.portal.content.client.tab.list.goButton.ButtonClickListener;
import com.tandbergtv.cms.portal.ui.title.client.model.title.UIDistributionStatus;
import com.tandbergtv.cms.portal.ui.title.client.model.title.UITrackingStatus;
import com.tandbergtv.cms.portal.ui.title.client.model.uiservice.IActionInput;
import com.tandbergtv.cms.portal.ui.title.client.model.uiservice.IView;
import com.tandbergtv.neptune.widgettoolkit.client.application.NeptuneApplication;
import com.tandbergtv.neptune.widgettoolkit.client.remote.NeptuneAsyncCallback;
import com.tandbergtv.neptune.widgettoolkit.client.security.NeptuneSecurity;
import com.tandbergtv.neptune.widgettoolkit.client.widget.basic.ButtonWidget;
import com.tandbergtv.neptune.widgettoolkit.client.widget.basic.ListBoxWidget;
import com.tandbergtv.neptune.widgettoolkit.client.widget.composite.BusyIndicator;
import com.tandbergtv.neptune.widgettoolkit.client.widget.composite.messagearea.MessageArea;
import com.tandbergtv.neptune.widgettoolkit.client.widget.composite.table.Column;
import com.tandbergtv.neptune.widgettoolkit.client.widget.composite.table.Table;
import com.tandbergtv.neptune.widgettoolkit.client.widget.composite.table.feature.impl.CookieStoreBasedPageFeatureImpl;
import com.tandbergtv.neptune.widgettoolkit.client.widget.composite.table.feature.impl.SortFeatureImpl;
import com.tandbergtv.neptune.widgettoolkit.client.widget.container.HorizontalContainer;
import com.tandbergtv.neptune.widgettoolkit.client.widget.container.ScrollContainer;
import com.tandbergtv.neptune.widgettoolkit.client.widget.container.SimpleContainer;
import com.tandbergtv.neptune.widgettoolkit.client.widget.container.VerticalContainer;
import com.tandbergtv.neptune.widgettoolkit.client.widget.style.StyleNames;

/**
 * Panel which is shows the list of assets.
 * 
 * @author spuranik
 */
public class TargetedTitlesListPanel extends Composite implements IView, ISiteTab, ClickHandler {
	
	private static final int MAIN_PANEL_OFFSET_HEIGHT = 250;

	private static final String STYLE_NAME = "content-TitleListPanel";
	private static final int SPACING = 2;

	// widgets
	private ScrollContainer rootContainer;
	private VerticalContainer container;
	
	// input
	private NeptuneSecurity security;
	private String specificationName;

	private final BusyIndicator busyIndicator = new BusyIndicator();
	
	// message area
	private MessageArea messageArea;
	
	// widgets
	private boolean selectButtonVisible = false;
	private Table<Long, TitleRecord> assetTable;
	private TargetedTitlesTableDataProvider dp;
	private ButtonWidget goButton;
	private TextBox searchBox = new TextBox();
	private ListBoxWidget<String> statusFilter = new ListBoxWidget<String>();
	private ListBoxWidget<String> actionsListBox;
	private ListBoxWidget<String> parameterListBox;
	private ListBoxWidget<String> multiSelectParameterListBox;
	private HorizontalContainer actionListContainer;
		
	private static final String ACTION_DISTRIBUTE = "DISTRIBUTE";
	private static final String ACTION_DELETE = "DELETE";
	private static final String ACTION_RUN_TEMPLATE = "RUN_TEMPLATE";
	
	private String TARGETED_TITLES_LIST_PANEL_REFRESH_KEY = "site.targetedtitles.refreshTime";
	
	/**
	 * Default Targeted Titles List Panel Refresh Time (in seconds)
	 */
	private int DEFAULT_TARGETED_TITLES_LIST_PANEL_REFRESH_TIME = 60;
	
	ButtonClickListener actionButtonHandler = null;
	boolean useMultiSelectParameters=false;
	
	private SitesConstants constants = GWT.create(SitesConstants.class);

	private IUISiteServiceAsync siteService = GWT.create(IUISiteService.class);

	private Timer refreshTimer;

	/**
	 * View Refresh Period in milliseconds
	 */
	private int refreshPeriod;

	private UISite uiSite;

	private HandlerRegistration windowRegistration = null;

	public TargetedTitlesListPanel(Map<String, String> info) {
		/* Not a good way to do this, but not enough control available */
		showBusyIndicator();

		this.security = NeptuneApplication.getApplication().getSecurity();
		this.specificationName = info.get(PreferenceKeys.SPECIFICATION_NAME);
		this.refreshPeriod = getRefreshPeriod(info);

		this.initWidgets();
		this.initializeDataProvider();

		this.refreshTimer = new Timer() {
			@Override
			public void run() {
				refreshView();
			}
		};

		Scheduler.get().scheduleDeferred(new ScheduledCommand() {
			@Override
			public void execute() {
				updateMainPanelSize();
			}
		});

		hideBusyIndicator();
	}

	/**
	 * Refresh the view for this panel.
	 */
	private void refreshView() {
		if (uiSite != null) {
			refreshView(uiSite);
		}
	}
	
	private Widget getSearchWidget() {
		VerticalPanel container = new VerticalPanel();
		HorizontalPanel mainPanel = new HorizontalPanel();
		Label title = new Label(constants.filterListLabel());
		SimpleContainer simpleContainer = new SimpleContainer();
		simpleContainer.setWidget(title);
		simpleContainer.addStyleName("targeted-titles-filter-panel-header");
		container.add(simpleContainer);
		container.add(mainPanel);
		searchBox.setVisibleLength(40);
		searchBox.setMaxLength(255);
		Label text = new Label(constants.textLabel());
		mainPanel.add(text);
		mainPanel.add(searchBox);
		searchBox.addKeyDownHandler(new KeyDownHandler() {

			@Override
			public void onKeyDown(KeyDownEvent event) {
				if (event.getNativeKeyCode() == 13) {
					cleanup();
					assetTable.refresh();
				}
			}

		});
		Button search = new Button(constants.filterButton());
		search.addStyleDependentName(StyleNames.DO_BUTTON_STYLE);
		search.addClickHandler(new ClickHandler() {
			
			@Override
			public void onClick(ClickEvent event) {
				cleanup();
				assetTable.refresh();
			}
		});
		
		Button clear = new Button(constants.clearButton());
		clear.addStyleDependentName(StyleNames.DATALOSS_BUTTON_STYLE);
		clear.addClickHandler(new ClickHandler() {
			
			@Override
			public void onClick(ClickEvent event) {
				cleanup();
				searchBox.setText("");
				statusFilter.setSelectedIndex(0);
				assetTable.refresh();
			}
		});
		Label status = new Label(constants.statusLabel());
		status.addStyleName("targeted-titles-filter-panel-label");
		mainPanel.add(status);
		mainPanel.add(statusFilter);
		mainPanel.add(search);
		mainPanel.add(clear);
		for(int i = 0; i < mainPanel.getWidgetCount(); i++) {
			Widget w = mainPanel.getWidget(i);
			mainPanel.setCellVerticalAlignment(w, HorizontalPanel.ALIGN_MIDDLE);
		}
		mainPanel.addStyleName("target-titles-filter-panel");
		return container;
	}
	
	public void clearQuickSearchData(UISiteType uiSiteType) {
		searchBox.setText("");
		statusFilter.clear();
		statusFilter.addItem("", "");
		if (uiSiteType.equals(UISiteType.DISTRIBUTION)
				|| uiSiteType.equals(UISiteType.MEDIAPATH_DISTRIBUTION)) {
			for(UIDistributionStatus item : UIDistributionStatus.values())
				statusFilter.addItem(constants.getString(item.name().toLowerCase()), item.name());
		} else if (uiSiteType.equals(UISiteType.TRACKING)) {
			for(UITrackingStatus item : UITrackingStatus.values())
				statusFilter.addItem(constants.getString(item.name().toLowerCase()), item.name());
		}
	}
	
	public String getTextFilter() {
		return searchBox.getText().trim();
	}
	
	public String getStatusFilter() {
		if(statusFilter.getSelectedItem() == null)
			return "";
		else
			return statusFilter.getSelectedItem();
	}

	/**
	 * Gets refresh period from configuration.
	 * 
	 * @param info
	 * @return
	 */
	private int getRefreshPeriod(Map<String, String> info){
		String refreshPeriod = info.get(TARGETED_TITLES_LIST_PANEL_REFRESH_KEY);
		try{
			int refresh = Integer.parseInt(refreshPeriod.trim());
			if(refresh < DEFAULT_TARGETED_TITLES_LIST_PANEL_REFRESH_TIME) 
				refresh = DEFAULT_TARGETED_TITLES_LIST_PANEL_REFRESH_TIME; // validate if configured value is less than default 
			
			return refresh * 1000; // in millis
			
		}
		catch(Exception ex){
			return DEFAULT_TARGETED_TITLES_LIST_PANEL_REFRESH_TIME * 1000; // in millis
		}
	}

	private void initWidgets() {
		rootContainer = new ScrollContainer();
		rootContainer.setStylePrimaryName(STYLE_NAME);
		initWidget(rootContainer);

		// for adding space between the error/success panel and the table
		container = new VerticalContainer();
		container.add(getSearchWidget());
		rootContainer.setWidget(container);
		container.setSpacing(SPACING);
		
		// table is added separately

		// Action List Container
		actionListContainer = new HorizontalContainer();
		actionListContainer.setSpacing(SPACING);

		// actionlist box
		actionsListBox = new ListBoxWidget<String>(false);
		
		// listbox for additional params used by the service 
		parameterListBox = new ListBoxWidget<String>(false);
		parameterListBox.setVisible(false);
		multiSelectParameterListBox = new ListBoxWidget<String>(true);
		multiSelectParameterListBox.setVisible(false);
		
		// go button
		goButton = new ButtonWidget(constants.goButtonName());
		goButton.addClickHandler(this); 
		goButton.addStyleDependentName(StyleNames.DO_BUTTON_STYLE);

		//add action items to actions listbox
		actionListContainer.add(actionsListBox);
		actionListContainer.add(parameterListBox);
		actionListContainer.add(goButton);

	}

	/* Initialize the Data Provider */
	private void initializeDataProvider() {
		dp = new TargetedTitlesTableDataProvider(this);
		dp.init(new NeptuneAsyncCallback<Void>() {
			@Override
			public void onNeptuneFailure(Throwable caught) {
				hideBusyIndicator();
				Window.alert("Failed to initialize Asset Table Data Provider. Reason: " + ((caught != null) ? caught.getMessage() : ""));
			}

			@Override
			public void onNeptuneSuccess(Void result) {
//				show(infoMessage);
				initializeTable();
			}});
	}
	
	/* Initialize the Table */
	private void initializeTable() {
		//build asset table
		assetTable = new Table<Long, TitleRecord>(dp);
		CookieStoreBasedPageFeatureImpl pageFeature = new CookieStoreBasedPageFeatureImpl("titleList");
		dp.setPageFeature(pageFeature);
		SortFeatureImpl<Long, TitleRecord> sortFeature = new SortFeatureImpl<Long, TitleRecord>(
				dp.getDefaultSortColumn(), dp.getDefaultSortOrder());
		for(Column<?, TitleRecord> sortableCol  : dp.getSortableColumns())
			sortFeature.addSortableColumn(sortableCol);
		dp.setSortFeature(sortFeature);
		assetTable.addPageFeature(pageFeature);
		assetTable.addSortFeature(sortFeature);
		
		//add actions list widget to table's action container
		assetTable.registerWidgetOnActionContainer(actionListContainer);
		
		//add asset table to this container
		container.add(assetTable);
		
		//init the table asynchronously
		assetTable.initialize(new NeptuneAsyncCallback<Void>() {
			@Override
			public void onNeptuneFailure(Throwable caught) {
				hideBusyIndicator();
			}

			@Override
			public void onNeptuneSuccess(Void result) {
				assetTable.refresh();
			}});
		
	}

	/**
	 * @return the security
	 */
	public NeptuneSecurity getSecurity() {
		return security;
	}

	/**
	 * @return the specificationName
	 */
	public String getSpecificationName() {
		return specificationName;
	}

	/**
	 * @return the assetTable
	 */
	public Table<Long, TitleRecord> getAssetTable() {
		return assetTable;
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @seecom.tandbergtv.cms.portal.ui.title.client.model.uiservice.IView#
	 * getParentContainer()
	 */
	public Widget getParentContainer() {
		return this.getParent();
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see
	 * com.tandbergtv.cms.portal.ui.title.client.model.uiservice.IView#reportFailure
	 * (java.lang.Throwable)
	 */
	public void reportFailure(Throwable caught) {
		// show the error message along with the current state
		messageArea.setErrorMessage(caught.getLocalizedMessage());
	}
	
	public void reportFailure(String message) {
		// show the error message along with the current state
		messageArea.setErrorMessage(message);
	}

	/* (non-Javadoc)
	 * @see com.tandbergtv.cms.portal.ui.title.client.model.uiservice.IView#cleanup()
	 */
	public void cleanup() {
		messageArea.reset();
	}

	/* (non-Javadoc)
	 * @see com.tandbergtv.cms.portal.ui.title.client.model.uiservice.IView#refresh(java.lang.String)
	 */
	public void refresh(String infoMessage) {
		if (this.assetTable != null) {
			/* Return to the first page and show the updated list view of the table */
			this.assetTable.getPageFeature().setPageNumber(1);
			this.assetTable.showListView();
			this.assetTable.refresh();
		}
		rootContainer.setWidget(container);

		this.show(infoMessage);

		setWidgetForNoAction();
	}

	
	/* (non-Javadoc)
	 * @see com.tandbergtv.cms.portal.ui.title.client.model.uiservice.IView#show(java.lang.String)
	 */
	public void show(String infoMessage) {
		messageArea.setInfoMessage(infoMessage);
	}
	
	/**
	 * @return the selectButtonVisible
	 */
	public boolean isSelectButtonVisible() {
		return selectButtonVisible;
	}

	/**
	 * @param selectButtonVisible the selectButtonVisible to set
	 */
	public void setSelectButtonVisible(boolean selectButtonVisible) {
		this.selectButtonVisible = selectButtonVisible;
	}


	@Override
	public IActionInput getInputToExecution() {
		return null;
	}

	@Override
	public List<Long> getSeletedRecordIds() {
		List<TitleRecord> selectedIds = getAssetTable().getSelectedRecords();
		List<Long> ids = new ArrayList<Long>();
		for (TitleRecord tr : selectedIds) {
			ids.add(tr.getKey());
		}
		return ids;
	}

	@Override
	public void show(boolean readOnly) {
		this.goButton.setEnabled(readOnly);
		this.actionsListBox.setEnabled(readOnly);
		this.parameterListBox.setEnabled(readOnly);
		this.multiSelectParameterListBox.setEnabled(readOnly);
	}
	
	
	@Override
	public String getName() {
		return "Targeted Titles";
	}

	@Override
	public String getDisplayName() {
		return constants.targetedTitlesTabName();
	}

	@Override
	public void setCRUDNotifier(SiteCRUDNotifier crudNotifier) {
	}
	
	/* (non-Javadoc)
	 * @see com.ericsson.cms.sites.ui.client.ISiteTab#setSelected(boolean)
	 */
	@Override
	public void setSelected(boolean selected) {
		if (selected) {
			if(uiSite != null)
				History.newItem(Anchors.getAnchor(uiSite.getId(), uiSite.getParentId(), getName()), false);
			refreshTimer.scheduleRepeating(refreshPeriod);
			refreshView();
		}
		else {
			refreshTimer.cancel();
		}
	}

	/* (non-Javadoc)
	 * @see com.ericsson.cms.sites.ui.client.ISiteTab#setSite(com.ericsson.cms.sites.ui.client.entities.UISite)
	 */
	@Override
	public boolean setSite(UISite site, Map<String, String> params) {
		boolean retVal = false;
		
		if (site != null) {
			this.setUiSite(site);
			
			UISiteType uiSiteType = uiSite.getType();
			if ((UISiteType.DISTRIBUTION.equals(uiSiteType)) 
					|| (UISiteType.MEDIAPATH_DISTRIBUTION.equals(uiSiteType)) 
					|| (UISiteType.TRACKING.equals(uiSiteType))) {
				retVal = true;
				clearQuickSearchData(uiSiteType);
			}
			
		}
		
		return retVal;
	}

	public void refreshView(UISite uiSite) {

		String infoMessage = null;
		refresh(infoMessage);

		actionListContainer.setVisible(false);

		if (uiSite != null) {
			UISiteType uiSiteType = uiSite.getType();
			if (UISiteType.LOGICAL.equals(uiSiteType)) {
				showSuccessMessage("Target Titles View is not applicable for Logical Site : "
						+ uiSite.getName());
			}
			else {
				showActions(uiSite);
			}
		} 
		else {
			showErrorMessage("Could not get Site data.");
		}
		
	}

	/**
	 * 
	 * @param message
	 */
	private void showErrorMessage(String message) {
		messageArea.setErrorMessage(message);
	}
	
	private void showSuccessMessage(String message) {
		messageArea.setInfoMessage(message);
	}

	private void showActions(final UISite uiSite) {
		if (uiSite != null) {
			
			parameterListBox.setVisible(false);
			actionsListBox.setVisible(false);
			goButton.setVisible(false);
			
			actionsListBox.clear();
			actionsListBox.addItem("", "");
			
			UISiteType uiSiteType = uiSite.getType();
			if (UISiteType.DISTRIBUTION.equals(uiSiteType) 
					|| UISiteType.MEDIAPATH_DISTRIBUTION.equals(uiSiteType)) {
				boolean isActive = uiSite.isActive();
				if ((isActive) && (security.isUserInRole(Permissions.DISTRIBUTE))) {
					actionsListBox.addItem("Distribute", ACTION_DISTRIBUTE);
				}
				if (security.isUserInRole(Permissions.ASSETMANAGEMENT_DELETE)) {
					actionsListBox.addItem("Delete", ACTION_DELETE);
				}
				if (security.isUserInRole(Permissions.RUN_TEMPLATE)) {
					actionsListBox.addItem("Run Template", ACTION_RUN_TEMPLATE);
				}
				actionsListBox.setVisible(true);
			}
			else if (UISiteType.TRACKING.equals(uiSiteType)) {
				if (security.isUserInRole(Permissions.ASSETMANAGEMENT_DELETE)) {
					actionsListBox.addItem("Delete", "DELETE");
				}
				actionsListBox.setVisible(true);
			}
			
			actionsListBox.addChangeHandler(new ChangeHandler() {
				public void onChange(ChangeEvent event) {
					String actionSelected = actionsListBox.getSelectedItem();
					if(!actionSelected.trim().isEmpty()) {
						goButton.setEnabled(true);
					} else {
						goButton.setEnabled(false);
					}

					if ("RUN_TEMPLATE".equals(actionSelected)) {
						siteService.getTemplateNames(uiSite.getId(), new NeptuneAsyncCallback<List<String>>() {

							@Override
							public void onNeptuneFailure(Throwable caught) {
								showErrorMessage(caught.getMessage());
							}

							@Override
							public void onNeptuneSuccess(List<String> result) {
								parameterListBox.clear();
								for (String templateName : result) {
									parameterListBox.addItem(templateName, templateName);
								}
								
							}
							
						});
						parameterListBox.setVisible(true);
					}
					else {
						parameterListBox.setVisible(false);
					}
				}
			});
			
			goButton.setVisible(true);
			
			actionListContainer.setVisible(true);
			setWidgetForNoAction();
		}

	}
	
	@Override
	public void onClick(ClickEvent event) {
		List<Long> selectedIds = getSeletedRecordIds();
		
		String action = actionsListBox.getSelectedItem();
		
		String templateName = parameterListBox.getSelectedItem();
		
		if ((selectedIds != null) && (selectedIds.size() > 0)) {
			executeAction(selectedIds, action, templateName);
		}
		else {
			showErrorMessage("Need selecting at least 1 Title Id to run an action.");
		}
		
	}

	public void executeAction(List<Long> selectedIds, String action, String templateName) {
		if (ACTION_DISTRIBUTE.equals(action)) {
			siteService.distributeImmediately(uiSite.getId(), selectedIds, new NeptuneAsyncCallback<Void>() {

				@Override
				public void onNeptuneFailure(Throwable caught) {
					showErrorMessage(caught.getMessage());
				}

				@Override
				public void onNeptuneSuccess(Void result) {
					refresh("Scheduled distribution of selected titles.");
				}

				
			});
		}
		else if (ACTION_DELETE.equals(action)) {
			siteService.deleteLocalTitles(uiSite.getId(), selectedIds, new NeptuneAsyncCallback<Void>() {

				@Override
				public void onNeptuneFailure(Throwable caught) {
					showErrorMessage(caught.getMessage());
				}

				@Override
				public void onNeptuneSuccess(Void result) {
					refresh("Success deleting selected titles.");
				}

				
			});
		}
		else if (ACTION_RUN_TEMPLATE.equals(action)) {
			siteService.runTemplate(uiSite.getId(), selectedIds, templateName, new NeptuneAsyncCallback<Void>() {

				@Override
				public void onNeptuneFailure(Throwable caught) {
					showErrorMessage(caught.getMessage());
				}

				@Override
				public void onNeptuneSuccess(Void result) {
					refresh("Scheduled work orders for selected titles.");
				}

				
			});
		}

	}

	private void setWidgetForNoAction() {
		actionsListBox.setSelectedItem("");
		parameterListBox.clear();
		parameterListBox.setVisible(false);
		goButton.setEnabled(false);
	}
	
	public boolean hasActions() {
		return actionsListBox.getItemCount() > 0;
	}

    private void showBusyIndicator() {
    	busyIndicator.center();
    }

    private void hideBusyIndicator() {
    	busyIndicator.hide();
    }

	/*
	 * Ideally, use an event from data provider to notify view or use package protected code.
	 */
    public void handleGetRecordsRequest() {
    	showBusyIndicator();
    }
    
    /*
	 * Ideally, use an event from data provider to notify view. This is a 'shortcut'
	 */
    public void handleGetRecordsSuccessResponse(List<TitleRecord> records) {
		hideBusyIndicator();
	}

    /*
     * Ideally, use an event from data provider to notify view. This is a 'shortcut' 
     */
	public void handleGetRecordsFailureResponse(Throwable caught) {
		hideBusyIndicator();
	}

	
	/**
	 * @return the uiSite
	 */
	public UISite getUiSite() {
		return uiSite;
	}

	/**
	 * @param uiSite the uiSite to set
	 */
	public void setUiSite(UISite uiSite) {
		this.uiSite = uiSite;
	}

	/* (non-Javadoc)
	 * @see com.google.gwt.user.client.ui.Widget#onLoad()
	 */
	@Override
	protected void onLoad() {
		super.onLoad();

		windowRegistration = Window.addResizeHandler(new ResizeHandler() {
			public void onResize(ResizeEvent event) {
				updateMainPanelSize();
			}
		});
	}
	
	/* (non-Javadoc)
	 * @see com.google.gwt.user.client.ui.Widget#onUnload()
	 */
	@Override
	protected void onUnload() {
		windowRegistration.removeHandler();
		windowRegistration = null;

		super.onUnload();
		refreshTimer.cancel();
		busyIndicator.hide();
	}

	@Override
	public void setMessageArea(MessageArea messageArea) {
		this.messageArea = messageArea;
	}
	
	protected void updateMainPanelSize() {
		int offsetHeight = MAIN_PANEL_OFFSET_HEIGHT;
		
		rootContainer.setHeight(Window.getClientHeight() - offsetHeight + "px");
	}
	
}
