package com.ericsson.cms.sites.ui.client;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import com.ericsson.cms.sites.ui.client.entities.TitleMetadata;
import com.ericsson.cms.sites.ui.client.entities.UISite;
import com.ericsson.cms.sites.ui.client.entities.UISiteTitleStatusData;
import com.ericsson.cms.sites.ui.client.entities.UISiteTitleStatusRecord;
import com.ericsson.cms.sites.ui.client.entities.UISiteType;
import com.ericsson.cms.sites.ui.client.i18n.SitesConstants;
import com.google.gwt.core.client.GWT;
import com.google.gwt.event.dom.client.ClickEvent;
import com.google.gwt.event.dom.client.ClickHandler;
import com.google.gwt.user.client.History;
import com.google.gwt.user.client.rpc.AsyncCallback;
import com.google.gwt.user.client.ui.Widget;
import com.tandbergtv.cms.portal.content.client.Permissions;
import com.tandbergtv.cms.portal.content.client.model.UiTable;
import com.tandbergtv.cms.portal.content.client.rpc.SpecificationService;
import com.tandbergtv.cms.portal.content.client.rpc.SpecificationServiceAsync;
import com.tandbergtv.cms.portal.content.client.tab.list.assettable.TableColumnNameHelper;
import com.tandbergtv.cms.portal.content.client.title.view.thumbnail.TitleThumbnailWidget;
import com.tandbergtv.cms.portal.ui.title.client.model.search.UIFieldType;
import com.tandbergtv.cms.portal.ui.title.client.model.search.UiColumn;
import com.tandbergtv.cms.portal.ui.title.client.view.bundle.TitleClientBundle;
import com.tandbergtv.neptune.widgettoolkit.client.application.NeptuneApplication;
import com.tandbergtv.neptune.widgettoolkit.client.remote.NeptuneAsyncCallback;
import com.tandbergtv.neptune.widgettoolkit.client.security.NeptuneSecurity;
import com.tandbergtv.neptune.widgettoolkit.client.widget.basic.ImageWidget;
import com.tandbergtv.neptune.widgettoolkit.client.widget.composite.table.Column;
import com.tandbergtv.neptune.widgettoolkit.client.widget.composite.table.ColumnBase;
import com.tandbergtv.neptune.widgettoolkit.client.widget.composite.table.DataProvider;
import com.tandbergtv.neptune.widgettoolkit.client.widget.composite.table.SortOrder;
import com.tandbergtv.neptune.widgettoolkit.client.widget.composite.table.TableConstants;
import com.tandbergtv.neptune.widgettoolkit.client.widget.composite.table.feature.impl.AnchorTokenizer;
import com.tandbergtv.neptune.widgettoolkit.client.widget.composite.table.feature.impl.PageFeatureImpl;
import com.tandbergtv.neptune.widgettoolkit.client.widget.composite.table.feature.impl.SortFeatureImpl;
import com.tandbergtv.neptune.widgettoolkit.client.widget.composite.table.view.View;
import com.tandbergtv.neptune.widgettoolkit.client.widget.composite.table.view.label.LabelStringView;

/**
 * Data Provider which helps the asset list table population. It fetches the
 * column names, default sort column and order on initialization. Records are
 * fetched from the server when requested.
 * 
 * @author spuranik
 */
public class TargetedTitlesTableDataProvider implements DataProvider<Long, TitleRecord> {
	private static String TABLE_NAME = "targetedTitlesColumns";
	private NeptuneSecurity security;
	/* special columns */
	private static String SITE_STATUS = "status";
	private static String SITE_STATUS_UPDATE_TIME = "statusUpdateTime";
	private static String TITLE_ID = "id";
	private static String THUMBNAIL = "thumbnail";

	private TargetedTitlesListPanel view;
	private List<UiColumn> uiColumns;
	List<UiColumn> queryColumns;
	private List<Column<?, TitleRecord>> columns = new ArrayList<Column<?, TitleRecord>>();
	private SortOrder defaultSortOrder;
	

	private Column<?, TitleRecord> defaultSortColumn;
	private List<Column<?, TitleRecord>> sortableColumns = new ArrayList<Column<?, TitleRecord>>();

	private PageFeatureImpl pageFeature;
	private SortFeatureImpl<Long, TitleRecord> sortFeature;
	private SitesConstants constants = GWT.create(SitesConstants.class);

	private IUISiteServiceAsync siteService = GWT.create(IUISiteService.class);
	private TitleClientBundle titleBundle = new TitleClientBundle();

	public TargetedTitlesTableDataProvider(TargetedTitlesListPanel view) {
		this.view = view;
		this.security = NeptuneApplication.getApplication().getSecurity();
	}

	

	/**
	 * Initializes with table column information from the server. This method
	 * has to be called before this data provider can be used by the Table. At
	 * the time of association with the Table, the Table expects column
	 * information to be present as it would call
	 * {@link TargetedTitlesTableDataProvider#getColumns()}.
	 */
	public void init(final AsyncCallback<Void> callback) {
		SpecificationServiceAsync specService = GWT.create(SpecificationService.class);
		specService.getTable(TABLE_NAME, new NeptuneAsyncCallback<UiTable>() {
			@Override
			public void onNeptuneFailure(Throwable caught) {
				callback.onFailure(caught);
			}

			@Override
			public void onNeptuneSuccess(UiTable table) {
				defaultSortOrder = SortOrder.valueOf(table.getDefaultSortingOrder());
				final TargetedTitlesListPanel ttView = view;
				uiColumns = table.getColumns();
				if (uiColumns != null) {
					for (final UiColumn uiColumn : uiColumns) {
						final String colName = TableColumnNameHelper.getColumnName(uiColumn.getName(), uiColumn.getSectionName());
						final String colDisplayName = uiColumn.getDisplayName();
						ColumnBase<String, TitleRecord> column = new ColumnBase<String, TitleRecord>(colName, colDisplayName) {
							public View<String> getView(final TitleRecord record) {
								LabelStringView view = new LabelStringView("");
								if (getName().equalsIgnoreCase(TITLE_ID)) {
									view = new LabelStringView(record.getTitleId().toString());
									if (security.isUserInRole(Permissions.TITLE_VIEW)) {

										view.getWidget().addClickHandler(new ClickHandler() {
											@Override
											public void onClick(ClickEvent event) {
												// Do we need this?
												UISite uiSite = ttView.getUiSite();

												// Create history token
												Map<String, String> map = new HashMap<String, String>();
												map.put("Id", "" + record.getTitleId());
												// Add sites anchor for this
												// page to History object so
												// that we can return to this
												// page
												// when user do a "cancel" on
												// the title details metadata
												// tab.
												// String sourceAnchor =
												// Anchors.getAnchor(uiSite.getId(),
												// uiSite.getParentId(),
												// ttView.getName());
												map.put("returnAnchor", History.getToken());
												AnchorTokenizer tkz = new AnchorTokenizer();

												String historyToken = "Content.Search." + tkz.buildAnchor(map);
												History.newItem(historyToken);
											}
										});

										view.setStyleName(TableConstants.STYLE_DATA_LINK);
									}
								 else {
									view.setStyleName(TableConstants.STYLE_DATA_TEXT);
								 }
								}
								if (getName().equalsIgnoreCase(SITE_STATUS)) {
									View<String> iView = new View<String>() {

										@Override
										public Widget getWidget() {
											return getImageWidget(record);
										}

										@Override
										public String getStyleName() {
											return null;
										}

										@Override
										public void release() {
										}
									};

									return iView;
								}
								if (getName().equalsIgnoreCase(THUMBNAIL)) {
									return new View<String>() {

										@Override
										public Widget getWidget() {
											TitleThumbnailWidget widget = new TitleThumbnailWidget();
											widget.setInput(record.getTitleId(), null);
											return widget;
										}

										@Override
										public String getStyleName() {
											return null;
										}

										@Override
										public void release() {
										}
									};
								}
								List<TitleMetadata> titleMetadataList = record.getMetadata();
								for (TitleMetadata titleMetadata : titleMetadataList) {
									String metadatColName = TableColumnNameHelper.getColumnName(titleMetadata.getName(), titleMetadata.getAssetType());
									if (getName().equalsIgnoreCase(metadatColName)) {
										view = new LabelStringView(titleMetadata.getValue());
									}
								}
								return view;
							}
						};

						column.setCellStyle(getColumnCellStyle(uiColumn));
						if (column.getName().equals(SITE_STATUS) || column.getName().equals(THUMBNAIL)) {
							column.setCellStyle(TableConstants.STYLE_DATACELL_ICON);
						}

						TargetedTitlesTableDataProvider.this.columns.add(column);
						if (column.getName().equals(table.getDefaultSortColumnName()))
							defaultSortColumn = column;
						if (uiColumn.isSortable())
							sortableColumns.add(column);
					}
				}
				callback.onSuccess(null);
			}
		});
	}

	/*
	 * Determine the style to use based on the data type of the column data
	 */
	private String getColumnCellStyle(UiColumn column) {
		String style = TableConstants.STYLE_DATACELL_GENERAL;
		if (column.getDataType() != null) {
			switch (column.getDataType()) {
			case BOOLEAN:
			case STRING:
				style = TableConstants.STYLE_DATACELL_TEXT;
				break;
			case INTEGER:
			case FLOAT:
				style = TableConstants.STYLE_DATACELL_NUMERIC;
				break;
			case DATE:
			case TIME:
				style = TableConstants.STYLE_DATACELL_DATE;
				break;
			}
		}

		return style;
	}

	public Column<?, TitleRecord> getDefaultSortColumn() {
		return defaultSortColumn;
	}

	public SortOrder getDefaultSortOrder() {
		return defaultSortOrder;
	}

	public List<Column<?, TitleRecord>> getSortableColumns() {
		return sortableColumns;
	}

	public void setPageFeature(PageFeatureImpl pageFeature) {
		this.pageFeature = pageFeature;
	}

	public void setSortFeature(SortFeatureImpl<Long, TitleRecord> sortFeature) {
		this.sortFeature = sortFeature;
	}

	@Override
	public void initialize(final AsyncCallback<Void> callback) {
		callback.onSuccess(null);
	}

	@Override
	public List<Column<?, TitleRecord>> getColumns() {
		return columns;
	}

	@Override
	public void getRecord(Long key, AsyncCallback<TitleRecord> callback) {
		throw new UnsupportedOperationException();
	}

	@Override
	public void getRecords(final AsyncCallback<List<TitleRecord>> callback) {

		UISite uiSite = view.getUiSite();

		if (uiSite != null) {
			view.handleGetRecordsRequest();

			Integer siteId = uiSite.getId();
			UISiteType siteType = uiSite.getType();
			UiTargetedTitlesRequest uiTargetedTitlesRequest = new UiTargetedTitlesRequest();
			uiTargetedTitlesRequest.setSrchTxt(view.getTextFilter());
			uiTargetedTitlesRequest.setSiteTitleStatus(view.getStatusFilter());
			uiTargetedTitlesRequest.setColumns(getQueryColumns(uiColumns));
			uiTargetedTitlesRequest.setSiteId(siteId);
			uiTargetedTitlesRequest.setSiteType(siteType);
			uiTargetedTitlesRequest.setStartIndex((pageFeature.getPageNumber() - 1) * pageFeature.getPageSize());
			uiTargetedTitlesRequest.setMaxRecordCount(pageFeature.getPageSize());
			String sortColumnName = sortFeature.getSortColumnName();
			String sortColumnAssetType = "";

			if (TableColumnNameHelper.isMetdataField(sortColumnName)) {
				String columnName = TableColumnNameHelper.getColumnName(sortFeature.getSortColumnName());
				String colSection = TableColumnNameHelper.getSectionName(sortFeature.getSortColumnName()).toUpperCase();

				sortColumnName = columnName;
				sortColumnAssetType = colSection;
			}

			uiTargetedTitlesRequest.setSortColumnName(sortColumnName);
			uiTargetedTitlesRequest.setSortColumnAssetType(sortColumnAssetType);

			UiSortColumnType uiSortColumnType = UiSortColumnType.METADATA;
			if (TITLE_ID.equalsIgnoreCase(sortColumnName)) {
				uiSortColumnType = UiSortColumnType.TITLE;
			}
			if (SITE_STATUS.equalsIgnoreCase(sortColumnName) || SITE_STATUS_UPDATE_TIME.equalsIgnoreCase(sortColumnName)) {
				uiSortColumnType = UiSortColumnType.SITE;
			}
			uiTargetedTitlesRequest.setSortColumnType(uiSortColumnType);
			uiTargetedTitlesRequest.setSortOrder(sortFeature.isAscending() ? SortOrder.ASCENDING.name() : SortOrder.DESCENDING.name());
			siteService.getSiteTitleStatusData(uiTargetedTitlesRequest, new NeptuneAsyncCallback<UISiteTitleStatusData>() {

				@Override
				public void onNeptuneFailure(Throwable caught) {
					if (caught instanceof TargetedTitleQuickSearchException) {
						view.reportFailure(constants.quickSrchErrorMessage());
					} else {
						callback.onFailure(caught);
						pageFeature.setRecordCount(-1);
					}

					/* Also notify the view */
					view.handleGetRecordsFailureResponse(caught);
				}

				@Override
				public void onNeptuneSuccess(UISiteTitleStatusData data) {
					List<TitleRecord> records = new ArrayList<TitleRecord>();

					List<UISiteTitleStatusRecord> ustsRecords = data.getUiSiteTitleStatusRecords();
					for (UISiteTitleStatusRecord ustsRecord : ustsRecords) {
						TitleRecord titleRecord = new TitleRecord();
						titleRecord.setTitleId(ustsRecord.getTitleId());
						titleRecord.setStatus(ustsRecord.getStatus());
						titleRecord.setMetadata(ustsRecord.getMetadata());
						titleRecord.setDistributionTitle(ustsRecord.isDistributionTitle());

						records.add(titleRecord);
					}

					pageFeature.setRecordCount(data.getTotalCount());
					callback.onSuccess(records);

					/* Also notify the view */
					view.handleGetRecordsSuccessResponse(records);
				}

			});
		}
		// End if (uiSite != null)

	}

	private List<UiColumn> getQueryColumns(List<UiColumn> uiColumns) {
		if (queryColumns != null) {
			return queryColumns;
		}

		queryColumns = new ArrayList<UiColumn>();
		for (int i = 0; i < uiColumns.size(); i++) {
			// add everything except the thumbnail column
			if (!uiColumns.get(i).getName().equals(THUMBNAIL)) {
				queryColumns.add(uiColumns.get(i));
				// if this is the status column, means its the site status and
				// not the title status
				if (uiColumns.get(i).getName().equals(SITE_STATUS)) {
					uiColumns.get(i).setFieldType(UIFieldType.SITE);
				} else if (uiColumns.get(i).getName().equals(SITE_STATUS_UPDATE_TIME)) {
					uiColumns.get(i).setFieldType(UIFieldType.SITE);
					uiColumns.get(i).setDataType(UiColumn.DataType.DATE);
				}
			}
		}
		return queryColumns;
	}

	@Override
	public boolean isCheckboxEnabled() {
		return true;
	}

	@Override
	public boolean isRecordCountEnabled() {
		return true;
	}

	private ImageWidget getImageWidget(TitleRecord record) {
		ImageWidget image = new ImageWidget();
		if (record.isDistributionTitle()) {
			titleBundle.updateDistributionStatusIcon(image, record.getStatus());
		} else {
			titleBundle.updateTrackingStatusIcon(image, record.getStatus());
		}
		return image;
	}

	/**
	 * @return the security
	 */
	public NeptuneSecurity getSecurity() {
		return security;
	}
	
}
