package com.ericsson.cms.sites.ui.client.activity;

import java.util.List;

import com.ericsson.cms.sites.ui.client.IUISiteService;
import com.ericsson.cms.sites.ui.client.IUISiteServiceAsync;
import com.ericsson.cms.sites.ui.client.entities.UISite;
import com.ericsson.cms.sites.ui.client.i18n.SitesConstants;
import com.google.gwt.core.client.GWT;
import com.google.gwt.event.dom.client.ChangeEvent;
import com.google.gwt.event.dom.client.ChangeHandler;
import com.google.gwt.event.dom.client.ClickEvent;
import com.google.gwt.event.dom.client.ClickHandler;
import com.google.gwt.i18n.client.DateTimeFormat;
import com.google.gwt.user.client.History;
import com.google.gwt.user.client.ui.Composite;
import com.google.gwt.user.client.ui.Widget;
import com.google.gwt.user.datepicker.client.DateBox;
import com.tandbergtv.cms.portal.ui.title.client.model.title.UIDistributionStatus;
import com.tandbergtv.cms.portal.ui.title.client.model.title.UITrackingStatus;
import com.tandbergtv.neptune.widgettoolkit.client.remote.NeptuneAsyncCallback;
import com.tandbergtv.neptune.widgettoolkit.client.widget.basic.ButtonWidget;
import com.tandbergtv.neptune.widgettoolkit.client.widget.basic.DateBoxWidget;
import com.tandbergtv.neptune.widgettoolkit.client.widget.basic.HTMLWidget;
import com.tandbergtv.neptune.widgettoolkit.client.widget.basic.LabelWidget;
import com.tandbergtv.neptune.widgettoolkit.client.widget.basic.ListBoxWidget;
import com.tandbergtv.neptune.widgettoolkit.client.widget.composite.BusyIndicator;
import com.tandbergtv.neptune.widgettoolkit.client.widget.composite.FormContainer;
import com.tandbergtv.neptune.widgettoolkit.client.widget.composite.HeaderPanel;
import com.tandbergtv.neptune.widgettoolkit.client.widget.composite.messagearea.MessageArea;
import com.tandbergtv.neptune.widgettoolkit.client.widget.composite.resizablecontainer.ResizableContainer;
import com.tandbergtv.neptune.widgettoolkit.client.widget.container.HorizontalContainer;
import com.tandbergtv.neptune.widgettoolkit.client.widget.container.VerticalContainer;
import com.tandbergtv.neptune.widgettoolkit.client.widget.servertime.internal.IServerTimeService;
import com.tandbergtv.neptune.widgettoolkit.client.widget.servertime.internal.IServerTimeServiceAsync;
import com.tandbergtv.neptune.widgettoolkit.client.widget.servertime.internal.ServerTime;
import com.tandbergtv.neptune.widgettoolkit.client.widget.style.StyleNames;

public class SitesActivitySearchPanel extends Composite {
	private static final int DATE_BOX_LENGTH = 10;
	static final DateTimeFormat DATE_FORMAT = DateTimeFormat.getFormat("yyyy-MM-dd");
	
	private VerticalContainer mainContainer;
	private MessageArea messageArea;
	private HeaderPanel header;
	private ResizableContainer resizableContainer;
	private FormContainer formContainer;
	private ListBoxWidget<UISite> distSiteWidget;
	private ListBoxWidget<UIDistributionStatus> distStatusWidget;
	private ListBoxWidget<UISite> trackSiteWidget;
	private ListBoxWidget<UITrackingStatus> trackStatusWidget;
	private DateBoxWidget fromDateBox, toDateBox;
	private BusyIndicator busyIndicator = new BusyIndicator();
	
	private IUISiteServiceAsync service = GWT.create(IUISiteService.class);
	private IServerTimeServiceAsync timeService = GWT.create(IServerTimeService.class);
	private SitesConstants constants = GWT.create(SitesConstants.class);
	
	private ServerTime serverTime;
	
	static class SearchForm {
		Integer distSiteId, trackSiteId;
		UIDistributionStatus distStatus;
		UITrackingStatus trackStatus; 
		String fromDateStr, toDateStr;
		boolean trackSiteEnabled;
	}
	static SearchForm searchForm;
		
	public SitesActivitySearchPanel() {
		buildComponents();
		loadData();
		this.initWidget(mainContainer);
	}

	public void buildComponents() {
		mainContainer = new VerticalContainer();
		mainContainer.setWidth("100%");
		mainContainer.setSpacing(2);

		messageArea = new MessageArea();
		messageArea.reset();
		header = new HeaderPanel("Sites Activity");

		formContainer = new FormContainer(HorizontalContainer.ALIGN_LEFT);
		
		distSiteWidget = new ListBoxWidget<UISite>();
		distSiteWidget.addChangeHandler(new ChangeHandler() {
			@Override
			public void onChange(ChangeEvent event) {
				reloadTrackingSites();
			}});
		
		distStatusWidget = new ListBoxWidget<UIDistributionStatus>();
		distStatusWidget.addItem("Distribution Failed", UIDistributionStatus.FAILED);
		distStatusWidget.addItem("Distributed", UIDistributionStatus.DISTRIBUTED);
		
		trackSiteWidget = new ListBoxWidget<UISite>();
		
		trackStatusWidget = new ListBoxWidget<UITrackingStatus>();
		trackStatusWidget.addItem("Deployment Failed", UITrackingStatus.DEPLOYMENT_FAILED);
		trackStatusWidget.addItem("Not Deployed", UITrackingStatus.NOT_DEPLOYED);
		trackStatusWidget.addItem("Deployed", UITrackingStatus.DEPLOYED);
		
		formContainer.addRow("Distribution Site", distSiteWidget);
		formContainer.addRow("Distribution Status", distStatusWidget);
		formContainer.addRow("Tracking Site", trackSiteWidget);
		formContainer.addRow("Tracking Status", trackStatusWidget);
		formContainer.addRow("Date Range", getDateRangeWidget(), true);
		
		resizableContainer = new ResizableContainer(formContainer, "Search", null);

		resizableContainer.addHeightOffsettingWidget(header);
		resizableContainer.addHeightOffsettingWidget(messageArea);
		resizableContainer.setOffsetHeight(10);
		ButtonWidget searchButton = new ButtonWidget("Search");
		searchButton.addStyleDependentName(StyleNames.DO_BUTTON_STYLE);
		searchButton.addClickHandler(new ClickHandler() {
			@Override
			public void onClick(ClickEvent event) {
				messageArea.reset();
				if(validate()) {
					searchForm = new SearchForm();
					searchForm.distSiteId = (distSiteWidget.getSelectedItem() == null) ?
							null : distSiteWidget.getSelectedItem().getId();
					searchForm.distStatus = distStatusWidget.getSelectedItem();
					
					searchForm.trackSiteEnabled = trackSiteWidget.isEnabled();
					searchForm.trackSiteId = null;
					searchForm.trackStatus = null;
					if(searchForm.trackSiteEnabled) {
						searchForm.trackSiteId = (trackSiteWidget.getSelectedItem() == null) ?
								null : trackSiteWidget.getSelectedItem().getId();
						searchForm.trackStatus = trackStatusWidget.getSelectedItem();
					}
					
					searchForm.fromDateStr = fromDateBox.getTextBox().getText();
					searchForm.toDateStr = toDateBox.getTextBox().getText();
					
					History.newItem("Sites.Activity.Results");
				}
			}
		});

		resizableContainer.addButton("Search", searchButton);

		mainContainer.add(header);
		mainContainer.add(messageArea);
		mainContainer.add(resizableContainer);
	}

	private Widget getDateRangeWidget() {
		HorizontalContainer container = new HorizontalContainer();
		container.setVerticalAlignment(HorizontalContainer.ALIGN_TOP);
		
		fromDateBox = new DateBoxWidget();
		fromDateBox.getTextBox().setVisibleLength(DATE_BOX_LENGTH);
		DateBox.DefaultFormat fmt = new DateBox.DefaultFormat(DATE_FORMAT);
		fromDateBox.setFormat(fmt);
		container.add(fromDateBox);
		
		container.add(new HTMLWidget("&nbsp;&nbsp;"));
		container.add(new LabelWidget("to"));
		container.add(new HTMLWidget("&nbsp;&nbsp;"));
		
		toDateBox = new DateBoxWidget();
		toDateBox.getTextBox().setVisibleLength(DATE_BOX_LENGTH);
		toDateBox.setFormat(fmt);
		container.add(toDateBox);
		
		return container;
	}
	
	private void loadData() {
		busyIndicator.center();
		service.getAllSites(new NeptuneAsyncCallback<List<UISite>>() {
			@Override
			public void onNeptuneFailure(Throwable caught) {
				messageArea.setErrorMessage("Unable to get list of sites."
						+ getFailureMessageSuffix(caught));
				busyIndicator.hide();
			}

			@Override
			public void onNeptuneSuccess(List<UISite> result) {
				distSiteWidget.clear();
				distSiteWidget.addItem("All", null);
				trackSiteWidget.clear();
				trackSiteWidget.addItem("All", null);
				for(UISite site : result) {
					switch(site.getType()) {
					case DISTRIBUTION:
						distSiteWidget.addItem(site.getName(), site);
						break;
					case TRACKING:
						trackSiteWidget.addItem(site.getName(), site);
						break;
					}
				}
				setDateRangeToServerDate();
			}
		});
	}
	
	private void setDateRangeToServerDate() {
		timeService.getCurrentTime(new NeptuneAsyncCallback<ServerTime>() {
			@Override
			public void onNeptuneFailure(Throwable caught) {
				messageArea.setErrorMessage("Unable to get server time."
						+ getFailureMessageSuffix(caught));
				busyIndicator.hide();
			}

			@Override
			public void onNeptuneSuccess(ServerTime result) {
				SitesActivitySearchPanel.this.serverTime = result;
				fromDateBox.setValue(result.getDate());
				toDateBox.setValue(result.getDate());
				busyIndicator.hide();
			}});
	}

	private void reloadTrackingSites() {
		busyIndicator.center();
		Integer distSiteId = (distSiteWidget.getSelectedItem() == null) ?
				null : distSiteWidget.getSelectedItem().getId();
		service.getAllTrackingSites(distSiteId,
				new NeptuneAsyncCallback<List<UISite>>() {
			@Override
			public void onNeptuneFailure(Throwable caught) {
				messageArea.setErrorMessage("Unable to get list of tracking sites."
						+ getFailureMessageSuffix(caught));
				busyIndicator.hide();
			}
			
			@Override
			public void onNeptuneSuccess(List<UISite> result) {
				boolean emptyResult = (result == null || result.isEmpty());
				
				trackSiteWidget.setEnabled(!emptyResult);
				trackStatusWidget.setEnabled(!emptyResult);
				
				trackSiteWidget.clear();
				trackSiteWidget.addItem("All", null);
				for(UISite site : result)
					trackSiteWidget.addItem(site.getName(), site);
				
				busyIndicator.hide();
			}
		});
	}
	
	private boolean validate() {
		boolean error = false;
		StringBuilder html = new StringBuilder();
		html.append("<ul>");
		if(fromDateBox.getValue() == null || toDateBox.getValue() == null) {
			html.append("Provide Date Range values in YYYY-MM-DD format.");
			error = true;
		} else {
			if(fromDateBox.getValue().after(toDateBox.getValue())) {
				html.append("For Date Range field, the From Date should not be later than To Date.");
				error = true;
			}
			if(toDateBox.getValue().after(serverTime.getDate())) {
				html.append("Values of Date Range field should not be in the future.");
				error = true;
			}
		}
		
		html.append("</ul>");
		if(error)
			messageArea.setErrorHTML(html.toString());
		return !error;
	}

	private String getFailureMessageSuffix(Throwable caught) {
		return " | " + constants.reason() + ": "
				+ (caught != null ? caught.getLocalizedMessage() : "");
	}

	@Override
	protected void onUnload() {
		if (busyIndicator != null)
			busyIndicator.hide();
		
		super.onUnload();
	}
}
