/*
 * Created on Sep 11, 2009
 * 
 * (C) Copyright TANDBERG Television Ltd.
 */

package com.tandbergtv.cms.portal.ui.title.client.service.specification;

import java.util.ArrayList;
import java.util.Collection;
import java.util.List;

import com.google.gwt.core.client.GWT;
import com.google.gwt.user.client.rpc.AsyncCallback;
import com.tandbergtv.cms.portal.ui.title.client.model.specification.UIAssetDefinition;
import com.tandbergtv.cms.portal.ui.title.client.model.specification.UIAssetSpecification;
import com.tandbergtv.cms.portal.ui.title.client.model.specification.UIAssetSpecificationListItem;
import com.tandbergtv.neptune.widgettoolkit.client.remote.NeptuneAsyncCallback;

/**
 * A proxy to the specification manager asynchronous service that adds caching
 * 
 * @author Vijay Silva
 */
public class UISpecificationClientManager implements IUISpecificationManagerAsync {

	/* The singleton */
	private static final UISpecificationClientManager manager = new UISpecificationClientManager();

	/* The state */
	private final IUISpecificationManagerAsync service;
	private final UISpecificationCacheService defaultCacheService;

	/* Prevent construction, this service is a singleton */
	private UISpecificationClientManager() {
		service = GWT.create(IUISpecificationManager.class);
		defaultCacheService = new UISpecificationCacheService(service);
	}

	/**
	 * Get the manager service instance
	 * 
	 * @return The manager
	 */
	public static UISpecificationClientManager getInstance() {
		return manager;
	}

	@Override
	public void getUISpecification(AsyncCallback<UIAssetSpecification> callback) {
		defaultCacheService.getSpecification(callback);
	}

	@Override
	public void getAll(AsyncCallback<List<UIAssetSpecificationListItem>> callback) {
		service.getAll(callback);
	}

	@Override
	public void getDefaultListItem(AsyncCallback<UIAssetSpecificationListItem> callback) {
		service.getDefaultListItem(callback);
	}

	/* Cache Service that gets the specification from the server and caches it */
	private static final class UISpecificationCacheService {
		private final IUISpecificationManagerAsync service;
		private final String name;
		private final List<AsyncCallback<UIAssetSpecification>> callbacks = new ArrayList<AsyncCallback<UIAssetSpecification>>();
		private boolean loading = false;
		private UIAssetSpecification specification = null;

		/* Constructor for specification with given name */
		public UISpecificationCacheService(IUISpecificationManagerAsync service, String name) {
			this.service = service;
			this.name = name;
		}

		/* Constructor for default specification */
		public UISpecificationCacheService(IUISpecificationManagerAsync service) {
			this.service = service;
			this.name = null;
		}

		/* Get the specification */
		public synchronized void getSpecification(AsyncCallback<UIAssetSpecification> callback) {
			if (this.loading) {
				/* The specification is currently loading, register callback */
				callbacks.add(callback);
			} else if (specification != null) {
				/* The specification has been cached before, notify callback immediately */
				callback.onSuccess(specification);
			} else {
				/* The specification needs to be fetched from the server and cached */
				this.loading = true;
				callbacks.add(callback);
				service.getUISpecification(new NeptuneAsyncCallback<UIAssetSpecification>() {
					@Override
					public void onNeptuneSuccess(UIAssetSpecification result) {
						handleGetSpecificationSuccess(result);
					}

					@Override
					public void onNeptuneFailure(Throwable caught) {
						handleGetSpecificationFailure(caught);
					}
				});
			}
		}

		/*
		 * Attempt to load the specification succeeded, notify all pending callbacks
		 */
		private synchronized void handleGetSpecificationSuccess(UIAssetSpecification result) {
			loading = false;
			specification = result;
			for (AsyncCallback<UIAssetSpecification> callback : callbacks) {
				try {
					callback.onSuccess(result);
				} catch (Exception e) {
					/* callback might fail, continue to notify all callbacks */
				}
			}

			callbacks.clear();
		}

		/*
		 * Attempt to load the specification failed, notify all pending callbacks
		 */
		private synchronized void handleGetSpecificationFailure(Throwable error) {
			loading = false;
			for (AsyncCallback<UIAssetSpecification> callback : callbacks) {
				try {
					callback.onFailure(error);
				} catch (Exception e) {
					/* callback might fail, continue to notify all callbacks */
				}
			}

			callbacks.clear();
		}
	}

	@Override
	public void getPreviewSpecs(AsyncCallback<List<UIAssetSpecificationListItem>> callback) 
	{
		service.getPreviewSpecs(callback);
	}

	@Override
	public void getAllUISpecifications(
			AsyncCallback<Collection<UIAssetDefinition>> callback) {
		service.getAllUISpecifications(callback);
		
	}
	
	@Override
	public void getUIAssetSpecification(String type,AsyncCallback<UIAssetSpecification> callback){
		service.getUIAssetSpecification(type, callback);
	}
}
