/*
 * Created on May 27, 2009
 * 
 * (C) Copyright TANDBERG Television Ltd.
 */

package com.tandbergtv.cms.portal.ui.title.client.view.datatype.factory;

import java.util.Set;

import com.google.gwt.core.client.GWT;
import com.tandbergtv.cms.portal.ui.title.client.model.specification.UIDateFieldDefinition;
import com.tandbergtv.cms.portal.ui.title.client.model.specification.UITimeFieldDefinition;
import com.tandbergtv.cms.portal.ui.title.client.model.specification.option.UIValueOption;
import com.tandbergtv.cms.portal.ui.title.client.widget.TypedPositionFieldWidget;
import com.tandbergtv.neptune.widgettoolkit.client.widget.datatype.DataTypeWidget;
import com.tandbergtv.neptune.widgettoolkit.client.widget.datatype.TypedComboBoxWidget;
import com.tandbergtv.neptune.widgettoolkit.client.widget.datatype.TypedDateBoxWidget;
import com.tandbergtv.neptune.widgettoolkit.client.widget.datatype.TypedListBoxWidget;
import com.tandbergtv.neptune.widgettoolkit.client.widget.datatype.TypedTextAreaWidget;
import com.tandbergtv.neptune.widgettoolkit.client.widget.datatype.TypedTextBoxWidget;
import com.tandbergtv.neptune.widgettoolkit.client.widget.datatype.converter.BooleanConverter;
import com.tandbergtv.neptune.widgettoolkit.client.widget.datatype.converter.DateConverter;
import com.tandbergtv.neptune.widgettoolkit.client.widget.datatype.converter.FloatConverter;
import com.tandbergtv.neptune.widgettoolkit.client.widget.datatype.converter.IDateConverter;
import com.tandbergtv.neptune.widgettoolkit.client.widget.datatype.converter.IValueConverter;
import com.tandbergtv.neptune.widgettoolkit.client.widget.datatype.converter.IntegerConverter;
import com.tandbergtv.neptune.widgettoolkit.client.widget.datatype.converter.StringConverter;

/**
 * The default widget factory that creates data type widgets.
 * 
 * @author Vijay Silva
 */
public class DefaultDataTypeWidgetFactory extends DataTypeWidgetFactory {

	/* Properties */
	private DataTypeWidgetMessages messages = GWT.create(DataTypeWidgetMessages.class);

	/* The patterns for the input date and time values */
	private String inputDateFormat = UIDateFieldDefinition.DATE_PATTERN;
	private String inputTimeFormat = UITimeFieldDefinition.TIME_PATTERN;

	/* The combo box width to use */
	private static final int COMBOBOX_WIDTH = 500;

	/**
	 * The default data type widget factory
	 */
	public DefaultDataTypeWidgetFactory() {
	}

	/**
	 * Get the messages to use for the default data type widgets.
	 * 
	 * @return The data type widgets
	 */
	protected DataTypeWidgetMessages getMessages() {
		return this.messages;
	}

	/**
	 * Create a widget for editing a boolean value. Creates a list box widget with default true /
	 * false options. If there are options provided in the configuration, uses the provided options
	 * (since they may have different display names).
	 * 
	 * @param configuration The widget configuration used for building the widget
	 * @return The Data Type Widget for boolean values
	 */
	@Override
	public DataTypeWidget<Boolean> createBooleanWidget(WidgetConfiguration configuration) {
		IValueConverter<Boolean> converter = new BooleanConverter();
		TypedListBoxWidget<Boolean> widget = new TypedListBoxWidget<Boolean>(converter);
		widget.addItem(messages.booleanWidgetUndefinedValue(), null);

		if (!configuration.hasOptions()) {
			/* No options, use plain list box widget with default options */
			widget.addItem(messages.booleanWidgetTrueValue(), new Boolean(true));
			widget.addItem(messages.booleanWidgetFalseValue(), new Boolean(false));
		} else {
			/* options present and values limited, use list box widget */
			Set<String> excludedKeys = configuration.getExcludedOptionKeys();
			for (UIValueOption valueOption : configuration.getOptions()) {
				// Don't add excluded options
				if (excludedKeys != null && excludedKeys.contains(valueOption.getName())) {
					continue;
				}
				widget.addItem(valueOption.getName(), Boolean.parseBoolean(valueOption.getValue()));
			}
		}

		/* Set the tool tips */
		if (!configuration.isReadOnly()) {
			widget.setToolTip(messages.booleanWidgetToolTip());
			widget.setErrorToolTip(messages.booleanWidgetErrorToolTip());
		} else {
			widget.setReadOnly(true);
		}

		return widget;
	}

	/**
	 * Create a widget for editing a integer value. Creates a TextBox widget, unless there are
	 * options available. If there are options available, creates a ListBox widget or a ComboBox
	 * widget.
	 * 
	 * @param options The list of value options allowed for entry by the user. The option list can
	 * be null / empty if no options are to be provided.
	 * @param anyValueAllowed a flag indicating if any value is allowed or only values from the
	 * provided options are allowed. This flag is ignored if no options are present.
	 * @return The Data Type Widget for string widgets
	 */
	@Override
	public DataTypeWidget<Long> createIntegerWidget(WidgetConfiguration configuration) {
		IValueConverter<Long> converter = new IntegerConverter();
		DataTypeWidget<Long> widget = null;

		if (!configuration.hasOptions()) {
			/* No options, use typed text box widget */
			widget = new TypedTextBoxWidget<Long>(converter);
		} else if (!configuration.isAnyValueAllowed()) {
			TypedListBoxWidget<Long> listBox = new TypedListBoxWidget<Long>(new IntegerConverter());
			listBox.addItem("", null);

			Set<String> excludedKeys = configuration.getExcludedOptionKeys();
			for (UIValueOption valueOption : configuration.getOptions()) {
				// Don't add excluded options
				if (excludedKeys != null && excludedKeys.contains(valueOption.getName())) {
					continue;
				}
				listBox.addItem(valueOption.getName(), Long.parseLong(valueOption.getValue()));
			}
			widget = listBox;
		} else {
			TypedComboBoxWidget<Long> combo = new TypedComboBoxWidget<Long>(converter,
					COMBOBOX_WIDTH);
			combo.addItem("", null);

			Set<String> excludedKeys = configuration.getExcludedOptionKeys();
			for (UIValueOption valueOption : configuration.getOptions()) {
				// Don't add excluded options
				if (excludedKeys != null && excludedKeys.contains(valueOption.getName())) {
					continue;
				}
				combo.addItem(valueOption.getName(), Long.parseLong(valueOption.getValue()));
			}
			widget = combo;
		}

		/* Set the tool tips */
		if (!configuration.isReadOnly()) {
			widget.setToolTip(messages.integerWidgetToolTip());
			widget.setErrorToolTip(messages.integerWidgetErrorToolTip());
		} else {
			widget.setReadOnly(true);
		}

		return widget;
	}

	/**
	 * Create a widget for editing a float value. Creates a TextBox widget, unless there are options
	 * available. If there are options available, creates a ComboBox widget.
	 * 
	 * @param configuration The widget configuration used for building the widget
	 * @return The Data Type Widget for float values
	 */
	@Override
	public DataTypeWidget<String> createFloatWidget(WidgetConfiguration configuration) {
		IValueConverter<String> converter = new FloatConverter();
		DataTypeWidget<String> widget = null;

		if (!configuration.hasOptions()) {
			/* No options, use typed text box widget */
			widget = new TypedTextBoxWidget<String>(converter);
		} else if (!configuration.isAnyValueAllowed()) {
			TypedListBoxWidget<String> listBox = new TypedListBoxWidget<String>(converter);
			listBox.addItem("", null);
			for (UIValueOption valueOption : configuration.getOptions())
				listBox.addItem(valueOption.getName(), valueOption.getValue());
			widget = listBox;
		} else {
			TypedComboBoxWidget<String> combo = new TypedComboBoxWidget<String>(converter,
					COMBOBOX_WIDTH);
			combo.addItem("", null);
			for (UIValueOption valueOption : configuration.getOptions())
				combo.addItem(valueOption.getName(), valueOption.getValue());
			widget = combo;
		}

		/* Set the tool tips */
		if (!configuration.isReadOnly()) {
			widget.setToolTip(messages.floatWidgetToolTip());
			widget.setErrorToolTip(messages.floatWidgetErrorToolTip());
		} else {
			widget.setReadOnly(true);
		}

		return widget;
	}

	/**
	 * Create a widget for editing a date value. Always creates a DateBox widget, without any
	 * options.
	 * 
	 * @param configuration The widget configuration
	 * @return The Date Data Type Widget
	 */
	@Override
	public DataTypeWidget<String> createDateWidget(DateWidgetConfiguration configuration) {
		String dateFormat = configuration.getDateFormat();
		if (dateFormat == null || dateFormat.trim().length() == 0)
			dateFormat = messages.dateWidgetFormat();
		IDateConverter converter = new DateConverter(this.inputDateFormat, dateFormat);
		DataTypeWidget<String> widget = new TypedDateBoxWidget(converter);

		// Tool tips
		if (!configuration.isReadOnly()) {
			widget.setToolTip(messages.dateWidgetToolTip(dateFormat));
			widget.setErrorToolTip(messages.dateWidgetErrorToolTip(dateFormat));
		} else {
			widget.setReadOnly(true);
		}

		return widget;
	}

	/**
	 * Create a widget for editing a time value. Always creates a TextBox widget, without any
	 * options.
	 * 
	 * @param configuration The widget configuration
	 * @return The Time Data Type Widget
	 */
	@Override
	public DataTypeWidget<String> createTimeWidget(TimeWidgetConfiguration configuration) {
		String timeFormat = configuration.getTimeFormat();
		if (timeFormat == null || timeFormat.trim().length() == 0)
			timeFormat = messages.timeWidgetFormat();
		IValueConverter<String> converter = new DateConverter(this.inputTimeFormat, timeFormat);
		DataTypeWidget<String> widget = new TypedTextBoxWidget<String>(converter);

		// Tool tips
		if (!configuration.isReadOnly()) {
			widget.setToolTip(messages.timeWidgetToolTip(timeFormat));
			widget.setErrorToolTip(messages.timeWidgetErrorToolTip(timeFormat));
		} else {
			widget.setReadOnly(true);
		}

		return widget;
	}

	/**
	 * Create a widget for editing a string value. Always creates a TextArea widget if the longForm
	 * flag is set. Creates a TextBox widget otherwise, unless there are options available. If there
	 * are options available, creates a ComboBox widget.
	 * 
	 * @param configuration The widget configuration
	 * @return The String Data Type Widget
	 */
	@Override
	public DataTypeWidget<String> createStringWidget(StringWidgetConfiguration configuration) {
		IValueConverter<String> converter = new StringConverter();
		DataTypeWidget<String> widget = null;

		if (configuration.isLongForm()) {
			TypedTextAreaWidget<String> textArea = new TypedTextAreaWidget<String>(converter);
			widget = textArea;
		} else if (configuration.hasOptions() && !configuration.isAnyValueAllowed()) {
			TypedListBoxWidget<String> listBox = new TypedListBoxWidget<String>(converter);
			/* The default item */
			listBox.addItem("", null);

			/* Add options */
			Set<String> excludedKeys = configuration.getExcludedOptionKeys();
			for (UIValueOption option : configuration.getOptions()) {
				// Don't add excluded options
				if (excludedKeys != null && excludedKeys.contains(option.getName())) {
					continue;
				}
				listBox.addItem(option.getName(), option.getValue());
			}

			widget = listBox;
		} else if (configuration.hasOptions()) {
			/* Need to switch to combo-box widget */
			TypedComboBoxWidget<String> comboBox = new TypedComboBoxWidget<String>(converter,
					COMBOBOX_WIDTH);
			/* The default item */
			comboBox.addItem("", null);

			/* Add options */
			Set<String> excludedKeys = configuration.getExcludedOptionKeys();
			for (UIValueOption option : configuration.getOptions()) {
				// Don't add excluded options
				if (excludedKeys != null && excludedKeys.contains(option.getName())) {
					continue;
				}
				comboBox.addItem(option.getName(), option.getValue());
			}

			widget = comboBox;
		} else if (configuration.isPositionField()) {
			widget = new TypedPositionFieldWidget<String>(converter);
		} else {
			/* No options, use plain text box widget */
			widget = new TypedTextBoxWidget<String>(converter);
		}

		if (!configuration.isReadOnly()) {
			widget.setToolTip(messages.stringWidgetToolTip());
			widget.setErrorToolTip(messages.stringWidgetErrorToolTip());
		} else {
			widget.setReadOnly(true);
		}

		return widget;
	}

	/**
	 * Get the format of the input date that the widget should expect
	 * 
	 * @return The input date format
	 */
	public String getInputDatePattern() {
		return this.inputDateFormat;
	}

	/**
	 * Set the format of the input date that the widget should expect
	 * 
	 * @param format The input date format
	 */
	public void setInputDateFormat(String format) {
		this.inputDateFormat = format;
	}

	/**
	 * Get the format of the input time that the widget should expect
	 * 
	 * @return The input time format
	 */
	public String getInputTimeFormat() {
		return this.inputTimeFormat;
	}

	/**
	 * Set the format of the input time that the widget should expect
	 * 
	 * @param format The input time format
	 */
	public void setInputTimeFormat(String format) {
		this.inputTimeFormat = format;
	}
}
