package com.tandbergtv.cms.portal.content.titlelist.client.tab;

import java.util.ArrayList;
import java.util.List;

import com.google.gwt.core.client.GWT;
import com.google.gwt.user.client.rpc.AsyncCallback;
import com.tandbergtv.cms.portal.content.titlelist.client.i18n.TitleListConstants;
import com.tandbergtv.cms.portal.content.titlelist.client.model.UiTitleList;
import com.tandbergtv.cms.portal.content.titlelist.client.model.UiTitleLists;
import com.tandbergtv.cms.portal.content.titlelist.client.rpc.ITitleListService;
import com.tandbergtv.cms.portal.content.titlelist.client.rpc.ITitleListServiceAsync;
import com.tandbergtv.neptune.widgettoolkit.client.remote.NeptuneAsyncCallback;
import com.tandbergtv.neptune.widgettoolkit.client.widget.composite.table.Column;
import com.tandbergtv.neptune.widgettoolkit.client.widget.composite.table.ColumnBase;
import com.tandbergtv.neptune.widgettoolkit.client.widget.composite.table.DataProvider;
import com.tandbergtv.neptune.widgettoolkit.client.widget.composite.table.TableConstants;
import com.tandbergtv.neptune.widgettoolkit.client.widget.composite.table.feature.PageFeature;
import com.tandbergtv.neptune.widgettoolkit.client.widget.composite.table.feature.impl.SortFeatureImpl;
import com.tandbergtv.neptune.widgettoolkit.client.widget.composite.table.view.View;
import com.tandbergtv.neptune.widgettoolkit.client.widget.composite.table.view.label.LabelStringView;

/**
 * Data provider which operates on title lists.
 * 
 * @author spuranik
 * 
 */
public class TitleListsDataProvider implements DataProvider<UiTitleListKey, UiTitleListRecord> {
	
	// columns for this table
	private final List<Column<?, UiTitleListRecord>> columns;
	private final ColumnBase<String, UiTitleListRecord> idColumn;
	private final ColumnBase<String, UiTitleListRecord> nameColumn;
	
	// RPC service to get data from the backend
	ITitleListServiceAsync titleListService = GWT.create(ITitleListService.class);
	
	private TitleListConstants constants = (TitleListConstants) GWT.create(TitleListConstants.class);
	
	private PageFeature parentPageFeature;
	private SortFeatureImpl<UiTitleListKey, UiTitleListRecord> parentSortFeature;
	
	// column names
	private static String ID_COLUMN = "id";
	private static String NAME_COLUMN = "name";
	
	// lists panel
	private TitleListsPanel listsPanel;
	private IBusyNotifier busyNotifier;
	
	public TitleListsDataProvider(TitleListsPanel listsPanel) {
		this.listsPanel = listsPanel;
		
		// create the columns - id and name
		columns = new ArrayList<Column<?, UiTitleListRecord>>();

		idColumn = new ColumnBase<String, UiTitleListRecord>(ID_COLUMN, constants.titleListIdColumn()) {
			public View<String> getView(UiTitleListRecord record) {
				return new LabelStringView(String.valueOf(record.getKey().getId()));
			}
		};
		idColumn.setCellStyle(TableConstants.STYLE_DATACELL_NUMERIC);
		columns.add(idColumn);

		nameColumn = new ColumnBase<String, UiTitleListRecord>(NAME_COLUMN, constants.titleListNameColumn()) {
			public View<String> getView(UiTitleListRecord record) {
				return new LabelStringView(record.getTitleList().getName());
			}
		};
		nameColumn.setCellStyle(TableConstants.STYLE_DATACELL_TEXT);
		columns.add(nameColumn);
	}
	
	public ColumnBase<String, UiTitleListRecord> getDefaultSortColumn() {
		return idColumn;
	}
	
	public List<Column<?, UiTitleListRecord>> getSortColumns() {
		return columns;
	}
	
	@Override
	public List<Column<?, UiTitleListRecord>> getColumns() {
		return columns;
	}

	@Override
	public void getRecord(UiTitleListKey key, final AsyncCallback<UiTitleListRecord> callback) {
		notifyBusyStart();
		titleListService.get(key.getId(), new NeptuneAsyncCallback<UiTitleList>() {

			@Override
			public void onNeptuneFailure(Throwable caught) {
				notifyBusyFinish();
				callback.onFailure(caught);
			}

			@Override
			public void onNeptuneSuccess(UiTitleList result) {				
				notifyBusyFinish();
				callback.onSuccess(new UiTitleListRecord(result));
			} });
	}

	@Override
	public void getRecords(final AsyncCallback<List<UiTitleListRecord>> callback) {
		// get all title lists in the system and prepare ui objects
		int length = getParentPageFeature().getPageSize();
		int start = (getParentPageFeature().getPageNumber() - 1) * length;
		notifyBusyStart();
		titleListService.getTitlelists(start, length, getParentSortFeature().getSortColumnName(), 
				getParentSortFeature().isAscending(), new NeptuneAsyncCallback<UiTitleLists>() {
			public void onNeptuneFailure(Throwable caught) {
				getParentPageFeature().setRecordCount(-1);
				notifyBusyFinish();
				callback.onFailure(caught);
			}

			public void onNeptuneSuccess(UiTitleLists result) {
				getParentPageFeature().setRecordCount(result.getTotalCount());
				notifyBusyFinish();
				callback.onSuccess(prepareUIObjects(result));
			}

			private List<UiTitleListRecord> prepareUIObjects(UiTitleLists result) {
				List<UiTitleListRecord> records = new ArrayList<UiTitleListRecord>();
				for(UiTitleList list : result.getLists()) {
					records.add(new UiTitleListRecord(list));
				}
				return records;
			}
		});		
	}

	@Override
	public void initialize(AsyncCallback<Void> callback) {
		callback.onSuccess(null);
	}

	@Override
	public boolean isCheckboxEnabled() {
		return listsPanel.hasActions();
	}

	@Override
	public boolean isRecordCountEnabled() {
		return true;
	}

	/**
	 * @return the parentPageFeature
	 */
	public PageFeature getParentPageFeature() {
		return parentPageFeature;
	}

	/**
	 * @param parentPageFeature the parentPageFeature to set
	 */
	public void setParentPageFeature(PageFeature parentPageFeature) {
		this.parentPageFeature = parentPageFeature;
	}

	/**
	 * @return the parentSortFeature
	 */
	public SortFeatureImpl<UiTitleListKey, UiTitleListRecord> getParentSortFeature() {
		return parentSortFeature;
	}

	/**
	 * @param parentSortFeature the parentSortFeature to set
	 */
	public void setParentSortFeature(
			SortFeatureImpl<UiTitleListKey, UiTitleListRecord> parentSortFeature) {
		this.parentSortFeature = parentSortFeature;
	}
	
	public static interface IBusyNotifier {
		void startingToBeBusy();
		void doneBeingBusy();
	}
	
	public void setBusyNotifier(IBusyNotifier busyNotifier) {
		this.busyNotifier = busyNotifier;
	}
	
	private void notifyBusyStart() {
		if(busyNotifier != null)
			busyNotifier.startingToBeBusy();
	}
	
	private void notifyBusyFinish() {
		if(busyNotifier != null)
			busyNotifier.doneBeingBusy();
	}

}
