package com.tandbergtv.cms.ui.workflow.client;

import java.util.Collections;
import java.util.Comparator;
import java.util.List;

import com.google.gwt.core.client.GWT;
import com.google.gwt.user.client.Timer;
import com.tandbergtv.neptune.widgettoolkit.client.remote.NeptuneAsyncCallback;
import com.tandbergtv.neptune.widgettoolkit.client.widget.basic.HyperlinkWidget;
import com.tandbergtv.neptune.widgettoolkit.client.widget.basic.LabelWidget;
import com.tandbergtv.neptune.widgettoolkit.client.widget.container.FlexTableContainer;
import com.tandbergtv.neptune.widgettoolkit.client.widget.container.Portlet;
import com.tandbergtv.neptune.widgettoolkit.client.widget.event.DocumentAttachEvent;
import com.tandbergtv.neptune.widgettoolkit.client.widget.event.DocumentAttachHandler;
import com.tandbergtv.neptune.widgettoolkit.client.widget.event.DocumentDetachEvent;
import com.tandbergtv.neptune.widgettoolkit.client.widget.event.DocumentDetachHandler;

/**
 * CSS:
 * portlet-userTasks-text - text
 * portlet-userTasks-link - hyperlink
 * 
 * @author rprakash
 */
public class UserTaskListPortlet extends Portlet {

	private static final int MAX_NUMBER_OF_COLUMNS = 4;
	
	private FlexTableContainer table;
	private IListUserTasksAsync service;
	private PortletConstants constants;
	private Timer refreshTimer;
	private int refreshPeriod;
	private UiTable uiTable;
	
	public UserTaskListPortlet(int refreshTimeMillis) {
        super("USER_TASK_LIST");
		this.refreshPeriod = refreshTimeMillis;
		
		service = GWT.create(IListUserTasks.class);
		constants = GWT.create(PortletConstants.class);
		
		setTitle(constants.userTaskTitle());
		setDisplayName(constants.userTaskTitle());

		table = new FlexTableContainer();
		table.setCellSpacing(0);
		table.setStyleName("portletFixedTable");
		table.setWidth("100%");
		add(table);

		this.refreshTimer = new Timer() {
			@Override
			public void run() {
				reload();
			}
		};
		
		EventHandler handler = new EventHandler();
		addDocumentAttachHandler(handler);
		addDocumentDetachHandler(handler);
	}
	
	/**
	 * prepares table based on default UI specification
	 */
	private void initializeTable() {
		// get table columns
		service.getUserTasksListTableConfiguration(new NeptuneAsyncCallback<UiTable>() {
			@Override
			public void onNeptuneFailure(Throwable caught) {
			}

			@Override
			public void onNeptuneSuccess(UiTable uiTable) {

				setTable(uiTable);
			}
		});
	}
	
	protected void setTable(UiTable uiTable) {
		this.uiTable = uiTable;
	}

	private void reload() {
		initializeTable();
		service.getUserTasks(new NeptuneAsyncCallback<UserTaskList>() {
			@Override
			public void onNeptuneFailure(Throwable caught) {
			}

			@Override
			public void onNeptuneSuccess(UserTaskList userTaskList) {
				updateTitle(userTaskList);

				clearTableData();
				
				if(userTaskList.getNumUserTasks() == 0)
					return;
				
				prepareTableHeader();
				populateTableData(userTaskList);
				update();
			}
			
		});
	}

	private void populateTableData(UserTaskList userTaskList) {
		List<UserTask> userTasks = userTaskList.getUserTasks();
		
		Collections.sort(userTasks, new Comparator<UserTask>() {
			@Override
			public int compare(UserTask t1, UserTask t2) {
				return t1.getName().compareTo(t2.getName());
			}
		});

		int dataRow = 1;
		for(UserTask userTask : userTasks) {
			if(userTask.hasEnded())
				continue;

			int colIndex = 0;
			String historyToken = "Workflow.User Tasks.Perform?id="
				+ userTask.getWorkOrderID() + ":" + userTask.getTaskInstanceId();
			HyperlinkWidget link = new HyperlinkWidget(userTask.getName(), historyToken);
			link.setStyleName("portlet-userTasks-link");
			link.setTitle(userTask.getName());
			table.setWidget(dataRow, colIndex, link);
			table.getCellFormatter().setStyleName(dataRow, colIndex, "portlet-table-text");
			populateTableDynamicData(dataRow, colIndex, userTask);
			dataRow++;
		}
	}
	
	private void populateTableDynamicData(int rowIndex, int colIndex, UserTask userTask) {
		for (UiColumn column : uiTable.getColumns()) {
			if (colIndex + 1 > MAX_NUMBER_OF_COLUMNS) {
				return;
			}
			colIndex++;
			String text = "";
			Object value = null;
			if (column.getParameter() != null && !column.getParameter().isEmpty()) {
				value = userTask.getProcessVariable(column.getParameter());
			} else {
				value = userTask.getTaskProperty(column.getMethodName());
			}
			
			if (value != null) {
				text = value.toString();
			} 
			LabelWidget label = new LabelWidget(text, false);
			label.setWidth("100%");
			label.setTitle(text);
			label.setStyleName("portletEllipsedCell");
			table.getCellFormatter().setStyleName(rowIndex, colIndex, "portlet-table-text");
			table.getCellFormatter().setWidth(rowIndex, colIndex, column.getWidth());
			table.setWidget(rowIndex, colIndex, label);
		}
	}
	
	/**
	 * 		Updates the portlet title 
	 * 
	 * @param userTaskList
	 */
	private void updateTitle(UserTaskList userTaskList) {
		setTitle(constants.userTaskTitle()
				+ " (" + userTaskList.getNumUserTasks() + "/"
				+ userTaskList.getTotalNumUserTasks() + ")");
	}
	
	private void clearTableData() {
		table.removeAllRows();
	}

	private void prepareTableHeader() {
		// add node name column header first.
		// node name column is fixed.
		int colIndex = 0;
		LabelWidget label = new LabelWidget(constants.nodeNameColumn(), false);
		label.setWidth("100%");
		label.setTitle(constants.nodeNameColumn());
		label.setStyleName("portletEllipsedCell");
		table.setWidget(0, colIndex, label);
		table.getCellFormatter().setStyleName(0, 0, "portlet-tableHeader");

		List<UiColumn> columns = uiTable.getColumns();
		if (columns == null)
			return;
		
		for (int i = 0; i < columns.size(); i++) {
			if (colIndex + 1 > MAX_NUMBER_OF_COLUMNS) {
				return;
			}
			colIndex++;
			UiColumn col = columns.get(i);

			String colHeader = col.getColumnName();
			label = new LabelWidget(colHeader, false);
			label.setWidth("100%");
			label.setTitle(colHeader);
			label.setStyleName("portletEllipsedCell");
			table.getCellFormatter().setStyleName(0, colIndex, "portlet-tableHeader");
			table.getCellFormatter().setWidth(0, colIndex, col.getWidth());
			table.getColumnFormatter().setWidth(colIndex, col.getWidth());
			table.setWidget(0, colIndex, label);
		}
	}

	/*
	 * Event Handling
	 */
	private final class EventHandler implements DocumentAttachHandler, DocumentDetachHandler {
        @Override
        public void onAttach(DocumentAttachEvent event) {
    		refreshTimer.scheduleRepeating(refreshPeriod);
    		reload();
        }

        @Override
        public void onDetach(DocumentDetachEvent event) {
    		refreshTimer.cancel();
        }
	}
}
