/**
 * WorkflowComponent.java
 * Created on Jan 21, 2009
 * (C) Copyright TANDBERG Television Ltd.
 */
package com.tandbergtv.cms.ui.workflow.client;

import static com.tandbergtv.cms.ui.workflow.client.Permission.HOME;
import static com.tandbergtv.cms.ui.workflow.client.Permission.MONITORING_VIEW;
import static com.tandbergtv.cms.ui.workflow.client.Permission.PERFORM_OTHERS_TASKS;
import static com.tandbergtv.cms.ui.workflow.client.Permission.RESOURCES_VIEW;
import static com.tandbergtv.cms.ui.workflow.client.Permission.TASKS_ASSIGN;
import static com.tandbergtv.cms.ui.workflow.client.Permission.TASKS_PERFORM;
import static com.tandbergtv.cms.ui.workflow.client.Permission.WORK_ORDERS_MODIFY;
import static com.tandbergtv.cms.ui.workflow.client.Permission.WORK_ORDERS_VIEW;

import java.util.ArrayList;
import java.util.List;

import com.google.gwt.core.client.GWT;
import com.google.gwt.user.client.Random;
import com.tandbergtv.cms.portal.ui.title.client.model.uiservice.ITitleProgressService;
import com.tandbergtv.cms.ui.workflow.client.portlet.ActiveResourcesPortlet;
import com.tandbergtv.cms.ui.workflow.client.portlet.QueuedUserTasksPortlet;
import com.tandbergtv.neptune.widgettoolkit.client.application.ClientAuthorizationManager;
import com.tandbergtv.neptune.widgettoolkit.client.application.PortletFactory;
import com.tandbergtv.neptune.widgettoolkit.client.application.Service;
import com.tandbergtv.neptune.widgettoolkit.client.component.Component;
import com.tandbergtv.neptune.widgettoolkit.client.component.ComponentBase;
import com.tandbergtv.neptune.widgettoolkit.client.menu.GroupMenuItem;
import com.tandbergtv.neptune.widgettoolkit.client.menu.MenuItemBase;
import com.tandbergtv.neptune.widgettoolkit.client.menu.WidgetMenuItem;
import com.tandbergtv.neptune.widgettoolkit.client.security.NeptuneSecurity;
import com.tandbergtv.neptune.widgettoolkit.client.widget.basic.AutosizeFrame;
import com.tandbergtv.neptune.widgettoolkit.client.widget.container.Portlet;


/**
 * @author Vlada Jakobac
 *
 */
public class WorkflowComponent extends ComponentBase {
	private static final String THIS_COMPONENT_NAME = "WatchPoint";
		
	private final PortletConstants constants = GWT.create( PortletConstants.class );
	
	public List<PortletFactory> listPortlets(final NeptuneSecurity security) {
		if (!isAuthorized(HOME))
			return null;

		List<PortletFactory> portletFactories = new ArrayList<PortletFactory>();
		
		//get configured refresh time for portlets
		int confRefreshTimeMillis = -1;
		try {
			confRefreshTimeMillis = Integer.parseInt(info.get("workflow.portlets.refreshTime"));
		} catch(Exception e) {}
		
		final int refreshTimeMillis = (confRefreshTimeMillis != -1) ? confRefreshTimeMillis : 10000;
		
		//user tasks
		if (isAuthorized(TASKS_PERFORM))
			portletFactories.add(new WorkflowPortletFactory(new UserTaskListPortlet(refreshTimeMillis), this));

		if (isAuthorized(WORK_ORDERS_VIEW)) {
			//status
			portletFactories.add(new WorkflowPortletFactory(new WOStatusPortlet(refreshTimeMillis), this));
			
			portletFactories.add(new WorkflowPortletFactory(new ActiveResourcesPortlet(refreshTimeMillis), this));
		}

		if (isAuthorized(RESOURCES_VIEW)) {
			//queues
			portletFactories.add(new WorkflowPortletFactory(new ResourceGroupQueueListPortlet(refreshTimeMillis), this));
			
			//resources
			portletFactories.add(new WorkflowPortletFactory(new ResourceListPortlet(refreshTimeMillis), this));
		}

		if (isAuthorized(TASKS_ASSIGN)) {
			// task assignment
			portletFactories.add(new WorkflowPortletFactory(new QueuedUserTasksPortlet(refreshTimeMillis), this));
		}

		return portletFactories;
	}

	@SuppressWarnings("unchecked")
	public <E extends Service> List<E> listServices(Class<E> clazz, final NeptuneSecurity security) {
		if(clazz.getName().equalsIgnoreCase(ITitleProgressService.class.getName())) {
			List<ITitleProgressService> tpServices = new ArrayList<ITitleProgressService>();
			tpServices.add(new ITitleProgressService() {
				@Override
				public Hyperlink getLink(String componentName,
						String entityName, String sourceId, String titleRevision) {
					if(THIS_COMPONENT_NAME.equals(componentName)) {
						String historyToken = null;
						if (isAuthorized(WORK_ORDERS_VIEW)) {
							// sourceId is of the form processId|tokenId or requestKey
							int index = (sourceId != null) ? sourceId.indexOf('|') : -1;
							if (index < 0) {
								historyToken = "Workflow.Work Orders.Search.RequestKey=" + sourceId;
							} else {
								String processId = sourceId.substring(0, index);
								historyToken = "Workflow.Work Orders.Search.Id=" + processId;
							}
						}
						return new Hyperlink(componentName + " - " + entityName, historyToken);
					}
					else
						return null;
				}});
			return (List<E>) tpServices;
		}
		
		return null;
	}
	
	public List<MenuItemBase> listTabs(NeptuneSecurity security) {
		List<MenuItemBase> tabs = new ArrayList<MenuItemBase>();
		String menuName;
		
		/* Workflow group menu */
		List<MenuItemBase> workflowGroup = new ArrayList<MenuItemBase>();
		
		/* Dashboard */
		if (isAuthorized(HOME)) {
			menuName = "Dashboard";
			final String url = "/workflow/dashboardview.do?method=getDashBoard";			
			addPanelToGroup(workflowGroup, url, menuName);		
		}
				
		/* Work Orders group */
		if (isAuthorized(WORK_ORDERS_VIEW)) {
			final AutosizeFrame panel = createFrame();
				
			WidgetMenuItem.AnchorChangeListener anchorChangeListener = new WidgetMenuItem.AnchorChangeListener() {
				public void anchorChanged(String anchor) {
					if (anchor.startsWith("WorkOrderNos")) {
						String url = "/workflow/workOrderDetailsAction.do?method=getWorkOrdersBySearch&sourcePage=activity&pid="
								+ anchor.substring("WorkOrderNos".length() + 1);
						showFrame(panel, url);
					} else if (anchor.startsWith("Id")) {
						String url = "/workflow/workOrderTask.do?method=getWorkOrderTask&workOrderNo="
								+ anchor.substring("Id".length() + 1);
						showFrame(panel, url);
					} else if (anchor.startsWith("RequestKey")) {
						String url = "/workflow/workOrderTask.do?method=getWorkOrderTask&requestKey="
					        + anchor.substring("RequestKey".length() + 1);
						showFrame(panel, url);
					} else if (anchor.startsWith("Status")) {
						String url = "/workflow/workOrderDetailsAction.do?method=getWorkOrdersBySearch&sourcePage=dashBoard&status="
							+ anchor.substring("Status".length() + 1);
						showFrame(panel, url);
					//Old links will have an extra token in the anchor due to old menus which had links under 3rd level "Search" menu
					} else if (anchor.startsWith("Search.WorkOrderNos")) {
						String url = "/workflow/workOrderDetailsAction.do?method=getWorkOrdersBySearch&sourcePage=activity&pid="
								+ anchor.substring("Search.WorkOrderNos".length() + 1);
						showFrame(panel, url);
					} else if (anchor.startsWith("Search.Id")) {
						String url = "/workflow/workOrderTask.do?method=getWorkOrderTask&workOrderNo="
								+ anchor.substring("Search.Id".length() + 1);
						showFrame(panel, url);
					} else if (anchor.startsWith("Search.RequestKey")) {
						String url = "/workflow/workOrderTask.do?method=getWorkOrderTask&requestKey="
					        + anchor.substring("Search.RequestKey".length() + 1);
						showFrame(panel, url);
					} else if (anchor.startsWith("Search.Status")) {
						String url = "/workflow/workOrderDetailsAction.do?method=getWorkOrdersBySearch&sourcePage=dashBoard&status="
							+ anchor.substring("Search.Status".length() + 1);
						showFrame(panel, url);
					} else {
						String url = "/workflow/workOrderSearch.do?method=getWorkOrderSearchPage";
						showFrame(panel, url);
					}
				}
			};
			

			final WidgetMenuItem woSearchViewPanel = new WidgetMenuItem("Work Orders", panel, anchorChangeListener);
			workflowGroup.add(woSearchViewPanel);	
			
			if (isAuthorized(WORK_ORDERS_MODIFY)) {
				menuName = "Pause / Resume all";
				final String url = "/workflow/pauseResumeAction.do?method=goToLogin";
				addPanelToGroup(workflowGroup, url, menuName);			
			}
		}
		
		/* Resources group */
		if (isAuthorized(RESOURCES_VIEW)) {
			final AutosizeFrame panel = createFrame();

			WidgetMenuItem.AnchorChangeListener anchorChangeListener = new WidgetMenuItem.AnchorChangeListener() {
				public void anchorChanged(String anchor) {
					if (anchor.startsWith("Modify")) {
						String url = "/workflow/resourceManagement.do?method=getResourceDetails&"
								+ anchor.substring(7);
						showFrame(panel, url);
					} else if (anchor.startsWith("Queue")) {
						String url = "/workflow/resourceGroupQueueManagement.do?method=getResourceGroupQueue&"
								+ anchor.substring(6);
						showFrame(panel, url);
					} else {
						String url = "/workflow/resourcesListManagement.do?method=getResourcesList";
						showFrame(panel, url);
					}
				}
			};
			
			final WidgetMenuItem resourcesViewPanel = new WidgetMenuItem("Resources", panel, anchorChangeListener);
			workflowGroup.add(resourcesViewPanel);
			
		}
		
		/* Monitoring */
		List<MenuItemBase> monitoringGroup = new ArrayList<MenuItemBase>();
		workflowGroup.add(new GroupMenuItem("Monitoring", monitoringGroup));
		if (isAuthorized(MONITORING_VIEW)){			
			menuName = "Alerts";
			String url = "/workflow/toModule.do?prefix=/sysmonui&page=/checkAlertServerStatus.do?method=toListAlerts";
			addPanelToGroup(monitoringGroup, url, menuName);				

			menuName = "Alert Patterns";
			url = "/workflow/toModule.do?prefix=/sysmonui&page=/checkAlertServerStatus.do?method=toListAlertPatterns";
			addPanelToGroup(monitoringGroup, url, menuName);

			menuName = "Alert History";
			url = "/workflow/toModule.do?prefix=/sysmonui&page=/checkAlertServerStatus.do?method=toListAlertHistory";
			addPanelToGroup(monitoringGroup, url, menuName);
			
			menuName = "Repository Browser";
			final String reposBrowsUrl = "/workflow/repositoryBrowsing.do?method=getRepository";
			addPanelToGroup(monitoringGroup, reposBrowsUrl, menuName);
			
			menuName = "Storage";
			final String storageURL = "/workflow/listStorage.do";
			addPanelToGroup(monitoringGroup, storageURL, menuName);
			
			addPanelToGroup(monitoringGroup, "/workflow/responsetime.do?method=getResponseTimes", "Response Time Report");
		}
		
		/* User Tasks */
		if (isAuthorized(TASKS_PERFORM) || isAuthorized(PERFORM_OTHERS_TASKS)){
			final AutosizeFrame panel = createFrame();

			WidgetMenuItem.AnchorChangeListener anchorChangeListener = new WidgetMenuItem.AnchorChangeListener() {
				public void anchorChanged(String anchor) {
					if (anchor.startsWith("Perform")) {
						String url = "/workflow/wfstask.do?method=selectTaskInstance&task="
										+ anchor.substring("Perform?id=".length());
						showFrame(panel, url);
					} else if (anchor.startsWith("Assign")) {
						String params = "";
						if (anchor.length() > "Assign".length() + 1) {
							params = "&" + anchor.substring("Assign".length() + 1);
						}
						showFrame(panel, "/workflow/queuedUserTasks.do?method=getTasks" + params);
					} else {
						String url = "/workflow/userTasksAction.do?method=getUserTasks";
						showFrame(panel, url);
					}
				}
			};
			workflowGroup.add(new WidgetMenuItem("User Tasks", panel, anchorChangeListener));					
		}		
		
		/* Always create the group menu item */
		tabs.add(new GroupMenuItem("Workflow", workflowGroup));		
		
		return tabs;
		
	}

	/**
	 * Adds a SimplePanel object to the specified group. The SimplePanel object is created with AnchorChangeListener
	 * (which resets the url every time there is a click on the menu)
	 * because of the behavior in the InternetExplorer, where the frame does not get loaded if there is no anchor listener  
	 * 
	 * @param group
	 * @param url
	 * @param menuName
	 */
	private void addPanelToGroup(List<MenuItemBase> group,
			final String url, String menuName) {
		
		final AutosizeFrame panel = createFrame();
		
		WidgetMenuItem.AnchorChangeListener anchorChangeListener = new WidgetMenuItem.AnchorChangeListener() {
			public void anchorChanged(String anchor) {
				showFrame(panel, url);
			}
		};
		final WidgetMenuItem menuItem = new WidgetMenuItem(menuName, panel, anchorChangeListener);
		group.add(menuItem);
	}

	
	/* Build the frame so it can be referenced from the anchor change listener */
	private AutosizeFrame createFrame() {
		AutosizeFrame frame = new AutosizeFrame ();
		return frame;
	}
	
	/* Show a frame in the given container using the given URL */
	private void showFrame(AutosizeFrame frame, String frameURL) {
		String randomizer = "&random=" + getRandom();
		
		if (frameURL.contains("?"))
			frameURL = frameURL + randomizer;
		else
			frameURL = frameURL + "?" + randomizer;
		
		frame.setVisible(true);
		frame.setUrl(frameURL);
	}

	
	/**
	 * Generates a random string
	 * 
	 * @return
	 */
	private String getRandom() {
		return String.valueOf(Random.nextDouble());
	}
	
	private boolean isAuthorized(Permission permission) {
		return ClientAuthorizationManager.isAuthorized(permission.toString());
	}
	
	/*
	 * Portlet Factory that maintains the single portlet instance created
	 */
	private static final class WorkflowPortletFactory implements PortletFactory {
		private final Portlet portlet;
		private WorkflowComponent workflowComponent;

		public WorkflowPortletFactory(Portlet portlet, WorkflowComponent workflowComponent) {
			this.portlet = portlet;
			this.workflowComponent = workflowComponent;
		}

		@Override
		public Portlet getInstance() {
			return this.portlet;
		}

		@Override
		public void release(Portlet widget) {
		}

                @Override
                public Component getComponent() {
                    return workflowComponent;
                }
	}

    @Override
    public String getName() {
        return "WORKFLOW";
    }

    @Override
    public String getDisplayName() {
        return constants.componentDisplayName();
    }
}
