/**
 * 
 */
package com.tandbergtv.cms.ui.workflow.client.portlet;

import static com.tandbergtv.cms.ui.workflow.client.Permission.RESOURCES_VIEW;

import java.io.Serializable;
import java.util.ArrayList;
import java.util.List;

import com.google.gwt.core.client.GWT;
import com.google.gwt.user.client.Timer;
import com.google.gwt.user.client.ui.FlexTable;
import com.google.gwt.user.client.ui.HTMLTable.CellFormatter;
import com.google.gwt.user.client.ui.Widget;
import com.tandbergtv.cms.ui.workflow.client.IListResources;
import com.tandbergtv.cms.ui.workflow.client.IListResourcesAsync;
import com.tandbergtv.cms.ui.workflow.client.PortletConstants;
import com.tandbergtv.cms.ui.workflow.client.model.ActiveResource;
import com.tandbergtv.cms.ui.workflow.client.model.ActiveResources;
import com.tandbergtv.neptune.widgettoolkit.client.application.ClientAuthorizationManager;
import com.tandbergtv.neptune.widgettoolkit.client.remote.NeptuneAsyncCallback;
import com.tandbergtv.neptune.widgettoolkit.client.widget.basic.HyperlinkWidget;
import com.tandbergtv.neptune.widgettoolkit.client.widget.basic.LabelWidget;
import com.tandbergtv.neptune.widgettoolkit.client.widget.container.FlexTableContainer;
import com.tandbergtv.neptune.widgettoolkit.client.widget.container.Portlet;
import com.tandbergtv.neptune.widgettoolkit.client.widget.event.DocumentAttachEvent;
import com.tandbergtv.neptune.widgettoolkit.client.widget.event.DocumentAttachHandler;
import com.tandbergtv.neptune.widgettoolkit.client.widget.event.DocumentDetachEvent;
import com.tandbergtv.neptune.widgettoolkit.client.widget.event.DocumentDetachHandler;

/**
 * Displays currently used resources
 * 
 * @author Sahil Verma
 */
public class ActiveResourcesPortlet extends Portlet {
	
	private static final String TABLE_STYLE = "portletTable";

	private static final String TEXT_STYLE = "portlet-table-text";

	private static final String HEADER_STYLE = "portlet-tableHeader";

	private IListResourcesAsync service;
	
	private PortletConstants constants;
	
	private Timer timer;
	
	private FlexTable table;
	
	private int periodmsec;
	
	private List<PortletTableColumn> columns;
	
	/**
	 * Creates the portlet
	 * 
	 * @param security
	 * @param period refresh interval
	 */
	public ActiveResourcesPortlet(int period) {
	        super("ACTIVE_RESOURCES");
		this.service = GWT.create(IListResources.class);
		constants = GWT.create(PortletConstants.class);
		
		setTitle(constants.activeProcesses());
		setDisplayName(constants.activeProcesses());
		
		createTable();
		
		this.periodmsec = period;
		this.timer = new Timer() {
			@Override
			public void run() {
				reload();
			}
		};

		EventHandler handler = new EventHandler();
		addDocumentAttachHandler(handler);
		addDocumentDetachHandler(handler);
	}
	
	/**
	 * Sets up the table widget and its columns
	 */
	private void createTable() {
		this.table = new FlexTableContainer();
		table.setStyleName(TABLE_STYLE);
		add(table);
		
		columns = new ArrayList<ActiveResourcesPortlet.PortletTableColumn>();
		columns.add(new PortletTableColumn(0, "Process Name"));
		columns.add(new PortletTableColumn(1, "Task"));
		columns.add(new PortletTableColumn(2, "Title Brief"));
		columns.add(new PortletTableColumn(3, "Progress"));
		columns.add(new PortletTableColumn(4, "Resource"));
	}
	
	/**
	 * Polls the server, callback is processed asynchronously
	 */
	private void reload() {
		service.getActiveResources(new NeptuneAsyncCallback<ActiveResources>() {
			@Override
			public void onNeptuneSuccess(ActiveResources resources) {
				doReload(resources);
				update();
			}
			
			@Override
			public void onNeptuneFailure(Throwable caught) {}
		});
	}
	
	/**
	 * Adds the data to the table
	 * 
	 * @param resources
	 */
	private void doReload(ActiveResources resources) {
		table.removeAllRows();
		
		setTitle(resources);
		
		List<ActiveResource> list = resources.getResources();
		
		if (list.isEmpty())
			return; // Nothing to do
		
		createTableHeader(table);
		
		int row = 1;
		
		for (ActiveResource resource : list)
			addRow(resource, row++);
	}
	
	/**
	 * Adds a row at an index with the specified data
	 * 
	 * @param resource
	 * @param row
	 */
	private void addRow(ActiveResource resource, int row) {
		CellFormatter formatter = table.getCellFormatter();
		
		for (PortletTableColumn column : columns) {
			column.resource = resource;
			formatter.setStyleName(row, column.index, column.style);
			table.setWidget(row, column.index, column.getWidget());
		}
	}
	
	private void createTableHeader(FlexTable table) {
		CellFormatter formatter = table.getCellFormatter();
		
		for (PortletTableColumn column : columns) {
			table.setWidget(0, column.index, new LabelWidget(column.header));
			formatter.setStyleName(0, column.index, column.headerstyle);
		}
	}
	
	private void setTitle(ActiveResources resources) {
		List<ActiveResource> list = resources.getResources();
		
		setTitle(constants.activeProcesses() + " (" + list.size() + "/" + resources.getTotal() + ")");
	}
	
	private final class EventHandler implements DocumentAttachHandler, DocumentDetachHandler {
		@Override
		public void onAttach(DocumentAttachEvent event) {
			timer.scheduleRepeating(periodmsec);
			reload();
		}

		@Override
		public void onDetach(DocumentDetachEvent event) {
			timer.cancel();
		}
	}
	
	private final class PortletTableColumn implements Serializable {
		private static final long serialVersionUID = 1L;
		int index;
		String header;
		String headerstyle = HEADER_STYLE;
		String style = TEXT_STYLE;
		ActiveResource resource;
		
		PortletTableColumn(int index, String header) {
			super();
			this.index = index;
			this.header = header;
		}

		public Widget getWidget() {
			String data = getData();
			
			switch (index) {
			case 0: return new HyperlinkWidget(data, "Workflow.Work Orders.Search.Id=" + resource.getPid());
			case 1: return new LabelWidget(data);
			case 2: 
				if (ClientAuthorizationManager.isAuthorized("AssetManagement_View"))
					return new HyperlinkWidget(data, "Content.Search.Id=" + resource.getTitleId());
				return new LabelWidget(data);
			case 3: return new LabelWidget(data + "%");
			case 4:
				if (ClientAuthorizationManager.isAuthorized(RESOURCES_VIEW.toString()))
					return new HyperlinkWidget(data, "Workflow.Resources.Modify?id=" + resource.getId());
				return new LabelWidget(data);
			default: return new LabelWidget();
			}
		}
		
		private String getData() {
			switch (index) {
			case 0: return resource.getTemplate(); 
			case 1: return resource.getNode();
			case 2: return resource.getTitleDisplayName();
			case 3: return resource.getPercentComplete();
			case 4: return resource.getResource();
			default: return ""; // WTH?
			}
		}
	}
}
