/*
 * Created on May 31, 2011
 * 
 * (C) Ericsson Television Inc.
 */
package com.tandbergtv.cms.ui.workflow.client.portlet;

import com.google.gwt.core.client.GWT;
import com.google.gwt.user.client.Timer;
import com.google.gwt.user.client.ui.Widget;
import com.tandbergtv.cms.ui.workflow.client.IQueuedUserTasksPortletService;
import com.tandbergtv.cms.ui.workflow.client.IQueuedUserTasksPortletServiceAsync;
import com.tandbergtv.cms.ui.workflow.client.PortletConstants;
import com.tandbergtv.cms.ui.workflow.client.PortletMessages;
import com.tandbergtv.cms.ui.workflow.client.model.QueuedUserTasksGroupData;
import com.tandbergtv.cms.ui.workflow.client.model.QueuedUserTasksPortletData;
import com.tandbergtv.neptune.widgettoolkit.client.remote.NeptuneAsyncCallback;
import com.tandbergtv.neptune.widgettoolkit.client.widget.basic.HyperlinkWidget;
import com.tandbergtv.neptune.widgettoolkit.client.widget.basic.LabelWidget;
import com.tandbergtv.neptune.widgettoolkit.client.widget.container.FlexTableContainer;
import com.tandbergtv.neptune.widgettoolkit.client.widget.container.Portlet;
import com.tandbergtv.neptune.widgettoolkit.client.widget.container.SimpleContainer;
import com.tandbergtv.neptune.widgettoolkit.client.widget.event.DocumentAttachEvent;
import com.tandbergtv.neptune.widgettoolkit.client.widget.event.DocumentAttachHandler;
import com.tandbergtv.neptune.widgettoolkit.client.widget.event.DocumentDetachEvent;
import com.tandbergtv.neptune.widgettoolkit.client.widget.event.DocumentDetachHandler;

/**
 * Portlet widget for queued user tasks
 * 
 * @author Vijay Silva
 */
public class QueuedUserTasksPortlet extends Portlet {

	/** The portlet ID */
	public static final String PORTLET_ID = "WFS.QueuedUserTasksPortlet";
	private static final long MIN_REFRESH_TIME = 5000L;

	private final PortletConstants constants = GWT.create(PortletConstants.class);
	private final PortletMessages messages = GWT.create(PortletMessages.class);
	private final IQueuedUserTasksPortletServiceAsync service = GWT
			.create(IQueuedUserTasksPortletService.class);

	private final Timer refreshTimer;
	private int refreshTime = 30000;
	private SimpleContainer mainContainer;
	private FlexTableContainer table;
	private boolean showError = true;

	private static final String MAIN_CONTAINER_STYLE = "workflow-QueuedUserTasksPortlet";
	private static final String LOADING_LABEL_STYLE = "workflow-QueuedUserTasksPortlet-loadingLabel";
	private static final String TABLE_STYLE = "workflow-QueuedUserTasksPortlet-dataTable";
	private static final String GROUP_NAME_CELL_STYLE = "workflow-QueuedUserTasksPortlet-groupNameCell";
	private static final String GROUP_COUNT_CELL_STYLE = "workflow-QueuedUserTasksPortlet-groupCountCell";
	private static final String GROUP_NAME_LABEL_STYLE = "workflow-QueuedUserTasksPortlet-groupNameLabel";
	private static final String GROUP_COUNT_LABEL_STYLE = "workflow-QueuedUserTasksPortlet-groupCountLabel";
	private static final String NO_DATA_LABEL_STYLE = "workflow-QueuedUserTasksPortlet-noDataLabel";
	private static final String DATA_ERROR_STYLE = "workflow-QueuedUserTasksPortlet-errorLabel";

	/**
	 * @param refreshTime The time in milliseconds to refresh the contents of the portlet
	 */
	public QueuedUserTasksPortlet(int refreshTime) {
		super(PORTLET_ID);
		if (refreshTime >= MIN_REFRESH_TIME)
			this.refreshTime = refreshTime;
		String displayName = constants.queuedUserTasksPortletTitle();
		this.setTitle(displayName);
		this.setDisplayName(displayName);

		this.refreshTimer = new Timer() {
			@Override
			public void run() {
				reload();
			}
		};

		initialize();

		EventHandler handler = new EventHandler();
		addDocumentAttachHandler(handler);
		addDocumentDetachHandler(handler);
	}

	private void initialize() {
		/* Set the initial loading message */
		mainContainer = new SimpleContainer();
		mainContainer.addStyleName(MAIN_CONTAINER_STYLE);
		this.add(mainContainer);

		/* Add an initial label indicating data is loading */
		LabelWidget label = new LabelWidget(constants.queuedUserTasksPortletLoading());
		label.addStyleName(LOADING_LABEL_STYLE);
		mainContainer.setWidget(label);

		table = new FlexTableContainer();
		table.addStyleName(TABLE_STYLE);
	}

	/*
	 * Reload the portlet view
	 */
	private void reload() {
		service.getData(new NeptuneAsyncCallback<QueuedUserTasksPortletData>() {
			@Override
			public void onNeptuneFailure(Throwable caught) {
				handleGetDataFailure(caught);
			}

			@Override
			public void onNeptuneSuccess(QueuedUserTasksPortletData result) {
				handleGetDataSuccess(result);
				update();
			}
		});
	}

	/*
	 * Handle the failure of getting the portlet data
	 */
	private void handleGetDataFailure(Throwable error) {
		if (showError) {
			super.setTitle(constants.queuedUserTasksPortletTitle());
			String text = messages.queuedUserTasksPortletGetDataError(error.getLocalizedMessage());
			LabelWidget errorLabel = new LabelWidget(text);
			errorLabel.addStyleName(DATA_ERROR_STYLE);
			this.mainContainer.setWidget(errorLabel);
		}
	}

	/*
	 * Handle the arrival of portlet data
	 */
	private void handleGetDataSuccess(QueuedUserTasksPortletData data) {
		/* Update the title */
		super.setTitle(messages.queuedUserTasksPortletTitleWithCount(data.getTotalTaskCount()));

		/* Update the data panel */
		table.removeAllRows();
		int row = 0;
		for (QueuedUserTasksGroupData groupData : data.getGroupData()) {
			table.setWidget(row, 0, createNameLabel(groupData));
			table.getCellFormatter().addStyleName(row, 0, GROUP_NAME_CELL_STYLE);
			table.setWidget(row, 1, createCountLabel(groupData));
			table.getCellFormatter().addStyleName(row, 1, GROUP_COUNT_CELL_STYLE);
			row++;
		}

		/* Update the displayed widget */
		showError = false;
		Widget displayWidget = (row == 0) ? createNoDataLabel() : table;
		mainContainer.setWidget(displayWidget);
	}

	/* Create the widget for showing that there are no queued tasks */
	private Widget createNoDataLabel() {
		LabelWidget widget = new LabelWidget(constants.queuedUserTasksPortletNoData());
		widget.addStyleName(NO_DATA_LABEL_STYLE);
		return widget;
	}

	/*
	 * Create the name widget for the group data
	 */
	private Widget createNameLabel(QueuedUserTasksGroupData data) {
		String token = "Workflow.User Tasks.Assign?selectedResourceGroupId=" + data.getResourceGroupId();
		HyperlinkWidget widget = new HyperlinkWidget(data.getResourceGroupName(), token);
		widget.addStyleName(GROUP_NAME_LABEL_STYLE);
		return widget;
	}

	/*
	 * Create the count widget for the group data
	 */
	private Widget createCountLabel(QueuedUserTasksGroupData data) {
		LabelWidget widget = new LabelWidget(Integer.toString(data.getTaskCount()));
		widget.addStyleName(GROUP_COUNT_LABEL_STYLE);
		return widget;
	}

	/*
	 * Event Handling
	 */
	private final class EventHandler implements DocumentAttachHandler, DocumentDetachHandler {
		@Override
		public void onAttach(DocumentAttachEvent event) {
			refreshTimer.scheduleRepeating(refreshTime);
			reload();
		}

		@Override
		public void onDetach(DocumentDetachEvent event) {
			refreshTimer.cancel();
		}
	}
}
