/*
 * Copyright 2009 Sikirulai Braheem
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations under
 * the License.
 */
package com.bramosystems.oss.player.core.client.impl;

import com.bramosystems.oss.player.core.event.client.PlayerStateEvent;
import com.bramosystems.oss.player.core.event.client.HasMediaStateHandlers;
import com.bramosystems.oss.player.core.event.client.PlayStateEvent;
import com.bramosystems.oss.player.core.event.client.MediaInfoEvent;
import com.bramosystems.oss.player.core.event.client.LoadingProgressEvent;
import com.bramosystems.oss.player.core.event.client.DebugEvent;
import com.bramosystems.oss.player.core.client.MediaInfo;
import com.google.gwt.dom.client.Document;
import com.google.gwt.dom.client.Element;
import com.google.gwt.dom.client.NativeEvent;
import com.google.gwt.event.dom.client.DomEvent;
import com.google.gwt.user.client.Timer;
import java.util.HashMap;
import java.util.Iterator;

/**
 * This class handles the events generated by the embedded Windows Media Player.
 * It maps the WMP states into the corresponding event in the API.
 *
 * @author Sikirulai Braheem
 */
public class WMPStateManager {

    protected HashMap<String, StateManager> cache;

    WMPStateManager() {
        cache = new HashMap<String, StateManager>();
        initGlobalEventListeners(this);
    }

    public void init(WinMediaPlayerImpl player, HasMediaStateHandlers handler, boolean resizing) {
        StateManager sm = new StateManager(player, handler, resizing);
        cache.put(player.getPlayerId(), sm);
        sm.checkPlayState();
    }

    /**
     * Provided for deferred binding enhancements. Resize fix required for
     * non-IE browsers only
     *
     * @return
     */
    public boolean shouldRunResizeQuickFix() {
        return true;
    }

    public final boolean isPlayerStateManaged(String playerId) {
        return cache.containsKey(playerId);
    }

    public void close(String playerId) {
        cache.remove(playerId);
    }

    public void stop(String playerId) {
        // do nothing, workaround for webkit implementation...
    }

    @SuppressWarnings("unused")
    private void firePlayStateChanged() {
        Iterator<String> keys = cache.keySet().iterator();
        while (keys.hasNext()) {
            cache.get(keys.next()).checkPlayState();
        }
    }

    @SuppressWarnings("unused")
    private void fireError() {
        Iterator<String> keys = cache.keySet().iterator();
        while (keys.hasNext()) {
            String id = keys.next();
            cache.get(id).checkError();
        }
    }

    @SuppressWarnings("unused")
    private void fireCMEvents(int type, int button, int shiftState, double fX, double fY) {
        Iterator<String> keys = cache.keySet().iterator();
        while (keys.hasNext()) {
            String id = keys.next();
            cache.get(id).doClickMouseEvents(type, button, shiftState, fX, fY);
        }
    }

    @SuppressWarnings("unused")
    private void fireBuffering(boolean buffering) {
        Iterator<String> keys = cache.keySet().iterator();
        while (keys.hasNext()) {
            cache.get(keys.next()).doBuffering(buffering);
        }
    }

    protected native void initGlobalEventListeners(WMPStateManager impl) /*-{
    $wnd.OnDSPlayStateChangeEvt = function(NewState) {
    impl.@com.bramosystems.oss.player.core.client.impl.WMPStateManager::firePlayStateChanged()();
    }
    $wnd.OnDSErrorEvt = function() {
    impl.@com.bramosystems.oss.player.core.client.impl.WMPStateManager::fireError()();
    }
    $wnd.OnDSBufferingEvt = function(Start) {
    impl.@com.bramosystems.oss.player.core.client.impl.WMPStateManager::fireBuffering(Z)(Start);
    }
    $wnd.OnDSMouseDownEvt = function(nButton,nShiftState,fX,fY) {
    impl.@com.bramosystems.oss.player.core.client.impl.WMPStateManager::fireCMEvents(IIIDD)(1,nButton,nShiftState,fX,fY);
    }
    $wnd.OnDSMouseUpEvt = function(nButton,nShiftState,fX,fY) {
    impl.@com.bramosystems.oss.player.core.client.impl.WMPStateManager::fireCMEvents(IIIDD)(2,nButton,nShiftState,fX,fY);
    }
    $wnd.OnDSMouseMoveEvt = function(nButton,nShiftState,fX,fY) {
    impl.@com.bramosystems.oss.player.core.client.impl.WMPStateManager::fireCMEvents(IIIDD)(3,nButton,nShiftState,fX,fY);
    }
    $wnd.OnDSClickEvt = function(nButton,nShiftState,fX,fY) {
    impl.@com.bramosystems.oss.player.core.client.impl.WMPStateManager::fireCMEvents(IIIDD)(10,nButton,nShiftState,fX,fY);
    }
    $wnd.OnDSDoubleClickEvt = function(nButton,nShiftState,fX,fY) {
    impl.@com.bramosystems.oss.player.core.client.impl.WMPStateManager::fireCMEvents(IIIDD)(11,nButton,nShiftState,fX,fY);
    }
    $wnd.OnDSDblClickEvt = function(nButton,nShiftState,fX,fY) {
    impl.@com.bramosystems.oss.player.core.client.impl.WMPStateManager::fireCMEvents(IIIDD)(11,nButton,nShiftState,fX,fY);
    }
    $wnd.OnDSKeyDownEvt = function(nKeyCode,nShiftState) {
    impl.@com.bramosystems.oss.player.core.client.impl.WMPStateManager::fireCMEvents(IIIDD)(20,nKeyCode,nShiftState,0,0);
    }
    $wnd.OnDSKeyUpEvt = function(nKeyCode,nShiftState) {
    impl.@com.bramosystems.oss.player.core.client.impl.WMPStateManager::fireCMEvents(IIIDD)(21,nKeyCode,nShiftState,0,0);
    }
    $wnd.OnDSKeyPressEvt = function(nKeyCode,nShiftState) {
    impl.@com.bramosystems.oss.player.core.client.impl.WMPStateManager::fireCMEvents(IIIDD)(22,nKeyCode,nShiftState,0,0);
    }
    }-*/;

    public void registerMediaStateHandlers(WinMediaPlayerImpl player) {
        // do nothing, provided for DOM event registration in IE.
    }

    protected class StateManager {

        protected HasMediaStateHandlers handlers;
        protected boolean canDoMetadata,  playerInitd,  resizing;
        private Timer downloadProgressTimer;
        protected WinMediaPlayerImpl player;

        public StateManager(final WinMediaPlayerImpl _player,
                HasMediaStateHandlers _handlers, boolean _resizing) {
            this.handlers = _handlers;
            this.player = _player;
            this.resizing = _resizing;
            canDoMetadata = false;
            playerInitd = false;
            downloadProgressTimer = new Timer() {

                @Override
                public void run() {
                    LoadingProgressEvent.fire(handlers, player.getDownloadProgress());
                }
            };
        }

        public void checkPlayState() {
            int state = player.getPlayState();
            if (state < 0) {
                return;
            }

            processPlayState(state);
        }

        public void onError(String message) {
            DebugEvent.fire(handlers, DebugEvent.MessageType.Error, message);
        }

        public void debug(String msg) {
            DebugEvent.fire(handlers, DebugEvent.MessageType.Info, msg);
        }

        public void doBuffering(boolean buffering) {
            PlayerStateEvent.fire(handlers,
                    buffering ? PlayerStateEvent.State.BufferingStarted : PlayerStateEvent.State.BufferingFinished);

            debug("Buffering " + (buffering ? " started" : " stopped"));
            if (buffering) {
                downloadProgressTimer.scheduleRepeating(1000);
            } else {
                downloadProgressTimer.cancel();
                LoadingProgressEvent.fire(handlers, 1.0);
                debug("Media loading complete");
            }
        }

        public void processPlayState(int state) {
            switch (state) {
                case 1:    // stopped..
                    debug("Media playback stopped");
                    PlayStateEvent.fire(handlers, PlayStateEvent.State.Stopped, 0);
                    break;
                case 2:    // paused..
                    debug("Media playback paused");
                    PlayStateEvent.fire(handlers, PlayStateEvent.State.Paused, 0);
                    break;
                case 3:    // playing..
                    PlayStateEvent.fire(handlers, PlayStateEvent.State.Started, 0);
                    doMetadata();        // do metadata ...
                    break;
                case 8:    // media ended...
                    PlayStateEvent.fire(handlers, PlayStateEvent.State.Finished, 0);
                    debug("Media playback finished");

                    // TODO: disable subsequent metadata (for playlist) until better resizing method
                    // is found for firefox brothers...
                    // resizing = false;
                    break;
                case 10:    // player ready...
                    PlayerStateEvent.fire(handlers, PlayerStateEvent.State.Ready);
                    if (!playerInitd && !resizing) {
                        debug("Windows Media Player plugin");
                        debug("Version : " + player.getPlayerVersion());
                        playerInitd = true;
                    }
                    break;
                case 6:    // buffering ...
                    debug("Buffering...");
                    break;
                case 11:    // reconnecting to stream  ...
                    break;
                case 9:     // preparing new item ...
                    canDoMetadata = true;
            }
        }

        public void checkError() {
            onError(player.getErrorDiscription());
        }

        protected void doMetadata() {
            if (resizing) {      // don't raise metadata event again, we've gotten it before;
                return;         // we're just resizing...
            }

            if (!canDoMetadata) {
                debug("Media playback resumed");
                return;
            }
            debug("Playing media at " + player.getURL());

            MediaInfo info = new MediaInfo();
            String err = "";
            player.fillMetadata(info, err);
            if (err.length() == 0) {
                canDoMetadata = false;
                MediaInfoEvent.fire(handlers, info);
            } else {
                onError(err);
            }
        }

        public void doClickMouseEvents(int type, int button, int shiftState, double fX, double fY) {
            boolean shift = (shiftState & 1) == 1;
            boolean alt = (shiftState & 2) == 2;
            boolean ctrl = (shiftState & 4) == 4;

            Element e = Element.as(player); //.getParentElement();
            int clientX = e.getAbsoluteLeft() + (int) fX - e.getOwnerDocument().getScrollLeft();
            int clientY = e.getAbsoluteTop() + (int) fY - e.getOwnerDocument().getScrollTop();
            int screenX = -1; //e.getAbsoluteLeft() + (int) fX; // - e.getScrollLeft();
            int screenY = -1; //e.getAbsoluteTop() + (int) fY; // - e.getScrollTop();

            Document _doc = Document.get();
            NativeEvent event = null;
            switch (type) {
                case 1:    // mouse down ..
                    event = _doc.createMouseDownEvent(button, screenX, screenY, clientX,
                            clientY, ctrl, alt, shift, false, button);
                    break;
                case 2:    // mouse up ...
                    event = _doc.createMouseUpEvent(button, screenX, screenY, clientX,
                            clientY, ctrl, alt, shift, false, button);
                    break;
                case 3:    // mouse move ...
                    event = _doc.createMouseMoveEvent(button, screenX, screenY, clientX,
                            clientY, ctrl, alt, shift, false, button);
                    break;
                case 10:    // click ...
                    event = _doc.createClickEvent(button, screenX, screenY, clientX,
                            clientY, ctrl, alt, shift, false);
                    break;
                case 11:    // double click ...
                    event = _doc.createDblClickEvent(button, screenX, screenY, clientX,
                            clientY, ctrl, alt, shift, false);
                    break;
                case 20:    // key down ...
                    event = _doc.createKeyDownEvent(ctrl, alt, shift, false, button, button);
                    break;
                case 21:    // key up ...
                    event = _doc.createKeyUpEvent(ctrl, alt, shift, false, button, button);
                    break;
                case 22:    // key press ...
                    event = _doc.createKeyPressEvent(ctrl, alt, shift, false, button, button);
                    break;
            }
            DomEvent.fireNativeEvent(event, handlers, e);
        }
    }
}
