/*
 * Decompiled with CFR 0.152.
 */
package org.apache.tools.tar;

import java.io.EOFException;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.text.SimpleDateFormat;
import java.util.Arrays;
import java.util.Date;

public class TarBuffer {
    public static final int DEFAULT_RCDSIZE = 512;
    public static final int DEFAULT_BLKSIZE = 10240;
    private static boolean debug = false;
    private InputStream inStream;
    private OutputStream outStream;
    private byte[] blockBuffer;
    private int currBlkIdx;
    private int currRecIdx;
    private int blockSize;
    private int recordSize;
    private int recsPerBlock;

    public TarBuffer(InputStream inStream) {
        this(inStream, 10240);
    }

    public TarBuffer(InputStream inStream, int blockSize) {
        this(inStream, blockSize, 512);
    }

    public TarBuffer(InputStream inStream, int blockSize, int recordSize) {
        TarBuffer.debug(String.format("TarBuffer(bsize=%d, rsize=%d", blockSize, recordSize));
        this.inStream = inStream;
        this.outStream = null;
        this.initialize(blockSize, recordSize);
    }

    public TarBuffer(OutputStream outStream) {
        this(outStream, 10240);
    }

    public TarBuffer(OutputStream outStream, int blockSize) {
        this(outStream, blockSize, 512);
    }

    public TarBuffer(OutputStream outStream, int blockSize, int recordSize) {
        this.inStream = null;
        this.outStream = outStream;
        this.initialize(blockSize, recordSize);
    }

    private void initialize(int blockSize, int recordSize) {
        this.blockSize = blockSize;
        this.recordSize = recordSize;
        this.recsPerBlock = this.blockSize / this.recordSize;
        this.blockBuffer = new byte[this.blockSize];
        if (this.inStream != null) {
            this.currBlkIdx = -1;
            this.currRecIdx = this.recsPerBlock;
        } else {
            this.currBlkIdx = 0;
            this.currRecIdx = 0;
        }
    }

    public int getBlockSize() {
        return this.blockSize;
    }

    public int getRecordSize() {
        return this.recordSize;
    }

    public static void setDebug(boolean isdebug) {
        debug = isdebug;
    }

    public boolean isEOFRecord(byte[] record) {
        int sz = this.getRecordSize();
        for (int i = 0; i < sz; ++i) {
            if (record[i] == 0) continue;
            return false;
        }
        return true;
    }

    public void skipRecord() throws IOException {
        TarBuffer.debug("SkipRecord: recIdx = " + this.currRecIdx + " blkIdx = " + this.currBlkIdx);
        if (this.inStream == null) {
            throw new IOException("reading (via skip) from an output buffer");
        }
        if (this.currRecIdx >= this.recsPerBlock && !this.readBlock()) {
            return;
        }
        ++this.currRecIdx;
    }

    public long skip(long bytes) throws IOException {
        long s;
        TarBuffer.debug(String.format("skip(%s)", bytes));
        long skip = bytes;
        long skipped = 0L;
        while (skip > 0L && this.currRecIdx < this.recsPerBlock) {
            skip -= (long)this.recordSize;
            skipped += (long)this.recordSize;
            ++this.currRecIdx;
        }
        long recs = skip / (long)this.recordSize * (long)this.recordSize;
        skip -= recs;
        while (recs > 0L) {
            s = this.inStream.skip(recs);
            if (s < 0L) {
                throw new EOFException();
            }
            recs -= s;
            skipped += s;
        }
        if (skip > 0L) {
            skip = this.recordSize;
            while (skip > 0L) {
                s = this.inStream.skip(skip);
                if (s < 0L) {
                    throw new EOFException();
                }
                skip -= s;
                skipped += s;
            }
        }
        return skipped;
    }

    public byte[] readRecord() throws IOException {
        if (this.inStream == null) {
            throw new IOException("reading from an output buffer");
        }
        if (this.currRecIdx >= this.recsPerBlock && !this.readBlock()) {
            return null;
        }
        byte[] result = new byte[this.recordSize];
        System.arraycopy(this.blockBuffer, this.currRecIdx * this.recordSize, result, 0, this.recordSize);
        ++this.currRecIdx;
        return result;
    }

    public boolean readRecord(byte[] buf) throws IOException {
        if (this.inStream == null) {
            throw new IOException("reading from an output buffer");
        }
        if (this.currRecIdx >= this.recsPerBlock && !this.readBlock()) {
            TarBuffer.debug("  this.readBlock() failed!");
            return false;
        }
        System.arraycopy(this.blockBuffer, this.currRecIdx * this.recordSize, buf, 0, this.recordSize);
        ++this.currRecIdx;
        return true;
    }

    public static String toChar(byte[] bytes) {
        if (bytes == null) {
            return null;
        }
        return TarBuffer.toChar(bytes, 0, bytes.length);
    }

    public static String toChar(byte[] bytes, int begin, int end) {
        if (bytes == null) {
            return null;
        }
        StringBuffer sb = new StringBuffer();
        for (int i = begin; i < end; ++i) {
            byte b = bytes[i];
            if (b >= 32 && b <= 126) {
                sb.append(String.format("%c", b));
                continue;
            }
            if (b == 0) {
                sb.append(String.format(".", b));
                continue;
            }
            sb.append(String.format("?", new Object[0]));
        }
        return sb.toString();
    }

    private boolean readBlock() throws IOException {
        if (this.inStream == null) {
            throw new IOException("reading from an output buffer");
        }
        this.currRecIdx = 0;
        int offset = 0;
        int bytesNeeded = this.blockSize;
        while (bytesNeeded > 0) {
            long numBytes = this.inStream.read(this.blockBuffer, offset, bytesNeeded);
            if (numBytes == -1L) {
                if (offset == 0) {
                    return false;
                }
                Arrays.fill(this.blockBuffer, offset, offset + bytesNeeded, (byte)0);
                break;
            }
            offset = (int)((long)offset + numBytes);
            bytesNeeded = (int)((long)bytesNeeded - numBytes);
            if (numBytes == (long)this.blockSize) continue;
            TarBuffer.debug("ReadBlock: INCOMPLETE READ " + numBytes + " of " + this.blockSize + " bytes read.");
        }
        ++this.currBlkIdx;
        return true;
    }

    public int getCurrentBlockNum() {
        return this.currBlkIdx;
    }

    public int getCurrentRecordNum() {
        return this.currRecIdx - 1;
    }

    public void writeRecord(byte[] record) throws IOException {
        TarBuffer.debug("WriteRecord: recIdx = " + this.currRecIdx + " blkIdx = " + this.currBlkIdx);
        if (this.outStream == null) {
            throw new IOException("writing to an input buffer");
        }
        if (record.length != this.recordSize) {
            throw new IOException("record to write has length '" + record.length + "' which is not the record size of '" + this.recordSize + "'");
        }
        if (this.currRecIdx >= this.recsPerBlock) {
            this.writeBlock();
        }
        System.arraycopy(record, 0, this.blockBuffer, this.currRecIdx * this.recordSize, this.recordSize);
        ++this.currRecIdx;
    }

    public void writeRecord(byte[] buf, int offset) throws IOException {
        TarBuffer.debug("WriteRecord: recIdx = " + this.currRecIdx + " blkIdx = " + this.currBlkIdx);
        if (this.outStream == null) {
            throw new IOException("writing to an input buffer");
        }
        if (offset + this.recordSize > buf.length) {
            throw new IOException("record has length '" + buf.length + "' with offset '" + offset + "' which is less than the record size of '" + this.recordSize + "'");
        }
        if (this.currRecIdx >= this.recsPerBlock) {
            this.writeBlock();
        }
        System.arraycopy(buf, offset, this.blockBuffer, this.currRecIdx * this.recordSize, this.recordSize);
        ++this.currRecIdx;
    }

    private void writeBlock() throws IOException {
        TarBuffer.debug("WriteBlock: blkIdx = " + this.currBlkIdx);
        if (this.outStream == null) {
            throw new IOException("writing to an input buffer");
        }
        this.outStream.write(this.blockBuffer, 0, this.blockSize);
        this.outStream.flush();
        this.currRecIdx = 0;
        ++this.currBlkIdx;
    }

    private void flushBlock() throws IOException {
        TarBuffer.debug("TarBuffer.flushBlock() called.");
        if (this.outStream == null) {
            throw new IOException("writing to an input buffer");
        }
        if (this.currRecIdx > 0) {
            this.writeBlock();
        }
    }

    public void close() throws IOException {
        TarBuffer.debug("TarBuffer.closeBuffer().");
        if (this.outStream != null) {
            this.flushBlock();
            if (this.outStream != System.out && this.outStream != System.err) {
                this.outStream.close();
                this.outStream = null;
            }
        } else if (this.inStream != null && this.inStream != System.in) {
            this.inStream.close();
            this.inStream = null;
        }
    }

    private static String tstamp() {
        SimpleDateFormat sdf = new SimpleDateFormat("H:m:s.S");
        return sdf.format(new Date(System.currentTimeMillis()));
    }

    private static void debug(String s) {
        if (debug) {
            System.err.println(String.format("[%s] %s", TarBuffer.tstamp(), s));
        }
    }
}

