package com.tandbergtv.neptune.ui.realm.client.tab.user;

import java.util.ArrayList;
import java.util.Collection;
import java.util.Date;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;

import com.google.gwt.core.client.GWT;
import com.google.gwt.event.dom.client.ClickEvent;
import com.google.gwt.event.dom.client.ClickHandler;
import com.google.gwt.user.client.Window;
import com.google.gwt.user.client.rpc.AsyncCallback;
import com.google.gwt.user.client.ui.HasHorizontalAlignment;
import com.google.gwt.user.client.ui.Widget;
import com.tandbergtv.neptune.ui.framework.client.i18n.NeptuneConstants;
import com.tandbergtv.neptune.ui.realm.client.event.RoleAddedEvent;
import com.tandbergtv.neptune.ui.realm.client.event.RoleDeletedEvent;
import com.tandbergtv.neptune.ui.realm.client.event.RoleUpdatedEvent;
import com.tandbergtv.neptune.ui.realm.client.event.UserAddedEvent;
import com.tandbergtv.neptune.ui.realm.client.event.UserDeletedEvent;
import com.tandbergtv.neptune.ui.realm.client.event.UserUpdatedEvent;
import com.tandbergtv.neptune.ui.realm.client.i18n.RealmConstants;
import com.tandbergtv.neptune.ui.realm.client.tab.role.RoleUiService;
import com.tandbergtv.neptune.ui.realm.client.tab.role.RoleUiServiceAsync;
import com.tandbergtv.neptune.ui.realm.client.tab.role.UiRole;
import com.tandbergtv.neptune.ui.realm.client.tab.role.UiRoleKey;
import com.tandbergtv.neptune.ui.realm.client.tab.role.UiRoleList;
import com.tandbergtv.neptune.widgettoolkit.client.application.ValidationException;
import com.tandbergtv.neptune.widgettoolkit.client.event.EventListener;
import com.tandbergtv.neptune.widgettoolkit.client.event.EventListenerRegistry;
import com.tandbergtv.neptune.widgettoolkit.client.event.EventSink;
import com.tandbergtv.neptune.widgettoolkit.client.event.NeptuneEvent;
import com.tandbergtv.neptune.widgettoolkit.client.menu.WidgetMenuItem.AnchorChangeListener;
import com.tandbergtv.neptune.widgettoolkit.client.remote.NeptuneAsyncCallback;
import com.tandbergtv.neptune.widgettoolkit.client.security.NeptuneSecurity;
import com.tandbergtv.neptune.widgettoolkit.client.widget.basic.ButtonWidget;
import com.tandbergtv.neptune.widgettoolkit.client.widget.basic.CheckBoxWidget;
import com.tandbergtv.neptune.widgettoolkit.client.widget.basic.LabelWidget;
import com.tandbergtv.neptune.widgettoolkit.client.widget.basic.ListBoxWidget;
import com.tandbergtv.neptune.widgettoolkit.client.widget.basic.TextBoxWidget;
import com.tandbergtv.neptune.widgettoolkit.client.widget.composite.FormContainer;
import com.tandbergtv.neptune.widgettoolkit.client.widget.composite.HeaderPanel;
import com.tandbergtv.neptune.widgettoolkit.client.widget.composite.table.Column;
import com.tandbergtv.neptune.widgettoolkit.client.widget.composite.table.ColumnBase;
import com.tandbergtv.neptune.widgettoolkit.client.widget.composite.table.DataProvider;
import com.tandbergtv.neptune.widgettoolkit.client.widget.composite.table.KeySerializer;
import com.tandbergtv.neptune.widgettoolkit.client.widget.composite.table.Record;
import com.tandbergtv.neptune.widgettoolkit.client.widget.composite.table.SortOrder;
import com.tandbergtv.neptune.widgettoolkit.client.widget.composite.table.Table;
import com.tandbergtv.neptune.widgettoolkit.client.widget.composite.table.feature.BookmarkFeatureImpl;
import com.tandbergtv.neptune.widgettoolkit.client.widget.composite.table.feature.DetailFeature;
import com.tandbergtv.neptune.widgettoolkit.client.widget.composite.table.feature.DetailView;
import com.tandbergtv.neptune.widgettoolkit.client.widget.composite.table.feature.PageFeatureImpl;
import com.tandbergtv.neptune.widgettoolkit.client.widget.composite.table.feature.SortFeatureImpl;
import com.tandbergtv.neptune.widgettoolkit.client.widget.composite.table.impl.TableInternal;
import com.tandbergtv.neptune.widgettoolkit.client.widget.composite.table.view.View;
import com.tandbergtv.neptune.widgettoolkit.client.widget.composite.table.view.label.LabelStringView;
import com.tandbergtv.neptune.widgettoolkit.client.widget.container.VerticalContainer;
import com.tandbergtv.neptune.widgettoolkit.client.widget.style.StyleNames;

public class UserTableProvider {
	private final EventListenerRegistry eventListenerRegistry;
	private final EventSink eventSink;
	private final UserUiServiceAsync userService = GWT.create(UserUiService.class);
	private final RoleUiServiceAsync roleService = GWT.create(RoleUiService.class);
	private final UserDataProvider dataProvider;
	private final PageFeatureImpl<UiUserKey, UserRecord> pageFeature;
	private final SortFeatureImpl<UiUserKey, UserRecord> sortFeature;
	private final BookmarkFeatureImpl<UiUserKey, UserRecord> bookmarkFeature;
	private final UserDetailFeature detailFeature;

	private List<UiRole> roles;
	private final Collection<RoleListener> roleListeners;
	private RoleUpdatedListener roleUpdatedListener;
	private RoleDeletedListener roleDeletedListener;
	private RoleAddedListener roleAddedListener;

	private final Table<UiUserKey, UserRecord> table;
	
	private RealmConstants constants = GWT.create(RealmConstants.class);
	
	private NeptuneSecurity security;
	private static String CREATE_PERMISSION = "UserManagement_Create";
	private static String DELETE_PERMISSION = "UserManagement_Delete";

	public UserTableProvider(EventSink eventSink, EventListenerRegistry eventListenerRegistry, NeptuneSecurity security) {
		this.eventSink = eventSink;
		this.eventListenerRegistry = eventListenerRegistry;
		this.roleListeners = new ArrayList<RoleListener>();
		this.security = security;

		dataProvider = new UserTableProvider.UserDataProvider();
		table = new Table<UiUserKey, UserRecord>(dataProvider);

		pageFeature = new PageFeatureImpl<UiUserKey, UserRecord>(table);
		detailFeature = new UserDetailFeature();
		sortFeature = new SortFeatureImpl<UiUserKey, UserRecord>(dataProvider.usernameColumn, SortOrder.ASCENDING);
		bookmarkFeature = new UserBookmarkFeatureImpl(table, new KeySerializer<UiUserKey>() {
			public UiUserKey fromString(String keyString) {
				return UiUserKey.parse(keyString);
			}

			public String toString(UiUserKey key) {
				return key.toString();
			}
		});
		sortFeature.addSortableColumn(dataProvider.usernameColumn);
		sortFeature.addSortableColumn(dataProvider.firstNameColumn);
		sortFeature.addSortableColumn(dataProvider.lastNameColumn);
	}

	public Table<?, ?> getTable(AnchorChangeListener reverseListener) {
		bookmarkFeature.addDetailFeature(reverseListener, table, dataProvider, detailFeature);
		bookmarkFeature.addPageFeature(reverseListener, table, pageFeature);

		table.addDetailFeature(detailFeature);
		table.addSortFeature(sortFeature);
		table.addPageFeature(pageFeature);
		table.addBookmarkFeature(bookmarkFeature);
		
		return table;
	}

	public AnchorChangeListener getAnchorChangeListener() {
		return new AnchorChangeListener() {
			public void anchorChanged(final String anchor) {
				table.init(new AsyncCallback<Void>() {
					@Override
					public void onFailure(Throwable caught) {
					}

					@Override
					public void onSuccess(Void result) {
						bookmarkFeature.anchorChanged(anchor);
					}});
			}
		};
	}

	private final class UserDataProvider implements DataProvider<UiUserKey, UserRecord> {
		private final List<Column<?, UserRecord>> columns;
		private final ColumnBase<String, UserRecord> usernameColumn;
		private final ColumnBase<String, UserRecord> firstNameColumn;
		private final ColumnBase<String, UserRecord> lastNameColumn;

		public UserDataProvider() {
			columns = new ArrayList<Column<?, UserRecord>>();

			usernameColumn = new ColumnBase<String, UserRecord>("userName", "User Name") {
				public View<String> getView(UserRecord record) {
					return new LabelStringView(record.getUser().getUserName());
				}
			};
			columns.add(usernameColumn);

			firstNameColumn = new ColumnBase<String, UserRecord>("firstName", "First Name") {
				public View<String> getView(UserRecord record) {
					return new LabelStringView(record.getUser().getFirstName());
				}
			};
			columns.add(firstNameColumn);
			lastNameColumn = new ColumnBase<String, UserRecord>("lastName", "Last Name") {
				public View<String> getView(UserRecord record) {
					return new LabelStringView(record.getUser().getLastName());
				}
			};
			columns.add(lastNameColumn);
		}

		public List<Column<?, UserRecord>> getColumns() {
			return columns;
		}

		public void getRecords(final AsyncCallback<List<UserRecord>> callback) {
			userService.listUsers(pageFeature.getStart(), pageFeature.getLength(), sortFeature.getSortColumnName(),
					sortFeature.isAscending(), new NeptuneAsyncCallback<UiUserList>() {

						public void onNeptuneFailure(Throwable caught) {
							Window.alert(caught.getMessage());
							callback.onFailure(caught);
						}

						public void onNeptuneSuccess(UiUserList result) {
							List<UserRecord> records = new ArrayList<UserRecord>();
							for (UiUser user : result.getUsers())
								records.add(new UserRecord(user));
							pageFeature.setTotalSize(result.getTotal());
							callback.onSuccess(records);
						}
					});

		}

		public void initialize(final AsyncCallback<Void> callback) {
			roleService.listRoles(0, 1000, "name", true, new NeptuneAsyncCallback<UiRoleList>() {

				public void onNeptuneFailure(Throwable caught) {
					callback.onFailure(caught);
				}

				public void onNeptuneSuccess(UiRoleList result) {
					roles = new ArrayList<UiRole>(result.getRoles());
					roleAddedListener = new RoleAddedListener();
					eventListenerRegistry.registerEventListener(RoleAddedEvent.class, roleAddedListener);
					roleUpdatedListener = new RoleUpdatedListener();
					eventListenerRegistry.registerEventListener(RoleUpdatedEvent.class, roleUpdatedListener);
					roleDeletedListener = new RoleDeletedListener();
					eventListenerRegistry.registerEventListener(RoleDeletedEvent.class, roleDeletedListener);
					callback.onSuccess(null);
				}
			});
		}

		public boolean isCheckboxEnabled() {
			return security.isUserInRole(DELETE_PERMISSION);
		}

		public boolean isRecordCountEnabled() {
			return true;
		}

		public void getRecord(UiUserKey key, final AsyncCallback<UserRecord> callback) {
			userService.getUser(key, new NeptuneAsyncCallback<UiUser>() {
				public void onNeptuneFailure(Throwable caught) {
					callback.onFailure(caught);
				}

				public void onNeptuneSuccess(UiUser result) {
					callback.onSuccess(new UserRecord(result));
				}
			});
		}
	}

	private final class UserDetailFeature implements DetailFeature<UiUserKey, UserRecord> {
		private final class UserRecordDetailView implements DetailView<UiUserKey, UserRecord> {
			private static final String STYLE_VALIDATION_MESSAGE = "realm-validationFailure-message";
			
			private final UserRecord record;
			private final TextBoxWidget usernameBox = new TextBoxWidget();
			private final TextBoxWidget passwordTextBox = new TextBoxWidget();
			private final RoleListener roleListener;
			private final Map<UiRoleKey, UiRole> keyToRoleMap = new HashMap<UiRoleKey, UiRole>();
			private final ListBoxWidget<UiRole> rolesListBox = new ListBoxWidget<UiRole>();
			private final TextBoxWidget firstNameTextBox = new TextBoxWidget();
			private final TextBoxWidget lastNameTextBox = new TextBoxWidget();
			private final CheckBoxWidget isActiveCheckBox = new CheckBoxWidget("", false);
			private final TextBoxWidget emailTextBox = new TextBoxWidget();
			private final TextBoxWidget employeeIdTextBox = new TextBoxWidget();
			private final TextBoxWidget departmentTextBox = new TextBoxWidget();
			private final TextBoxWidget locationTextBox = new TextBoxWidget();
			private final TextBoxWidget phoneTextBox = new TextBoxWidget();
			private final TextBoxWidget extTextBox = new TextBoxWidget();
			private final LabelWidget errorMessageLabel = new LabelWidget(); 
			private final FormContainer formContainer = new FormContainer(HasHorizontalAlignment.ALIGN_LEFT);

			private final VerticalContainer headerAndMainContainer = new VerticalContainer();
			
			NeptuneConstants constants = GWT.create(NeptuneConstants.class);
			
			private UserRecordDetailView(final UserRecord record,
					final DetailViewCallback<UiUserKey, UserRecord> callback) {
				this.record = record;
				
				errorMessageLabel.setStyleName(STYLE_VALIDATION_MESSAGE);

				/* Add a null default entry in the role list box */
				rolesListBox.addItem("", null);

				usernameBox.setReadOnly(record.getUser().getKey().getId() != null);
				roleListener = new RoleListener() {
					public void roleAdded(UiRole role, boolean initial) {
						rolesListBox.addItem(role.getName(), role);
						keyToRoleMap.put(role.getKey(), role);
						if (initial)
							return;

						roles.add(role);
					}

					public void roleDeleted(UiRole role, boolean initial) {
						rolesListBox.removeItem(role);
						keyToRoleMap.remove(role.getKey());
						if (initial)
							return;

						roles.remove(role);
					}

					public void roleUpdated(UiRole role, boolean initial) {
						/* Update the role at its current index */
						int index = rolesListBox.getIndex(role);
						if (index == -1) {
							rolesListBox.addItem(role.getName(), role);
						} else {
							rolesListBox.setItem(role.getName(), role, index);
						}

						if (initial)
							return;

						if(index == -1) {
							roles.add(role);
						} else {
							roles.remove(index - 1);
							roles.add(index - 1, role);
						}
					}
				};
				addRoleListener(roleListener);

				// invoke it now to build the initial list of roles
				for (UiRole role : roles)
					roleListener.roleAdded(role, true);

				formContainer.addRow("User Name:", usernameBox, true);
				formContainer.addRow("Password:", passwordTextBox);
				formContainer.addRow("Roles:", rolesListBox, true);
				formContainer.addRow("First Name:", firstNameTextBox, true);
				formContainer.addRow("Last Name:", lastNameTextBox, true);
				formContainer.addRow("Is Active?:", isActiveCheckBox);
				formContainer.addRow("Email:", emailTextBox, true);
				formContainer.addRow("Employee ID:", employeeIdTextBox, true);
				formContainer.addRow("Department:", departmentTextBox, true);
				formContainer.addRow("Location:", locationTextBox, true);
				formContainer.addRow("Phone:", phoneTextBox);
				formContainer.addRow("Extension:", extTextBox);

				// make the form buttons call back the table
				
				ButtonWidget saveButton = new ButtonWidget("Save", new ClickHandler() {
					public void onClick(ClickEvent event) {
						errorMessageLabel.setText(null);
						callback.save(UserRecordDetailView.this);
					}
				});
				saveButton.addStyleDependentName(StyleNames.COMMIT_BUTTON_STYLE);
				formContainer.addButton(saveButton);
				
				ButtonWidget cancelButton = new ButtonWidget("Cancel", new ClickHandler() {
					public void onClick(ClickEvent event) {
						errorMessageLabel.setText(null);
						callback.cancel(UserRecordDetailView.this);
					}
				});
				cancelButton.addStyleDependentName(StyleNames.DATALOSS_BUTTON_STYLE);
				formContainer.addButton(cancelButton);
				
				ButtonWidget revertButton = new ButtonWidget("Revert", new ClickHandler() {
					public void onClick(ClickEvent event) {
						errorMessageLabel.setText(null);
						callback.revert(UserRecordDetailView.this);
					}
				});
				revertButton.addStyleDependentName(StyleNames.DATALOSS_BUTTON_STYLE);
				formContainer.addButton(revertButton);				
				revert();
			}

			public void commit() {
				record.getUser().setUserName(usernameBox.getText());
				if ((passwordTextBox.getText() != null && passwordTextBox.getText().trim().length() > 0)) {
					record.getUser().setChangePassword(true);
					record.getUser().setPasswordModifiedDate(new Date());
					record.getUser().setPassword(passwordTextBox.getText().trim());
				}

				/* Get selected Role IDs */
				final HashSet<UiRoleKey> roleIds = new HashSet<UiRoleKey>();
				for (int index = 0; index < rolesListBox.getItemCount(); index++) {
					if (rolesListBox.isItemSelected(index)) {
						UiRole role = rolesListBox.getItem(index);
						if (role != null) {
							roleIds.add(role.getKey());
						}
					}
				}

				record.getUser().setRoleIds(roleIds);
				record.getUser().setFirstName(firstNameTextBox.getText());
				record.getUser().setLastName(lastNameTextBox.getText());
				record.getUser().setActive(isActiveCheckBox.getValue());
				record.getUser().setEmail(emailTextBox.getText());
				record.getUser().setEmployeeId(employeeIdTextBox.getText());
				record.getUser().setDepartment(departmentTextBox.getText());
				record.getUser().setLocation(locationTextBox.getText());
				record.getUser().setPhone(phoneTextBox.getText());
				record.getUser().setExtension(extTextBox.getText());
			}

			@Override
			public Widget getWidget() {
				String userName = record.getUser().getUserName();
				if(userName == null) {
					userName = "";
				}
				headerAndMainContainer.add(new HeaderPanel(constants.userDetails() + " " + userName));
				headerAndMainContainer.add(errorMessageLabel);
				headerAndMainContainer.add(formContainer);
				return headerAndMainContainer;
			}

			public void revert() {
				usernameBox.setText(record.getUser().getUserName());
				final Set<UiRoleKey> roleIds = record.getUser().getRoleIds();
				rolesListBox.setSelectedIndex(-1);
				for (UiRoleKey key : roleIds) {
					int index = rolesListBox.getIndex(keyToRoleMap.get(key));
					rolesListBox.setItemSelected(index, true);
				}
				firstNameTextBox.setText(record.getUser().getFirstName());
				lastNameTextBox.setText(record.getUser().getLastName());
				isActiveCheckBox.setValue(record.getUser().isActive());
				emailTextBox.setText(record.getUser().getEmail());
				employeeIdTextBox.setText(record.getUser().getEmployeeId());
				departmentTextBox.setText(record.getUser().getDepartment());
				locationTextBox.setText(record.getUser().getLocation());
				phoneTextBox.setText(record.getUser().getPhone());
				extTextBox.setText(record.getUser().getExtension());
			}

			public void release() {
				removeRoleListener(roleListener);
			}

			public UserRecord getRecord() {
				return record;
			}

			public String getStyleName() {
			    return null;
			}

			@Override
			public void saveFailed(Throwable throwable) {
				if(throwable instanceof ValidationException) {
					ValidationException ve = (ValidationException) throwable;
					StringBuffer messagesSB = new StringBuffer();
					for(String message : ve.getValidationMessages()) {
						messagesSB.append(message).append("\n");
					}
					errorMessageLabel.setText(messagesSB.toString());
				} else {
					errorMessageLabel.setText(throwable.getLocalizedMessage());
				}
			}
		}

		private Listener<UiUserKey, UserRecord> listener;

		public void delete(final List<UserRecord> records, final AsyncCallback<Void> callback) {
			List<UiUserKey> userKeys = new ArrayList<UiUserKey>();
			for (UserRecord record : records)
				userKeys.add(record.getUser().getKey());
			userService.deleteUsers(userKeys, new AsyncCallback<Void>() {
				public void onFailure(Throwable caught) {
					callback.onFailure(caught);
				}

				public void onSuccess(Void result) {
					for (UserRecord record : records)
						eventSink.fireEvent(new UserDeletedEvent(record.getUser()));
					callback.onSuccess(null);
				}
			});
		}

		public void getNew(AsyncCallback<UserRecord> callback) {
			UserRecord record = new UserRecord(new UiUser());
			callback.onSuccess(record);
		}

		public DetailView<UiUserKey, UserRecord> getView(final UserRecord record,
				final DetailViewCallback<UiUserKey, UserRecord> callback) {
			final UserRecordDetailView userRecordDetailView = new UserRecordDetailView(record, callback);

			if(record.getKey().isValueAssigned())
				listener.detailViewShownForEdit(userRecordDetailView);
			else
				listener.detailViewShownForCreate(userRecordDetailView);
			
			return userRecordDetailView;
			
		}

		public boolean hasDetailLink(Column<?, UserRecord> column) {
			return column.getName().equals("userName");
		}

		public void save(final UserRecord record, final AsyncCallback<Void> callback) {
			final UiUser user = record.getUser();
			userService.saveUser(user, new NeptuneAsyncCallback<UiUserKey>() {

				public void onNeptuneFailure(Throwable caught) {
					callback.onFailure(caught);
				}

				public void onNeptuneSuccess(UiUserKey userId) {
					NeptuneEvent neptuneEvent = user.getId() == null ? new UserAddedEvent(user) : new UserUpdatedEvent(record
							.getUser());
					user.setKey(userId);
					eventSink.fireEvent(neptuneEvent);
					callback.onSuccess(null);
				}
			});
		}

		public void registerListener(DetailFeature.Listener<UiUserKey, UserRecord> listener) {
			this.listener = listener;
		}

		@Override
		public boolean showCreateButton() {
			return security.isUserInRole(CREATE_PERMISSION);
		}

		@Override
		public boolean showDeleteButton() {
			return security.isUserInRole(DELETE_PERMISSION);
		}
	}

	private final static class UserRecord implements Record<UiUserKey> {
		private final UiUser user;

		public UserRecord(UiUser user) {
			this.user = user;
		}

		public UiUser getUser() {
			return user;
		}

		public UiUserKey getKey() {
			return user.getKey();
		}
	}

	private final class RoleAddedListener implements EventListener<RoleAddedEvent> {
		public void eventOccured(RoleAddedEvent event) {
			for (RoleListener roleListener : roleListeners)
				roleListener.roleAdded(event.getRole(), false);
		}
	}

	private final class RoleUpdatedListener implements EventListener<RoleUpdatedEvent> {

		public void eventOccured(RoleUpdatedEvent event) {
			for (RoleListener roleListener : roleListeners)
				roleListener.roleUpdated(event.getRole(), false);
		}
	}

	private final class RoleDeletedListener implements EventListener<RoleDeletedEvent> {
		public void eventOccured(RoleDeletedEvent event) {
			for (RoleListener roleListener : roleListeners)
				roleListener.roleDeleted(event.getRole(), false);
		}
	}

	private interface RoleListener {
		void roleAdded(UiRole role, boolean initial);

		void roleUpdated(UiRole role, boolean initial);

		void roleDeleted(UiRole role, boolean initial);
	}

	private void addRoleListener(RoleListener roleListener) {
		roleListeners.add(roleListener);
	}

	private void removeRoleListener(RoleListener roleListener) {
		roleListeners.remove(roleListener);
	}

	private class UserBookmarkFeatureImpl extends BookmarkFeatureImpl<UiUserKey, UserRecord> {
		private TableInternal<UiUserKey, UserRecord> tableInternal;
		private static final String USER_NAME_ANCHOR = "Edit?userName=";
		
		public UserBookmarkFeatureImpl(
				TableInternal<UiUserKey, UserRecord> tableInternal,
				KeySerializer<UiUserKey> keySerializer) {
			super(tableInternal, keySerializer);
			this.tableInternal = tableInternal;
		}

		@Override
		public void anchorChanged(String anchor) {
			/* Check for anchor with user name */
			if (anchor != null && anchor.startsWith(USER_NAME_ANCHOR)) {
				final String userName = anchor.substring(USER_NAME_ANCHOR.length());
				userService.getUser(userName, new NeptuneAsyncCallback<UiUser>() {
							public void onNeptuneFailure(Throwable caught) {
								if(caught instanceof UserNotFoundException)
									Window.alert(constants.userNotFoundForUserName() + userName);
								else
									Window.alert(caught.getLocalizedMessage());
							}

							public void onNeptuneSuccess(UiUser result) {
								tableInternal.showRecord(new UserRecord(result));
							}
						});
			} else {
				super.anchorChanged(anchor);
			}
		}
	}


}
