package com.tandbergtv.neptune.ui.home.client;

import java.util.ArrayList;
import java.util.List;

import com.google.gwt.core.client.GWT;
import com.google.gwt.event.dom.client.ClickEvent;
import com.google.gwt.event.dom.client.ClickHandler;
import com.google.gwt.user.client.Window;
import com.google.gwt.user.client.rpc.AsyncCallback;
import com.google.gwt.user.client.ui.Composite;
import com.google.gwt.user.client.ui.VerticalPanel;
import com.gwtext.client.widgets.Component;
import com.gwtext.client.widgets.layout.ColumnLayoutData;
import com.gwtext.client.widgets.portal.Portal;
import com.tandbergtv.neptune.ui.framework.client.i18n.NeptuneConstants;
import com.tandbergtv.neptune.ui.realm.client.i18n.RealmConstants;
import com.tandbergtv.neptune.ui.realm.client.tab.userpref.UiUserPreference;
import com.tandbergtv.neptune.ui.realm.client.tab.userpref.UserPreferenceService;
import com.tandbergtv.neptune.ui.realm.client.tab.userpref.UserPreferenceServiceAsync;
import com.tandbergtv.neptune.widgettoolkit.client.application.PortletFactory;
import com.tandbergtv.neptune.widgettoolkit.client.remote.NeptuneAsyncCallback;
import com.tandbergtv.neptune.widgettoolkit.client.widget.basic.AnchorWidget;
import com.tandbergtv.neptune.widgettoolkit.client.widget.composite.BusyIndicator;
import com.tandbergtv.neptune.widgettoolkit.client.widget.container.Portlet;
import com.tandbergtv.neptune.widgettoolkit.client.widget.container.SimpleContainer;
import com.tandbergtv.neptune.widgettoolkit.client.widget.container.VerticalContainer;
import com.tandbergtv.neptune.widgettoolkit.client.widget.event.DocumentAttachEvent;
import com.tandbergtv.neptune.widgettoolkit.client.widget.event.DocumentDetachEvent;

public class HomeTabPanel extends Composite implements PortalChangeListener {
    
    public static final String PORTLET_ORDER_TYPE = "portlet.order";
    public static final String HOME_COMPONENT = "Home";
    private final SimpleContainer mainContainer;
    private final List<PortletFactory> portletFactories;
    private final List<Portlet> portlets;
    private Portal portal;
    private HomePortalColumn col1;
    private HomePortalColumn col2;
    private AddPortletPanel addPortletPanel;
    private int i = 0;
    private BusyIndicator busyIndicator = new BusyIndicator();

    private final UserPreferenceServiceAsync userPreferenceService = GWT.create( UserPreferenceService.class );
    private RealmConstants realmConstants = GWT.create( RealmConstants.class );
    private NeptuneConstants constants = GWT.create(NeptuneConstants.class);

    private static final String STYLE_NAME = "home-HomeTabPanel";
    private static final String STYLE_PORTAL = "home-HomeTabPanel-portal";
    private static final String STYLE_PORTAL_COLUMN = "home-HomeTabPanel-portalColumn";

    /**
     * Constructor
     * @param portletFactories the list of all portlet factories
     */
    public HomeTabPanel( List<PortletFactory> portletFactories ) {
        mainContainer = new SimpleContainer();
        mainContainer.addStyleName( STYLE_NAME );
        this.portletFactories = portletFactories;
        this.portlets = new ArrayList<Portlet>();
        initPortal();

        initWidget( mainContainer );
    }

    /**
     * Initializes the portal by loading the portlets
     */
    public void initPortal() {
        portal = createPortal();
        loadPortlets();
    }
    
    private Portal createPortal() {
        i = 0;
        VerticalContainer mainPanel = new VerticalContainer();
        mainPanel.addStyleName( STYLE_NAME );
        AnchorWidget addPortletLink = new AnchorWidget( constants.addPortlet() );
        addPortletLink.addClickHandler( new ClickHandler() {

            @Override
            public void onClick( ClickEvent event ) {
                if ( addPortletPanel == null ) {
                    addPortletPanel = new AddPortletPanel( HomeTabPanel.this, portletFactories );
                } else {
                    addPortletPanel.refresh();
                }
                mainContainer.setWidget( addPortletPanel );
            }
        } );
        mainPanel.add( addPortletLink );
        mainPanel.setCellHorizontalAlignment( addPortletLink, VerticalPanel.ALIGN_RIGHT );

        Portal portal = new MyPortal();
        portal.setBorder( true );
        portal.addClass( STYLE_PORTAL );
        mainPanel.add( portal );
        mainPanel.setCellHorizontalAlignment( portal, VerticalPanel.ALIGN_CENTER );
        mainContainer.clear();
        mainContainer.setWidget( mainPanel );

        ColumnLayoutData layoutData = new ColumnLayoutData( .5 );
        col1 = createColumn();
        col2 = createColumn();

        portal.add( col1, layoutData );
        portal.add( col2, layoutData );
        
        return portal;
    }

    private void loadPortlets() {
        userPreferenceService.getPreferences( HOME_COMPONENT, PORTLET_ORDER_TYPE,
            new NeptuneAsyncCallback<List<UiUserPreference>>() {

                @Override
                public void onNeptuneFailure( Throwable caught ) {
                    Window.alert( realmConstants.errorLoadingState() );
                }

                @Override
                public void onNeptuneSuccess( List<UiUserPreference> result ) {
                    updatePortlets( result );
                }
            } 
        );
    }

    private void updatePortlets( List<UiUserPreference> userPreferencesList ) {
        if ( userPreferencesList.size() == 0 ) {
            for ( PortletFactory portletFactory : portletFactories ) {
                addPortlet( portletFactory );
            }
        } else {
            int portletsMaxIndex = getPortletsMaxIndex( userPreferencesList );
            PortletFactory[] orderedPortletFactories = new PortletFactory[portletsMaxIndex + 1];
            for ( UiUserPreference userPreference : userPreferencesList ) {
                Integer index = getIntValue( userPreference.getValue() );
                if ( (index != null) && (index >= 0) ) {
                    PortletFactory portletFactory = getPortletFactoryByName( portletFactories,
                            userPreference.getKey() );
                    if ( portletFactory != null ) {
                        orderedPortletFactories[index] = portletFactory;
                    }
                }
            }
            for ( PortletFactory portletFactory : orderedPortletFactories ) {
                addPortlet( portletFactory );
            }
        }
        portal.doLayout();
    }

    private int getPortletsMaxIndex( List<UiUserPreference> userPreferencesList ) {
        int maxIndex = 0;
        for ( UiUserPreference uiUserPreference : userPreferencesList ) {
            Integer index = getIntValue( uiUserPreference.getValue() );
            if ( (index != null) && (index > maxIndex) ) {
                maxIndex = index;
            }
        }
        return maxIndex;
    }

    private PortletFactory getPortletFactoryByName( List<PortletFactory> portletFactories, String key ) {
        PortletFactory portletFactory = null;
        for ( PortletFactory factory : portletFactories ) {
            Portlet portlet = (Portlet) factory.getInstance();
            if ( portlet.getId().equals( key ) ) {
                portletFactory = factory;
                break;
            }
        }
        return portletFactory;
    }

    private Integer getIntValue( String stringValue ) {
        Integer value = null;
        try {
            value = Integer.valueOf( stringValue );
        } catch ( Exception e ) {
            // do nothing
        }
        return value;
    }

    private void addPortlet( PortletFactory portletFactory ) {
        HomePortalColumn col = getColumn( i++ );
        if ( portletFactory != null ) {
            Portlet portlet = portletFactory.getInstance();
            addPortletToColumn( portlet, col );
        }
    }

    /**
     * Adds a new portlet to the portal
     * @param portlet the new portlet
     */
    public void addPortlet( Portlet portlet ) {
        HomePortalColumn col = getColumn( i++ );
        if ( portlet != null ) {
            addPortletToColumn( portlet, col );
        }
    }

    private void addPortletToColumn( Portlet portlet, HomePortalColumn col ) {
        col.add( portlet );
        portlet.setCloseListener( col );

        /* Cache the portlet, and fire attach event if required */
        portlets.add( portlet );
        portlet.setVisible( true );
        if ( this.isAttached() ) {
            portlet.fireEvent( new DocumentAttachEvent() );
        }
    }

    private HomePortalColumn getColumn( int i ) {
        switch ( i % 2 ) {
        case 0:
            return col1;
        case 1:
            return col2;
        default:
            return col1;
        }
    }

    private HomePortalColumn createColumn() {
        HomePortalColumn col = new HomePortalColumn( this );
        col.addClass( STYLE_PORTAL_COLUMN );
        return col;
    }

    @Override
    protected void onLoad() {
        super.onLoad();

        for ( Portlet portlet : portlets ) {
            portlet.fireEvent( new DocumentAttachEvent() );
        }
    }

    @Override
    protected void onUnload() {
        for ( Portlet portlet : portlets ) {
            portlet.fireEvent( new DocumentDetachEvent() );
        }

        super.onUnload();
    }

    private static final class MyPortal extends Portal {
        @Override
        protected void onAttach() {
            doLayout();
            super.onAttach();
        }
    }

    @Override
    /**
     * Notifies that the portal has changed by adding a new portlet, 
     * removing a portlet or by moving portlets.
     */
    public void notifyPortalChanged() {

        saveCurrentState( new NeptuneAsyncCallback<Void>() {

            @Override
            public void onNeptuneFailure( Throwable caught ) {
                Window.alert( realmConstants.errorSavingState() );
            }

            @Override
            public void onNeptuneSuccess( Void result ) {
                rebuildPortal();
            }
        } );

        
    }

    /**
     * Saves the current portal state by saving the current portlets positioning
     * @param callback the callback that will be called to receive the return value
     */
    public void saveCurrentState( final AsyncCallback<Void> callback ) {
        busyIndicator.center();
        userPreferenceService.deletePreferences( HOME_COMPONENT, PORTLET_ORDER_TYPE,
            new NeptuneAsyncCallback<Void>() {

                @Override
                public void onNeptuneFailure( Throwable caught ) {
                    busyIndicator.hide();
                    callback.onFailure( caught );
                }

                @Override
                public void onNeptuneSuccess( Void result ) {
                    createPreferencesForCurrentState( callback );
                }

            } 
        );
    }

    private void createPreferencesForCurrentState( final AsyncCallback<Void> callback ) {
        List<Portlet> portletsTemp1 = getPortletsFromColumn( this.col1 );
        List<Portlet> portletsTemp2 = getPortletsFromColumn( this.col2 );

        int maxSize = portletsTemp1.size();
        if ( portletsTemp2.size() > maxSize ) {
            maxSize = portletsTemp2.size();
        }

        List<UiUserPreference> uiUserPreferences = new ArrayList<UiUserPreference>();
        uiUserPreferences.add( createUserPreference( "___CONTROL___", -1  ) ); // Needed to allow to hide all portlets
        int portletIndex = 0;
        for ( int x = 0; x < maxSize; x++ ) {
            if ( x < portletsTemp1.size() ) {
                uiUserPreferences.add( createUserPreference( portletsTemp1.get( x ).getId(), portletIndex ) );
            }
            portletIndex++;
            if ( x < portletsTemp2.size() ) {
                uiUserPreferences.add( createUserPreference( portletsTemp2.get( x ).getId(), portletIndex ) );
            }
            portletIndex++;
        }

        userPreferenceService.create( uiUserPreferences, new NeptuneAsyncCallback<List<UiUserPreference>>() {

            @Override
            public void onNeptuneFailure( Throwable caught ) {
                busyIndicator.hide();
                callback.onFailure( caught );
            }

            @Override
            public void onNeptuneSuccess( List<UiUserPreference> result ) {
                busyIndicator.hide();
                callback.onSuccess( null );
            }
        } );

    }
    
    private UiUserPreference createUserPreference( String key, int index ) {
        UiUserPreference uiUserPreference = new UiUserPreference();
        uiUserPreference.setComponent( HOME_COMPONENT );
        uiUserPreference.setType( PORTLET_ORDER_TYPE );
        uiUserPreference.setKey( key );
        uiUserPreference.setValue( String.valueOf( index ) );

        return uiUserPreference;
    }

    private List<Portlet> getPortletsFromColumn( HomePortalColumn col ) {
        List<Portlet> portlets = new ArrayList<Portlet>();
        Component[] comps = col.getComponents();
        for ( Component component : comps ) {
            if ( component instanceof Portlet ) {
                Portlet portlet = (Portlet) component;
                portlets.add( portlet );
            }
        }
        return portlets;
    }

    private void rebuildPortal() {

        List<Portlet> portletsTemp1 = getPortletsFromColumn( this.col1 );
        List<Portlet> portletsTemp2 = getPortletsFromColumn( this.col2 );

        portal = createPortal();
        portlets.clear();

        for ( Portlet portlet : portletsTemp1 ) {
            addPortletToColumn( portlet, col1 );
        }
        for ( Portlet portlet : portletsTemp2 ) {
            addPortletToColumn( portlet, col2 );
        }

        portal.doLayout();
    }

    public void showBusyIndicator() {
        busyIndicator.center();
    }
    
    public void hideBusyIndicator() {
        busyIndicator.hide();
    }
    
}
