/*
 * Created on Sep 9, 2009
 * 
 * (C) Copyright TANDBERG Television Ltd.
 */

package com.tandbergtv.neptune.widgettoolkit.client.widget.composite.table.impl;

import java.util.HashMap;
import java.util.Map;

/**
 * Class that breaks the anchor string into tokens or builds the anchor string from tokens
 * 
 * @author Vijay Silva
 */
public class AnchorTokenizer {

	/**
	 * The separator used between tokens in the anchor
	 */
	public static final String ANCHOR_SEPARATOR = "&";

	/**
	 * The separator between the token key and value
	 */
	public static final String TOKEN_SEPARATOR = "=";

	/**
	 * The sequence of characters that serve as the escape sequence for tokens that contain the
	 * anchor separator
	 */
	public static final String ESCAPED_ANCHOR_SEPARATOR = "&amp;";

	/**
	 * The sequence of characters that serve as the escape sequence for tokens that contain the
	 * anchor separator
	 */
	public static final String ESCAPED_TOKEN_SEPARATOR = "&eq;";

	/**
	 * Constructor
	 */
	public AnchorTokenizer() {
	}

	/**
	 * Build the anchor given the anchor tokens
	 * 
	 * @param anchorTokens The anchor tokens
	 * @return the anchor
	 */
	public String buildAnchor(Map<String, String> anchorTokens) {
		if (anchorTokens == null)
			return "";

		/* Serialize the tokens into a single string */
		StringBuilder anchor = new StringBuilder();

		boolean first = true;
		for (String key : anchorTokens.keySet()) {
			String value = anchorTokens.get(key);
			if (key != null && value != null) {
				/* Add separator if required */
				if (first)
					first = false;
				else
					anchor.append(ANCHOR_SEPARATOR);

				/* Add the token */
				anchor.append(escapeValue(key));
				anchor.append(TOKEN_SEPARATOR);
				anchor.append(escapeValue(value));
			}
		}

		return anchor.toString();
	}

	/**
	 * Build the map of tokens given the anchor
	 * 
	 * @param anchor The anchor
	 * @return The anchor tokens
	 */
	public Map<String, String> parseAnchor(String anchor) {
		Map<String, String> tokens = new HashMap<String, String>();
		if (anchor == null)
			return tokens;

		int startIndex = 0;
		int index = 0;
		while (index < anchor.length()) {
			String token = null;
			if (isMatch(anchor, index, ANCHOR_SEPARATOR) && !isEscapeSequence(anchor, index)) {
				token = anchor.substring(startIndex, index);
				startIndex = index + ANCHOR_SEPARATOR.length();
				index += ANCHOR_SEPARATOR.length();
			} else if (index == (anchor.length() - 1)) {
				token = anchor.substring(startIndex);
				startIndex = index + 1;
				index++;
			} else {
				index++;
			}

			/* Process the token */
			if (token != null) {
				String[] tokenParts = token.split(TOKEN_SEPARATOR);
				if (tokenParts.length == 2) {
					String key = unescapeValue(tokenParts[0]);
					String value = unescapeValue(tokenParts[1]);
					if (key != null && value != null)
						tokens.put(key, value);
				}
			}
		}

		return tokens;
	}

	/*
	 * Escape the presence of all anchor and token separators in the value
	 */
	private String escapeValue(String value) {
		if (value == null)
			return null;

		StringBuilder result = new StringBuilder();
		int index = 0;
		while (index < value.length()) {
			if (isMatch(value, index, ANCHOR_SEPARATOR)) {
				result.append(ESCAPED_ANCHOR_SEPARATOR);
				index += ANCHOR_SEPARATOR.length();
			} else if (isMatch(value, index, TOKEN_SEPARATOR)) {
				result.append(ESCAPED_TOKEN_SEPARATOR);
				index += TOKEN_SEPARATOR.length();
			} else {
				result.append(value.charAt(index));
				index++;
			}
		}
		return result.toString();
	}

	/*
	 * Un-escape the presence of anchor and token separators in the value
	 */
	private String unescapeValue(String value) {
		if (value == null)
			return null;

		StringBuilder result = new StringBuilder();
		int index = 0;
		while (index < value.length()) {
			if (isMatch(value, index, ESCAPED_ANCHOR_SEPARATOR)) {
				result.append(ANCHOR_SEPARATOR);
				index += ESCAPED_ANCHOR_SEPARATOR.length();
			} else if (isMatch(value, index, ESCAPED_TOKEN_SEPARATOR)) {
				result.append(TOKEN_SEPARATOR);
				index += ESCAPED_TOKEN_SEPARATOR.length();
			} else {
				result.append(value.charAt(index));
				index++;
			}
		}
		return result.toString();
	}

	/*
	 * Check if any of the escape sequences match the current value starting from the given index
	 */
	private boolean isEscapeSequence(String value, int startIndex) {
		return isMatch(value, startIndex, ESCAPED_ANCHOR_SEPARATOR)
		        || isMatch(value, startIndex, ESCAPED_TOKEN_SEPARATOR);
	}

	/*
	 * Determine if the substring of value starting from the start index matches the match string
	 */
	private boolean isMatch(String value, int startIndex, String match) {
		int length = value.length();
		int endIndex = startIndex + match.length();
		return ((length >= endIndex) && value.substring(startIndex, endIndex).equals(match));
	}
}
