/*
 * Created on Sep 10, 2009
 * 
 * (C) Copyright TANDBERG Television Ltd.
 */

package com.tandbergtv.neptune.widgettoolkit.client.widget.composite.table.impl;

import static com.google.gwt.user.client.ui.HasHorizontalAlignment.ALIGN_CENTER;
import static com.google.gwt.user.client.ui.HasHorizontalAlignment.ALIGN_LEFT;
import static com.google.gwt.user.client.ui.HasHorizontalAlignment.ALIGN_RIGHT;
import static com.google.gwt.user.client.ui.HasVerticalAlignment.ALIGN_MIDDLE;

import com.google.gwt.core.client.GWT;
import com.google.gwt.event.dom.client.ClickEvent;
import com.google.gwt.event.dom.client.ClickHandler;
import com.google.gwt.event.dom.client.KeyCodes;
import com.google.gwt.event.dom.client.KeyPressEvent;
import com.google.gwt.event.dom.client.KeyPressHandler;
import com.google.gwt.user.client.Window;
import com.google.gwt.user.client.ui.AbstractImagePrototype;
import com.google.gwt.user.client.ui.Composite;
import com.google.gwt.user.client.ui.Grid;
import com.tandbergtv.neptune.widgettoolkit.client.widget.basic.ImageWidget;
import com.tandbergtv.neptune.widgettoolkit.client.widget.basic.LabelWidget;
import com.tandbergtv.neptune.widgettoolkit.client.widget.basic.TextBoxWidget;
import com.tandbergtv.neptune.widgettoolkit.client.widget.composite.table.TableConstants;
import com.tandbergtv.neptune.widgettoolkit.client.widget.composite.table.impl.images.PageFeatureImages;
import com.tandbergtv.neptune.widgettoolkit.client.widget.container.HorizontalContainer;

/**
 * The widget that contains all paging information used by the page feature handler for the table
 * 
 * @author Vijay Silva
 */
class PageFeatureWidget extends Composite {

	/* The Widgets */
	private final PageFeatureHandler<?, ?> handler;
	private final HorizontalContainer container;
	private final PageFeatureImages imageBundle;

	/* 'Page Size' widgets */
	private HorizontalContainer pageSizePanel;
	private Grid pageSizeGrid;

	/* 'Page Navigation' widgets */
	private HorizontalContainer pageNavigationPanel;
	private LabelWidget pageCountWidget;
	private ImageWidget leftLeftWidget;
	private ImageWidget leftWidget;
	private ImageWidget rightWidget;
	private ImageWidget rightRightWidget;

	/* 'Goto Page' widgets */
	private HorizontalContainer gotoPagePanel;
	private TextBoxWidget pageNumberWidget;

	/**
	 * Constructor
	 */
	public PageFeatureWidget(PageFeatureHandler<?, ?> handler) {
		this.handler = handler;
		this.container = new HorizontalContainer();
		this.imageBundle = GWT.create(PageFeatureImages.class);

		initialize();
		initWidget(container);
	}

	private void initialize() {
		/* Initialize the container */
		container.setSpacing(15);
		container.setHorizontalAlignment(ALIGN_CENTER);
		container.setVerticalAlignment(ALIGN_MIDDLE);

		/* Build and add the 'Page Size' panel */
		pageSizePanel = buildPageSizePanel();
		container.add(pageSizePanel);
		container.setCellHorizontalAlignment(pageSizePanel, ALIGN_LEFT);

		/* Build and add the 'Page Navigation' panel */
		pageNavigationPanel = buildPageNavigationPanel();
		container.add(pageNavigationPanel);
		container.setCellHorizontalAlignment(pageNavigationPanel, ALIGN_LEFT);

		/* Build and add the 'Goto Page' panel */
		gotoPagePanel = buildGotoPagePanel();
		container.add(gotoPagePanel);
		container.setCellHorizontalAlignment(gotoPagePanel, ALIGN_RIGHT);
	}

	/* Build the 'Page Size' panel */
	private HorizontalContainer buildPageSizePanel() {
		HorizontalContainer panel = new HorizontalContainer();
		panel.setVerticalAlignment(ALIGN_MIDDLE);

		int[] pageSizes = handler.getFeature().getPageSizes();
		int pageSizeCount = pageSizes.length;
		pageSizeGrid = new Grid(1, pageSizeCount);
		pageSizeGrid.setStyleName(TableConstants.STYLE_PAGE_SIZE_TABLE);
		for (int i = 0; i < pageSizeCount; ++i) {
			pageSizeGrid.setWidget(0, i, new LabelWidget(String.valueOf(pageSizes[i])));
		}
		pageSizeGrid.addClickHandler(new ClickHandler() {
			@Override
			public void onClick(ClickEvent event) {
				handlePageSizeGridClick(event);
			}
		});
		panel.add(pageSizeGrid);
		panel.setSpacing(TableConstants.PAGESIZE_TEXT_SPACING);

		/* add label after page options */
		LabelWidget suffixLabel = new LabelWidget("rows per page");
		suffixLabel.setStyleName(TableConstants.STYLE_GENERAL_TEXT);
		panel.add(suffixLabel);

		return panel;
	}

	/* Build the 'Page Navigation' panel */
	private HorizontalContainer buildPageNavigationPanel() {
		HorizontalContainer panel = new HorizontalContainer();
		panel.setVerticalAlignment(ALIGN_MIDDLE);
		panel.setSpacing(TableConstants.NAV_SPACING);

		leftLeftWidget = new ImageWidget();
		leftLeftWidget.setStyleName(TableConstants.STYLE_NAV_BUTTON);
		leftLeftWidget.addClickHandler(new ClickHandler() {
			@Override
			public void onClick(ClickEvent event) {
				handleGoToFirstPageClick(event);
			}
		});
		imageBundle.firstPageDisabled().applyTo(leftLeftWidget);
		panel.add(leftLeftWidget);

		leftWidget = new ImageWidget();
		leftWidget.setStyleName(TableConstants.STYLE_NAV_BUTTON);
		leftWidget.addClickHandler(new ClickHandler() {
			@Override
			public void onClick(ClickEvent event) {
				handleGoToPreviousPageClick(event);
			}
		});
		imageBundle.previousPageDisabled().applyTo(leftWidget);
		panel.add(leftWidget);

		pageCountWidget = new LabelWidget("");
		pageCountWidget.setStyleName(TableConstants.STYLE_GENERAL_TEXT);
		panel.add(pageCountWidget);

		rightWidget = new ImageWidget();
		rightWidget.setStyleName(TableConstants.STYLE_NAV_BUTTON);
		rightWidget.addClickHandler(new ClickHandler() {
			@Override
			public void onClick(ClickEvent event) {
				handleGoToNextPageClick(event);
			}
		});
		imageBundle.nextPageDisabled().applyTo(leftWidget);
		panel.add(rightWidget);

		rightRightWidget = new ImageWidget();
		rightRightWidget.setStyleName(TableConstants.STYLE_NAV_BUTTON);
		rightRightWidget.addClickHandler(new ClickHandler() {
			@Override
			public void onClick(ClickEvent event) {
				handleGoToLastPageClick(event);
			}
		});
		imageBundle.lastPageDisabled().applyTo(leftWidget);
		panel.add(rightRightWidget);

		return panel;
	}

	private HorizontalContainer buildGotoPagePanel() {
		HorizontalContainer panel = new HorizontalContainer();
		panel.setVerticalAlignment(ALIGN_MIDDLE);

		/* Add the 'goto' label */
		LabelWidget gotoPageLabelWidget = new LabelWidget("go to page:");
		gotoPageLabelWidget.setStyleName(TableConstants.STYLE_GENERAL_TEXT);
		panel.add(gotoPageLabelWidget);

		/* Add the page number text box */
		pageNumberWidget = new TextBoxWidget();
		pageNumberWidget.setMaxLength(5);
		pageNumberWidget.setVisibleLength(2);
		pageNumberWidget.addKeyPressHandler(new KeyPressHandler() {
			@Override
			public void onKeyPress(KeyPressEvent event) {
				handlePageNumberKeyPress(event);
			}
		});
		panel.add(pageNumberWidget);

		return panel;
	}

	// ========================================================================
	// ===================== EVENT HANDLING
	// ========================================================================

	/* The 'page size' grid click */
	private void handlePageSizeGridClick(ClickEvent event) {
		int clickedCellIndex = pageSizeGrid.getCellForEvent(event).getCellIndex();
		int length = handler.getFeature().getPageSizes()[clickedCellIndex];
		int start = 0;
		handler.updatePage(start, length);
	}

	/* The 'first page' button click */
	private void handleGoToFirstPageClick(ClickEvent event) {
		int length = handler.getFeature().getLength();
		int start = 0;
		handler.updatePage(start, length);
	}

	/* The 'previous page' button click */
	private void handleGoToPreviousPageClick(ClickEvent event) {
		int length = handler.getFeature().getLength();
		int start = handler.getFeature().getStart() - length;
		if (start < 0)
			start = 0;
		handler.updatePage(start, length);
	}

	/* The 'next page' button click */
	private void handleGoToNextPageClick(ClickEvent event) {
		int length = handler.getFeature().getLength();
		int start = handler.getFeature().getStart() + length;
		if (start < handler.getFeature().getTotalSize())
			handler.updatePage(start, length);
	}

	/* The 'last page' button click */
	private void handleGoToLastPageClick(ClickEvent event) {
		int length = handler.getFeature().getLength();
		int toMinus = handler.getFeature().getTotalSize() % length;
		if (toMinus == 0)
			toMinus = length;
		int start = handler.getFeature().getTotalSize() - toMinus;
		handler.updatePage(start, length);
	}

	/* The key press event for the page number widget */
	private void handlePageNumberKeyPress(KeyPressEvent event) {
		/* Determine if a complete page number has been entered */
		if (event.getCharCode() == KeyCodes.KEY_ENTER) {
			int pageNumber;
			try {
				pageNumber = Integer.parseInt(pageNumberWidget.getText());
			} catch (NumberFormatException e) {
				Window.alert("Please enter a valid page number");
				return;
			}

			if (pageNumber <= 0) {
				Window.alert("Please enter a valid page number");
				return;
			}

			// find total pages
			int totalPages = getTotalPages();

			// bug fix #3836: if the entered page number is later than last page, goto last page
			if (pageNumber > totalPages) {
				pageNumber = totalPages;
				setPageNumberText(pageNumber);
			}

			/* Notify the handler that the page number for the feature needs changing */
			int length = handler.getFeature().getLength();
			int start = (pageNumber - 1) * length;
			handler.updatePage(start, length);
		}
	}

	// ========================================================================
	// ===================== REFRESH
	// ========================================================================

	public void refresh() {
		/* Update the style for the page size grid cells */
		updatePageSizePanel();

		/* Update the buttons for the page navigation */
		updatePageNavigationPanel();

		/* The page number widget is unchanged, and must be explicitly cleared */
	}

	/**
	 * Update the page number shown in the page number widget
	 * 
	 * @param pageNumber The page number, or null to clear the widget
	 */
	public void setPageNumberText(Integer pageNumber) {
		String text = pageNumber == null ? "" : pageNumber.toString();
		this.pageNumberWidget.setText(text);
	}

	/*
	 * Update the panel containing the page size options based on the current feature state
	 */
	private void updatePageSizePanel() {
		int[] pageSizes = handler.getFeature().getPageSizes();
		int size = handler.getFeature().getLength();
		for (int i = 0; i < pageSizes.length; i++) {
			String style = (size == pageSizes[i]) ? TableConstants.STYLE_CURRENT_PAGE_SIZE_CELL
			        : TableConstants.STYLE_PAGE_SIZE_CELL;
			pageSizeGrid.getCellFormatter().setStyleName(0, i, style);
		}
	}

	/*
	 * Update the panel containing the page navigation options based on the current feature state
	 */
	private void updatePageNavigationPanel() {
		/* Update the page count */
		int recordsPerPage = handler.getFeature().getLength();
		int totalPages = this.getTotalPages();
		int currentPage = handler.getFeature().getStart() / recordsPerPage + 1;
		pageCountWidget.setText("page " + currentPage + " of " + totalPages);

		/* Update the page navigation images */
		AbstractImagePrototype image = null;
		boolean isLeftEnabled = currentPage != 1;
		image = isLeftEnabled ? imageBundle.firstPageEnabled() : imageBundle.firstPageDisabled();
		image.applyTo(leftLeftWidget);
		image = isLeftEnabled ? imageBundle.previousPageEnabled() : imageBundle
		        .previousPageDisabled();
		image.applyTo(leftWidget);

		boolean isRightEnabled = currentPage != totalPages;
		image = isRightEnabled ? imageBundle.lastPageEnabled() : imageBundle.lastPageDisabled();
		image.applyTo(rightRightWidget);
		image = isRightEnabled ? imageBundle.nextPageEnabled() : imageBundle.nextPageDisabled();
		image.applyTo(rightWidget);
	}

	/*
	 * Get the total number of pages
	 */
	private int getTotalPages() {
		int recordTotal = handler.getFeature().getTotalSize();
		int recordsPerPage = handler.getFeature().getLength();
		int totalPages = recordTotal / recordsPerPage;
		if (recordTotal % recordsPerPage > 0)
			totalPages++;

		return totalPages;
	}
}
