package com.tandbergtv.neptune.widgettoolkit.client.widget.composite.table.impl;

import java.util.HashMap;
import java.util.Map;

import com.google.gwt.core.client.GWT;
import com.google.gwt.event.dom.client.ClickEvent;
import com.google.gwt.event.dom.client.ClickHandler;
import com.google.gwt.user.client.ui.AbstractImagePrototype;
import com.google.gwt.user.client.ui.Label;
import com.tandbergtv.neptune.widgettoolkit.client.widget.basic.ImageWidget;
import com.tandbergtv.neptune.widgettoolkit.client.widget.composite.table.Column;
import com.tandbergtv.neptune.widgettoolkit.client.widget.composite.table.Record;
import com.tandbergtv.neptune.widgettoolkit.client.widget.composite.table.SortOrder;
import com.tandbergtv.neptune.widgettoolkit.client.widget.composite.table.TableConstants;
import com.tandbergtv.neptune.widgettoolkit.client.widget.composite.table.feature.SortFeature;
import com.tandbergtv.neptune.widgettoolkit.client.widget.composite.table.impl.images.SortFeatureImages;

public class SortFeatureHandler<K, R extends Record<K>> implements TableConstants, FeatureHandler {

	/* State */
	private final TableInternal<K, R> table;
	private final SortFeature<K, R> sortFeature;
	private final SortFeatureImages bundle;

	/* Only one column can be sortable, only one image needs to be maintained */
	private final ImageWidget sortImage;

	/* The anchor tokens added by the sort feature */
	private static final String SORT_ORDER_ANCHOR_TOKEN = "SortOrder";
	private static final String SORT_COLUMN_ANCHOR_TOKEN = "SortColumn";

	/**
	 * Constructor
	 * 
	 * @param table The table
	 * @param sortFeature The sort feature
	 */
	public SortFeatureHandler(TableInternal<K, R> table, SortFeature<K, R> sortFeature) {
		this.table = table;
		this.sortFeature = sortFeature;
		this.bundle = GWT.create(SortFeatureImages.class);
		this.sortImage = new ImageWidget();

		this.table.addListener(new TableInternal.Listener<K, R>() {
			@Override
			public void recordsUpdated() {
				refreshView();
			}
		});

		reset();
	}

	@Override
	public void init() {
		/* Set the default sort column and order */
		updateSortColumn(sortFeature.getDefaultSortColumn(), sortFeature.getDefaultSortOrder());

		/* For each sortable column, add a click listener to change the sort column / order */
		for (final Column<?, R> column : table.listVisibleColumns()) {
			if (sortFeature.isColumnSortable(column)) {
				final Label headerLabel = table.getHeaderLabelForColumn(column);
				headerLabel.setStyleName(STYLE_HEADER_SORTABLE_COL_TEXT);
				headerLabel.addClickHandler(new ClickHandler() {
					public void onClick(ClickEvent event) {
						handleColumnHeaderLabelClick(column, event);
					}
				});
			}
		}
	}

	@Override
	public void reset() {
		updateSortColumn(sortFeature.getDefaultSortColumn(), sortFeature.getDefaultSortOrder());
	}

	@Override
	public Map<String, String> getAnchorTokens() {
		Map<String, String> tokenMap = new HashMap<String, String>();

		/* Set the sort order for the anchor */
		if (sortFeature.getSortOrder() != null
		        && !sortFeature.getSortOrder().equals(sortFeature.getDefaultSortOrder())) {
			tokenMap.put(SORT_ORDER_ANCHOR_TOKEN, sortFeature.getSortOrder().name());
		}

		/* Set the sort column for the anchor */
		if (sortFeature.getSortColumn() != null
		        && !sortFeature.getSortColumn().equals(sortFeature.getDefaultSortColumn())) {
			tokenMap.put(SORT_COLUMN_ANCHOR_TOKEN, sortFeature.getSortColumn().getName());
		}

		return tokenMap;
	}

	// ========================================================================
	// ===================== FEATURE EVENTS BY USER
	// ========================================================================

	private void handleColumnHeaderLabelClick(Column<?, R> column, ClickEvent event) {
		SortOrder order = sortFeature.getDefaultSortOrder();
		if (sortFeature.getSortColumn().equals(column)) {
			SortOrder oldOrder = sortFeature.getSortOrder();
			order = (oldOrder != SortOrder.ASCENDING) ? SortOrder.ASCENDING : SortOrder.DESCENDING;
		}

		updateSortColumn(column, order);
		table.refresh();
	}

	// ========================================================================
	// ===================== UPDATING FEATURE / VIEW
	// ========================================================================

	/*
	 * Update the sort feature state (column and sort order) and update the view
	 */
	private void updateSortColumn(Column<?, R> column, SortOrder sortOrder) {
		sortFeature.setSortColumn(column);
		sortFeature.setSortOrder(sortOrder);
	}

	/*
	 * Refresh the view based on the state maintained by the sort feature
	 */
	private void refreshView() {
		Column<?, R> column = sortFeature.getSortColumn();
		SortOrder sortOrder = sortFeature.getSortOrder();

		/* There is only one sort image shown, remove it from current location and update it */
		sortImage.removeFromParent();
		updateSortImage(sortOrder);

		/* Add the image to the correct column */
		if (column != null && sortOrder != null)
			table.getHeaderColPanelForColumn(column).add(sortImage);
	}

	/*
	 * Update the image displayed for the sort ordering
	 */
	private void updateSortImage(SortOrder sortOrder) {
		boolean ascending = (sortOrder == SortOrder.ASCENDING);
		AbstractImagePrototype prototype = ascending ? bundle.ascending() : bundle.descending();
		prototype.applyTo(sortImage);
	}
}
