package com.tandbergtv.neptune.widgettoolkit.client.widget.composite;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.Queue;

import com.google.gwt.core.client.GWT;
import com.google.gwt.user.client.Command;
import com.google.gwt.user.client.ui.Composite;
import com.google.gwt.user.client.ui.MenuBar;
import com.google.gwt.user.client.ui.MenuItem;
import com.google.gwt.user.client.ui.TabPanel;
import com.tandbergtv.neptune.widgettoolkit.client.application.WidgetFactory;
import com.tandbergtv.neptune.widgettoolkit.client.i18n.NeptuneWidgetConstants;
import com.tandbergtv.neptune.widgettoolkit.client.widget.INeptuneWidget;
import com.tandbergtv.neptune.widgettoolkit.client.widget.basic.MenuBarWidget;
import com.tandbergtv.neptune.widgettoolkit.client.widget.container.HorizontalContainer;
import com.tandbergtv.neptune.widgettoolkit.client.widget.container.SimpleContainer;
import com.tandbergtv.neptune.widgettoolkit.client.widget.container.VerticalContainer;

/**
 * Similar to {@link TabPanel}, but has a tree of tabs instead of a flat list.
 * 
 * @author trybak
 */
public class TreeTabPanel extends Composite implements INeptuneWidget {
	private static final String STYLE_NAME = "nwt-TreeTabPanel";
	private static final String STYLE_MENU = "nwt-TreeTabPanel-menu";
	private static final String STYLE_MENU_LEVEL2 = "nwt-TreeTabPanel-menu2";
	private static final String STYLE_SEPARATOR = "nwt-TreeTabPanel-separator";
	private static final String STYLE_MENU_ITEM = "nwt-TreeTabPanel-menuItem";
	
	private final VerticalContainer mainContainer;
	private final MenuBarWidget tabMenuBar;
	private final SimpleContainer centerContainer;
	private final MenuNode rootNode;

	private WidgetFactory currentFactory;

	private final List<Listener> listeners;
	private final AnchorChangeListener frameworkListener;
	
	private NeptuneWidgetConstants constants = GWT.create(NeptuneWidgetConstants.class);

	/**
	 * Instantiate a tree tab panel with a given {@link AnchorChangeListener}. Anchor change
	 * messages initiated by the managed tabs will be forwarded to the framework listener, after the
	 * tab anchor is prepended.
	 * 
	 * @param frameworkListener an external {@link AnchorChangeListener} that will be informed of
	 *        any anchor change messages initiated by any of the managed tabs.
	 */
	public TreeTabPanel(AnchorChangeListener frameworkListener) {
		this.frameworkListener = frameworkListener;
		listeners = new ArrayList<Listener>();
		rootNode = new MenuNode("", "");
		tabMenuBar = new MenuBarWidget();
		tabMenuBar.setStylePrimaryName(STYLE_MENU);

		mainContainer = new VerticalContainer();
		mainContainer.setStyleName(STYLE_NAME);

		MenuItem menuItem = new MenuItem(constants.loading(), new Command() {
			public void execute() {
			}
		});
		menuItem.setStylePrimaryName(STYLE_MENU_ITEM);
		tabMenuBar.addItem(menuItem);

		HorizontalContainer menuContainer = new HorizontalContainer();
		menuContainer.setWidth("100%");
		
		menuContainer.add(tabMenuBar);
		menuContainer.setCellHorizontalAlignment(tabMenuBar, HorizontalContainer.ALIGN_LEFT);
		
		mainContainer.add(menuContainer);

		/* Add secondary menu */
		SimpleContainer secondaryMenuContainer = new SimpleContainer();
		secondaryMenuContainer.setStylePrimaryName(STYLE_MENU_LEVEL2);
		mainContainer.add(secondaryMenuContainer);

		/* Add separator container */
		SimpleContainer separatorContainer = new SimpleContainer();
		separatorContainer.setStylePrimaryName(STYLE_SEPARATOR);
		mainContainer.add(separatorContainer);
		
		centerContainer = new SimpleContainer();
		centerContainer.setStyleName("nwt-TreeTabPanel-widgetContainer");
		mainContainer.add(centerContainer);
		mainContainer.setCellHorizontalAlignment(centerContainer, HorizontalContainer.ALIGN_CENTER);
		initWidget(mainContainer);
	}

	public void addListener(Listener listener) {
		listeners.add(listener);
	}

	public MenuNode getRootMenu() {
		return rootNode;
	}

	public void doLayout() {
		tabMenuBar.clearItems();
		for (Node node : rootNode.subNodes)
			tabMenuBar.addItem(node.createMenuItem());
	}

	public abstract class Node {
		protected final String name;
		protected final String displayName;
		protected MenuNode parentNode;

		protected Node(String name) {
			this.name = name;
			this.displayName = name;
		}

		protected Node(String name, String displayName) {
			this.name = name;
			this.displayName = displayName;
		}

		protected abstract MenuItem createMenuItem();

		public abstract void accept(NodeVisitor visitor);

		/**
		 * @return the displayName
		 */
		public String getDisplayName() {
			return displayName;
		}
	}

	public interface NodeVisitor {
		void visit(MenuNode node);

		void visit(TabNode node);

		void visit(ActionNode node);
	}

	public final class MenuNode extends Node {
		private final List<Node> subNodes;
		private final Map<String, Node> nameToSubNode;

		private MenuNode(String name, String displayName) {
			super(name, displayName);
			subNodes = new ArrayList<Node>();
			nameToSubNode = new HashMap<String, Node>();
		}

		/**
		 * Returns a top-level menu node whose name matches the specified one
		 * 
		 * @param name
		 * @return
		 */
		public MenuNode getMenuNode(String name) {
			for (Node node : subNodes) {
				if (node instanceof MenuNode && node.name.equals(name))
					return (MenuNode) node;
			}
			
			return null;
		}
		
		@Override
		protected MenuItem createMenuItem() {
			MenuBar subMenu = new MenuBarWidget(true);
			subMenu.setStylePrimaryName(STYLE_MENU);
			for (Node node : subNodes)
				subMenu.addItem(node.createMenuItem());
			MenuItem item = new MenuItem(displayName, subMenu);
			item.setStylePrimaryName(STYLE_MENU_ITEM);
			return item;
		}

		public MenuNode addMenuNode(String name, String displayName) {
			MenuNode subMenuNode = new MenuNode(name, displayName);
			addNode(subMenuNode);
			return subMenuNode;
		}

		public TabNode addTabNode(String name, String displayName, WidgetFactory widgetFactory,
		        AnchorChangeListener tabListener) {
			TabNode panelTabNode = new TabNode(name, displayName, widgetFactory, tabListener);
			addNode(panelTabNode);

			return panelTabNode;

		}

		public TabNode addTabNode(String name, WidgetFactory widgetFactory) {
			return addTabNode(name, name, widgetFactory, null);
		}

		public ActionNode addActionNode(String name, String displayName, Runnable runnable) {
			ActionNode actionNode = new ActionNode(name, displayName, runnable);
			addNode(actionNode);
			return actionNode;
		}

		private void addNode(Node node) {
			subNodes.add(node);
			node.parentNode = this;
			nameToSubNode.put(node.name, node);
		}

		@Override
		public void accept(NodeVisitor visitor) {
			visitor.visit(this);
		}
	}

	public final class ActionNode extends Node {
		private final Runnable runnable;

		public ActionNode(String name, String displayName, Runnable runnable) {
			super(name, displayName);
			this.runnable = runnable;
		}

		@Override
		public void accept(NodeVisitor visitor) {
			visitor.visit(this);
		}

		@Override
		protected MenuItem createMenuItem() {
			MenuItem item = new MenuItem(displayName, new Command() {
				public void execute() {
					runnable.run();
				}
			});
			item.setStylePrimaryName(STYLE_MENU_ITEM);
			return item;
		}
	}

	public final class TabNode extends Node {
		private final WidgetFactory widgetFactory;
		private final AnchorChangeListener tabAnchorListener;
		private final AnchorChangeListener reverseAnchorListener;

		private TabNode(String name, String displayName, WidgetFactory widgetFactory, AnchorChangeListener tabListener) {
			super(name, displayName);
			this.widgetFactory = widgetFactory;
			this.tabAnchorListener = tabListener;

			reverseAnchorListener = new AnchorChangeListener() {
				public void anchorChanged(final String anchor) {
					StringBuilder realAnchor = new StringBuilder();
					realAnchor.append(getToken());
					if (!anchor.startsWith("?"))
						realAnchor.append('.');
					realAnchor.append(anchor);
					frameworkListener.anchorChanged(realAnchor.toString());
				}
			};
		}

		private TabNode(String name, WidgetFactory widgetFactory) {
			this(name, name, widgetFactory, null);
		}

		public String getToken() {
			StringBuilder token = new StringBuilder();
			constructToken(this, token);
			return token.toString();
		}

		public AnchorChangeListener getReverseAnchorChangeListener() {
			return reverseAnchorListener;
		}

		public String getName() {
			return name;
		}

		public WidgetFactory getWidgetFactory() {
			return widgetFactory;
		}

		private void constructToken(Node node, StringBuilder builder) {
			if (node == null)
				return;

			constructToken(node.parentNode, builder);

			if (builder.length() > 0)
				builder.append('.');
			builder.append(node.name);
		}

		@Override
		protected MenuItem createMenuItem() {
			MenuItem item = new MenuItem(displayName, new Command() {
				public void execute() {
					activatePanel(true);
				}
			});
			item.setStylePrimaryName(STYLE_MENU_ITEM);
			return item;
		}

		public void activatePanel(Queue<String> queue, String params, boolean notifyListener) {
			StringBuilder anchor = new StringBuilder();
			for (String bit : queue)
				anchor.append(bit).append('.');
			if (anchor.length() > 0)
				anchor.setLength(anchor.length() - 1);
			if (params != null)
				anchor.append(params);

			if (currentFactory != null && centerContainer.getWidget() != null) {
				currentFactory.release(centerContainer.getWidget());
			}

			currentFactory = widgetFactory;
			centerContainer.clear();
			centerContainer.add(widgetFactory.getInstance());
			if (tabAnchorListener != null)
				tabAnchorListener.anchorChanged(anchor.toString());
			if (notifyListener)
				for (Listener listener : listeners)
					listener.tabChanged(TabNode.this);
		}

		public void activatePanel(boolean notifyListener) {
			activatePanel(new LinkedList<String>(), null, notifyListener);
		}

		@Override
		public void accept(NodeVisitor visitor) {
			visitor.visit(this);
		}

	}

	public interface AnchorChangeListener {
		void anchorChanged(String anchor);
	}

	public void setActiveTab(String token) {
		int indexOfParams = token.indexOf('?');
		String params = null;
		if (indexOfParams != -1) {
			params = token.substring(indexOfParams);
			token = token.substring(0, indexOfParams);
		}
		Queue<String> queue = new LinkedList<String>();
		String[] tokenBits = token.split("\\.", -1);
		for (String tokenBit : tokenBits)
			queue.add(tokenBit);

		final NodeActivator locator = new NodeActivator(queue, params);
		rootNode.accept(locator);
	}

	public interface Listener {
		void tabChanged(TabNode tabNode);
	}

	private final class NodeActivator implements NodeVisitor {
		private final Queue<String> queue;
		private final String params;

		public NodeActivator(Queue<String> queue, String params) {
			this.queue = queue;
			this.params = params;
		}

		public void visit(MenuNode menuNode) {
			if (queue.peek() == null)
				return;
			String name = queue.remove();
			Node node = menuNode.nameToSubNode.get(name);
			if (node != null) {
				NodeActivator subLocator = new NodeActivator(queue, params);
				node.accept(subLocator);
			} else {
				/*
				 * If the sub node is not found for any reason, show the invalid URL page rather 
				 * than doing nothing.
				 */
				centerContainer.clear();
				centerContainer.add(new InvalidPagePanel(constants.invalidURL()));
			}
		}

		public void visit(TabNode panelTabNode) {
			panelTabNode.activatePanel(queue, params, false);
		}

		public void visit(ActionNode node) {
		}

	}
}
