/*
 * Created on Feb 8, 2010
 * 
 * (C) Copyright TANDBERG Television Inc.
 */

package com.tandbergtv.neptune.widgettoolkit.client.widget.dnd;

import com.allen_sauer.gwt.dnd.client.DragContext;
import com.allen_sauer.gwt.dnd.client.PickupDragController;
import com.google.gwt.user.client.ui.AbsolutePanel;
import com.google.gwt.user.client.ui.FlexTable;
import com.google.gwt.user.client.ui.HTML;
import com.google.gwt.user.client.ui.Widget;
import com.tandbergtv.neptune.widgettoolkit.client.widget.container.FlexTableContainer;

/**
 * Drag Controller used for drag-and-drop operations for rows of a HTML Table widget.
 * 
 * @author Vijay Silva
 */
public class TableRowDragController extends PickupDragController {

	private final FlexTable table;
	private int draggedRowIndex = -1;

	/**
	 * Constructor
	 * 
	 * @param boundaryPanel the desired boundary panel or <code>RootPanel.get()</code> if entire
	 *        document body is to be the boundary
	 * @param table The table for which row drag support is being added
	 */
	public TableRowDragController(AbsolutePanel boundaryPanel, FlexTable table) {
		super(boundaryPanel, false);
		setBehaviorDragProxy(true);
		setBehaviorMultipleSelection(false);
		this.table = table;
	}

	/**
	 * Get the table for which drag support is being added.
	 * 
	 * @return The table with drag support
	 */
	public FlexTable getTable() {
		return table;
	}

	/**
	 * Get the index of the row being dragged
	 * 
	 * @return the row index of the row being dragged
	 */
	public int getDraggedRowIndex() {
		return draggedRowIndex;
	}

	/**
	 * Provide a widget to show as the drag proxy so that the user is aware of the contents being
	 * dragged.
	 * 
	 * @see com.allen_sauer.gwt.dnd.client.PickupDragController#newDragProxy(com.allen_sauer.gwt.dnd.client.DragContext)
	 */
	@Override
	protected Widget newDragProxy(DragContext context) {
		this.draggedRowIndex = getDragRowIndex(context);

		/* no matching row found */
		if (draggedRowIndex == -1) {
			return null;
		}

		/* create the drag proxy widget - a table with a single row (the dragged row) */
		FlexTableContainer proxyTable = new FlexTableContainer();
		proxyTable.setStyleName(table.getStyleName());

		/* Limit the table width to the width of current table */
		proxyTable.setWidth(table.getOffsetWidth() + "px");

		/* add dragged row to the proxy widget */
		proxyTable.insertRow(0);
		String rowStyle = table.getRowFormatter().getStyleName(draggedRowIndex);
		proxyTable.getRowFormatter().setStyleName(0, rowStyle);

		/* add a copy of all the cells from the dragged row to the new proxy row */
		for (int column = 0; column < table.getCellCount(draggedRowIndex); column++) {
			HTML contents = new HTML(table.getHTML(draggedRowIndex, column));
			String cellStyle = table.getCellFormatter().getStyleName(draggedRowIndex, column);
			proxyTable.setWidget(0, column, contents);
			proxyTable.getCellFormatter().setStyleName(0, column, cellStyle);
		}

		return proxyTable;
	}

	/*
	 * Get the index of the row being dragged
	 */
	private int getDragRowIndex(DragContext context) {
		int draggedRow = -1;

		/* No widget being dragged */
		if (context.draggable == null)
			return draggedRow;

		/* determine the row being dragged */
		for (int row = 0; row < table.getRowCount(); row++) {
			for (int column = 0; column < table.getCellCount(row); column++) {
				Widget cellWidget = table.getWidget(row, column);
				if (isDescendant(context.draggable, cellWidget)) {
					draggedRow = row;
					break;
				}
			}
		}

		return draggedRow;
	}

	/*
	 * Determine if the input widget is equal to or descendant of the input ancestor widget
	 */
	private boolean isDescendant(Widget widget, Widget ancestor) {
		boolean descendant = false;

		if (widget == null || ancestor == null || widget.equals(table))
			descendant = false;
		else if (widget.equals(ancestor)) {
			descendant = true;
		} else {
			descendant = isDescendant(widget.getParent(), ancestor);
		}

		return descendant;
	}
}
