package com.tandbergtv.neptune.widgettoolkit.client.widget.composite.orderabletable;

import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;

import com.google.gwt.dom.client.Style;
import com.google.gwt.event.dom.client.ClickEvent;
import com.google.gwt.event.dom.client.ClickHandler;
import com.google.gwt.user.client.ui.AbsolutePanel;
import com.google.gwt.user.client.ui.CheckBox;
import com.google.gwt.user.client.ui.Composite;
import com.google.gwt.user.client.ui.RootPanel;
import com.google.gwt.user.client.ui.Widget;
import com.google.gwt.user.client.ui.HTMLTable.CellFormatter;
import com.tandbergtv.neptune.widgettoolkit.client.widget.basic.CheckBoxWidget;
import com.tandbergtv.neptune.widgettoolkit.client.widget.basic.ImageWidget;
import com.tandbergtv.neptune.widgettoolkit.client.widget.basic.LabelWidget;
import com.tandbergtv.neptune.widgettoolkit.client.widget.composite.table.TableConstants;
import com.tandbergtv.neptune.widgettoolkit.client.widget.container.FlexTableContainer;
import com.tandbergtv.neptune.widgettoolkit.client.widget.dnd.TableRowBeforeMoveEvent;
import com.tandbergtv.neptune.widgettoolkit.client.widget.dnd.TableRowDragController;
import com.tandbergtv.neptune.widgettoolkit.client.widget.dnd.TableRowMoveController;
import com.tandbergtv.neptune.widgettoolkit.client.widget.dnd.TableRowMoveEvent;
import com.tandbergtv.neptune.widgettoolkit.client.widget.dnd.TableRowMoveHandler;

public class OrderableTable extends Composite {

	// constants
	private static final String STYLE_NAME = "rules-OrderableTable";
	private static final String STYLE_HEADER_ROW = "rules-OrderableTable-headerRow";
	private static final String DRAG_HANDLE_IMAGE_URL = "cms_rules_ui/images/drag_drop_cue.gif";
	private static int HEADER_ROW = 0;
	private int DRAG_COLUMN;
	private int CHKBOX_COLUMN;

	// member variables
	private boolean hasChkBox;
	private boolean hasDragCol;
	private boolean isDirty;
	private int curColIndex;
	private FlexTableContainer table;
	private TableRowDragController rowDragger;
	private TableRowMoveController rowMoveController;
	private List<String> columns = new ArrayList<String>();
	private List<String> displayNames = new ArrayList<String>();
	private String idColName;
	private int startIndexOfDraggableRow;

	/**
	 * Create a rule listing table with the ID column name specified
	 * <p>
	 * Adds a drag column if <code>isOrderable</code> passed is <code>true</code>.
	 * 
	 * @param idColName
	 * @param isOrderable
	 */
	public OrderableTable(String idColName, boolean isOrderable,
			boolean showCheckBox, int startIndexOfDraggableRow) {
		this.hasDragCol = isOrderable; 
		this.isDirty = false;
		this.idColName = idColName;
		this.hasChkBox = false;
		this.curColIndex = 0;
		this.table = new FlexTableContainer();
		table.addStyleName(STYLE_NAME);
		this.hasChkBox = showCheckBox;
		this.startIndexOfDraggableRow = startIndexOfDraggableRow;
		
		// add features!
		table.getRowFormatter()
				.setStyleName(0, TableConstants.STYLE_HEADER_ROW);
		table.getRowFormatter().addStyleName(0, STYLE_HEADER_ROW);
		if (hasDragCol) {
			createDragDropHandler();
			addDragColumn();
		}
		if (hasChkBox) {
			addCheckboxColumn();
		}

		// add the bloody widgets
		this.initWidget(table);
	}

	/**
	 * Create a rule listing table with the ID column name specified
	 * 
	 * @param idColName
	 */
	public OrderableTable(String idColName) {
		this(idColName, true, true, 0);
	}

	/**
	 * Add a column to this table so we can match when adding column
	 * 
	 * @param name
	 * @param displayName
	 */
	public void addColumn(String displayName, String name) {
		int index = curColIndex++;
		String txt_style = TableConstants.STYLE_HEADER_NONSORTABLE_COL_TEXT;
		LabelWidget label = new LabelWidget(displayName);
		label.addStyleName(txt_style);
		table.setWidget(HEADER_ROW, index, label);
		columns.add(name);
		displayNames.add(displayName);
	}

	/**
	 * Add an item to the row and column. Column name must exist otherwise
	 * undefined behaviour will occur
	 * 
	 * @param row
	 *            row which data will be added to
	 * @param col
	 *            column which data will be added to
	 * @param value
	 *            value of the cell
	 * @param dragAllowed
	 * 			this is a draggable row           
	 */
	public void addItem(int row, String col, Widget value, String cellStyle) {
	 	addItem(row, col, value, cellStyle, true, false);
	}
	
	/**
	 * Add an item to the row and column. Column name must exist otherwise
	 * undefined behaviour will occur
	 * 
	 * @param row
	 *            row which data will be added to
	 * @param col
	 *            column which data will be added to
	 * @param value
	 *            value of the cell
	 * @param dradAllowed
	 * 			whether or not to show the draggable icon for this row           
	 */
	public void addItem(int row, String col, Widget value, String cellStyle,
			boolean dragAllowed, boolean isChecked) {
		// add data then add checkbox if master exists
		int index = this.columns.indexOf(col);
		int dataRow = row + HEADER_ROW + 1;
		String styleName = (dataRow % 2 == 0) ? TableConstants.STYLE_DATA_EVEN_ROW
				: TableConstants.STYLE_DATA_ODD_ROW;

		if (index == -1) {
			throw new RuntimeException("adding to unknown column: " + col);
		}

		CellFormatter cellFormatter = table.getCellFormatter();
		if (table.getRowCount() <= dataRow) {
			if(hasDragCol) {
				ImageWidget handle = new ImageWidget(DRAG_HANDLE_IMAGE_URL);
				rowDragger.makeDraggable(handle);
				table.setWidget(dataRow, DRAG_COLUMN, handle);
				cellFormatter.addStyleName(dataRow, DRAG_COLUMN, TableConstants.STYLE_DATACELL);
				cellFormatter.addStyleName(dataRow, DRAG_COLUMN, TableConstants.STYLE_DATACELL_ICON);
				if(dragAllowed) {
					handle.setVisible(true);
				} else {
					handle.setVisible(false);
				}
			}
			if (hasChkBox) {
				table.setWidget(dataRow, CHKBOX_COLUMN, createCheckbox(isChecked));
				cellFormatter.addStyleName(dataRow, CHKBOX_COLUMN, TableConstants.STYLE_DATACELL);
				cellFormatter.addStyleName(dataRow, CHKBOX_COLUMN, TableConstants.STYLE_DATACELL_CHECKBOX);
			}
			table.getRowFormatter().setStyleName(dataRow, styleName);
		}

		int colIndex = index + getColOffset();
		table.setWidget(dataRow, colIndex, value);
		cellFormatter.addStyleName(dataRow, colIndex, TableConstants.STYLE_DATACELL);
		if (cellStyle != null) {
			cellFormatter.addStyleName(dataRow, colIndex, cellStyle);
		}

		if (getIdColumn() == -1) {
			throw new RuntimeException("ID Column not set");
		}
	}

	private int getColOffset() {
		int colOffset = 0;
		if(hasDragCol)
			colOffset++;
		if(hasChkBox)
			colOffset++;	
		return colOffset;
	}

	/**
	 * Returns all table rows for this table
	 * 
	 * @return
	 */
	public List<TableRow> getTableRows() {
		List<TableRow> rows = new ArrayList<TableRow>();
		for (int i = HEADER_ROW + 1; i < table.getRowCount(); i++) {
			rows.add(getTableRow(i));
		}
		return rows;
	}

	/**
	 * Returns the row of given row count
	 * 
	 * @param rowIndex
	 * @return
	 */
	public TableRow getTableRow(int rowIndex) {
		TableRow row = new TableRow(columns, isChecked(rowIndex));
		for (int j = 0; j < getColumnCount(); j++) {
			String cellValue = getText(rowIndex, j);
			row.setValue(this.columns.get(j), cellValue);
		}
		return row;
	}

	/**
	 * make this clean
	 */
	public void resetDirty() {
		this.isDirty = false;
	}

	public boolean isDirty() {
		return this.isDirty;
	}

	/**
	 * Returns the number of column this table has
	 * 
	 * @return
	 */
	public int getColumnCount() {
		return this.curColIndex - CHKBOX_COLUMN - 1;
	}

	public int getRowCount() {
		return table.getRowCount() - (HEADER_ROW + 1);
	}

	/**
	 * Returns list of IDs of rows with enabled checkboxes. by default ID row is
	 * first column after
	 * 
	 * @return
	 */
	public List<Integer> getSelectedIds() {
		List<Integer> ids = new ArrayList<Integer>();
		int idColIndex = getIdColumn();
		for (int row = HEADER_ROW + 1; row < table.getRowCount(); row++) {
			String id = table.getText(row, idColIndex);
			CheckBox checkBox = (CheckBox) table.getWidget(row, CHKBOX_COLUMN);
			if (checkBox.getValue()) {
				ids.add(Integer.parseInt(id));
			}
		}
		return ids;
	}

	/**
	 * Returns true if this row has been checked
	 * 
	 * @param row
	 * @return
	 */
	public boolean isChecked(int row) {
		Widget w = table.getWidget(row + HEADER_ROW + 1, CHKBOX_COLUMN);
		return (w instanceof CheckBox) && ((CheckBox) w).getValue();
	}

	/**
	 * Iterator for each data row of this table
	 * 
	 * @return
	 */
	public Iterator<TableRow> rowIterator() {
		return new RowIterator(this);
	}

	public void clear() {
		// TODO: revamp to selectively delete rows properly
		int offset = curColIndex - columns.size();
		String hdr_style = TableConstants.STYLE_HEADER_ROW;
		String txt_style = TableConstants.STYLE_HEADER_NONSORTABLE_COL_TEXT;

		// clear table then really clear it
		int rowCnt = table.getRowCount();
		for (int row = 0; row < rowCnt; row++) {
			table.removeRow(0);
		}
		
		// rebuild header column
		if (this.hasChkBox) {
			table.setWidget(HEADER_ROW, CHKBOX_COLUMN, createTopCheckBox());
		}
		if(this.hasDragCol) {
			table.setWidget(HEADER_ROW, DRAG_COLUMN, new LabelWidget(""));
		}

		for (int i = 0; i < columns.size(); i++) {
			String display = displayNames.get(i);
			LabelWidget label = new LabelWidget(display);
			label.addStyleName(txt_style);
			table.setWidget(HEADER_ROW, i + offset, label);
		}
		table.getRowFormatter().setStyleName(0, hdr_style);
	}

	/*
	 * Add the checkbox column to this only if it has security issues
	 */
	private void addCheckboxColumn() {
		CheckBoxWidget chkBox = createTopCheckBox();
		CHKBOX_COLUMN = this.curColIndex++;
		table.setWidget(HEADER_ROW, CHKBOX_COLUMN, chkBox);
	}

	/*
	 * Add drag column which will be a blank nothing bitches
	 */
	private void addDragColumn() {
		DRAG_COLUMN = this.curColIndex++;
		LabelWidget dragHeaderLabel = new LabelWidget("");
		dragHeaderLabel
				.addStyleName(TableConstants.STYLE_HEADER_NONSORTABLE_COL_TEXT);
		table.setWidget(HEADER_ROW, DRAG_COLUMN, dragHeaderLabel);
	}

	/*
	 * returns the index of the id column
	 */
	private int getIdColumn() {
		return this.columns.indexOf(this.idColName) + getColOffset();
	}

	/*
	 * Drag drop handler.
	 */
	private void createDragDropHandler() {
		AbsolutePanel boundaryPanel = RootPanel.get();
		Style boundaryStyle = boundaryPanel.getElement().getStyle();
		boundaryStyle.setProperty("position", "relative");
		rowDragger = new TableRowDragController(boundaryPanel, this.table);
		rowMoveController = new TableRowMoveController(this.table);
		rowDragger.registerDropController(rowMoveController);
		TableRowMoveHandler handler = new TableRowMoveHandler() {
			@Override
			public void onBeforeRowMove(TableRowBeforeMoveEvent event) {
			    if (event.getTargetRowIndex() <= startIndexOfDraggableRow)
			    	event.cancel();
			}

			@Override
			public void onRowMove(TableRowMoveEvent event) {
				isDirty = true;

				int oldIndex = event.getOldRowIndex();
				int newIndex = event.getRowIndex();
				int start = (oldIndex < newIndex) ? oldIndex : newIndex;
				int end = (oldIndex >= newIndex) ? oldIndex : newIndex;
				restyleTableRows(start, end);
			}
		};
		addMoveHandler(handler);
	}

	private void restyleTableRows(int start, int end) {
		/* Go through table, and re-style the rows */
		for (int index = start; index <= end; index++) {
			String styleName = (index % 2 == 0) ? TableConstants.STYLE_DATA_EVEN_ROW
			        : TableConstants.STYLE_DATA_ODD_ROW;
			table.getRowFormatter().setStyleName(index, styleName);
		}
	}
	
	public void addMoveHandler(TableRowMoveHandler handler) {
		rowMoveController.addTableRowMoveHandler(handler);
	}

	/*
	 * create a checkbox
	 */
	private CheckBoxWidget createCheckbox(boolean isChecked) {
		CheckBoxWidget cBox = new CheckBoxWidget();
		cBox.setValue(isChecked);
		cBox.addClickHandler(new ClickHandler() {
			@Override
			public void onClick(ClickEvent event) {
				// uncheck the master box if anything clicked
				int i = HEADER_ROW;
				int j = CHKBOX_COLUMN;
				CheckBoxWidget topChk = (CheckBoxWidget) table.getWidget(i, j);
				topChk.setValue(false);
			}
		});
		return cBox;
	}

	/*
	 * the top check box needs to check and uncheck all rows
	 */
	private CheckBoxWidget createTopCheckBox() {
		CheckBoxWidget cbox = new CheckBoxWidget();
		cbox.addClickHandler(new ClickHandler() {
			public void onClick(ClickEvent event) {
				// check all rows to match value of top checkbox
				for (int i = 1; i < table.getRowCount(); i++) {
					int j = CHKBOX_COLUMN;
					CheckBoxWidget src = (CheckBoxWidget) event.getSource();
					CheckBoxWidget cur = (CheckBoxWidget) table.getWidget(i, j);
					cur.setValue(src.getValue());
				}
			}
		});
		return cbox;
	}

	/*
	 * Gets the text for rol and column accounting for header information
	 */
	private String getText(int row, int col) {
		return table.getText(row + HEADER_ROW + 1, col + CHKBOX_COLUMN + 1);
	}

	/**
	 * Private class to iterate through each table row
	 * 
	 * @author rchu
	 * 
	 */
	private class RowIterator implements Iterator<TableRow> {

		int cursor = 0;
		OrderableTable tbl;

		public RowIterator(OrderableTable table) {
			this.tbl = table;
		}

		@Override
		public boolean hasNext() {
			return cursor < tbl.getRowCount();
		}

		@Override
		public TableRow next() {
			return tbl.getTableRow(cursor++);
		}

		@Override
		public void remove() {
			// not implemented!
		}
	}

	public boolean isReorderable() {
		return hasDragCol;
	}
	
	public Widget getWidget(int row, String columnName) {
		int col = this.columns.indexOf(columnName);
		if(hasChkBox) {
			col++;
		}
		if(hasDragCol) {
			col++;
		}
		return table.getWidget(row, col);
	}
	
	public CheckBoxWidget getCheckBoxWidget(int row) {
		int col = 0;
		if(hasChkBox) {
			col++;
		}
		return (CheckBoxWidget) table.getWidget(row, col);
	}
}
