/*
 * Created on May 29, 2009
 * 
 * (C) Copyright TANDBERG Television Ltd.
 */

package com.tandbergtv.neptune.widgettoolkit.client.widget.datatype;

import com.google.gwt.event.dom.client.BlurEvent;
import com.google.gwt.event.dom.client.BlurHandler;
import com.google.gwt.event.dom.client.ChangeEvent;
import com.google.gwt.event.dom.client.ChangeHandler;
import com.google.gwt.event.dom.client.ClickEvent;
import com.google.gwt.event.dom.client.ClickHandler;
import com.google.gwt.event.dom.client.FocusEvent;
import com.google.gwt.event.dom.client.FocusHandler;
import com.google.gwt.event.dom.client.HasClickHandlers;
import com.google.gwt.event.dom.client.HasKeyDownHandlers;
import com.google.gwt.event.dom.client.HasKeyPressHandlers;
import com.google.gwt.event.dom.client.HasKeyUpHandlers;
import com.google.gwt.event.dom.client.HasMouseDownHandlers;
import com.google.gwt.event.dom.client.HasMouseMoveHandlers;
import com.google.gwt.event.dom.client.HasMouseOutHandlers;
import com.google.gwt.event.dom.client.HasMouseOverHandlers;
import com.google.gwt.event.dom.client.HasMouseUpHandlers;
import com.google.gwt.event.dom.client.HasMouseWheelHandlers;
import com.google.gwt.event.dom.client.KeyDownEvent;
import com.google.gwt.event.dom.client.KeyDownHandler;
import com.google.gwt.event.dom.client.KeyPressEvent;
import com.google.gwt.event.dom.client.KeyPressHandler;
import com.google.gwt.event.dom.client.KeyUpEvent;
import com.google.gwt.event.dom.client.KeyUpHandler;
import com.google.gwt.event.dom.client.MouseDownEvent;
import com.google.gwt.event.dom.client.MouseDownHandler;
import com.google.gwt.event.dom.client.MouseMoveEvent;
import com.google.gwt.event.dom.client.MouseMoveHandler;
import com.google.gwt.event.dom.client.MouseOutEvent;
import com.google.gwt.event.dom.client.MouseOutHandler;
import com.google.gwt.event.dom.client.MouseOverEvent;
import com.google.gwt.event.dom.client.MouseOverHandler;
import com.google.gwt.event.dom.client.MouseUpEvent;
import com.google.gwt.event.dom.client.MouseUpHandler;
import com.google.gwt.event.dom.client.MouseWheelEvent;
import com.google.gwt.event.dom.client.MouseWheelHandler;
import com.google.gwt.event.logical.shared.ValueChangeEvent;
import com.google.gwt.event.logical.shared.ValueChangeHandler;
import com.google.gwt.event.shared.HandlerRegistration;
import com.google.gwt.user.client.ui.HasName;
import com.tandbergtv.neptune.widgettoolkit.client.widget.basic.ListBoxWidget;
import com.tandbergtv.neptune.widgettoolkit.client.widget.datatype.converter.IValueConverter;
import com.tandbergtv.neptune.widgettoolkit.client.widget.datatype.converter.ValueFormatException;

/**
 * A list box widget where the values are of the specified data type.
 * 
 * @author Vijay Silva
 */
public class TypedListBoxWidget<DataType> extends DataTypeWidget<DataType> implements HasName,
        HasClickHandlers, HasKeyDownHandlers, HasKeyPressHandlers, HasKeyUpHandlers,
        HasMouseDownHandlers, HasMouseMoveHandlers, HasMouseOutHandlers, HasMouseOverHandlers,
        HasMouseUpHandlers, HasMouseWheelHandlers {

	/* State */
	private ListBoxWidget<DataType> listBox;
	private IValueConverter<DataType> valueConverter;
	private ListBoxEventHandler eventHandler = new ListBoxEventHandler();
	private boolean hasAdditionalValue = false;
	private boolean isAdditionalValueInvalid = false;

	/**
	 * Constructor. Creates a single-select list box.
	 * 
	 * @param converter The value converter to use when setting the list box with a value that is
	 *        not present in the list box.
	 */
	public TypedListBoxWidget(IValueConverter<DataType> converter) {
		this(converter, false);
	}
	
	/**
         * Constructor. Creates a single-select list box.
         * 
         * @param converter The value converter to use when setting the list box with a value that is
         *        not present in the list box.
         * @param isMultipleSelect true if the ListBox is multiple select, false otherwise
         */
        public TypedListBoxWidget(IValueConverter<DataType> converter, boolean isMultipleSelect) {
                this.valueConverter = converter;
                this.listBox = new ListBoxWidget<DataType>(isMultipleSelect);
                this.listBox.addChangeHandler(eventHandler);
                this.initWidget(this.listBox);
        }


	// ========================================================================
	// ===================== HAS NAME
	// ========================================================================

	@Override
	public String getName() {
		return listBox.getName();
	}

	@Override
	public void setName(String name) {
		listBox.setName(name);
	}

	// ========================================================================
	// ===================== EVENT HANDLERS
	// ========================================================================

	@Override
	public HandlerRegistration addClickHandler(ClickHandler handler) {
		return addDomHandler(handler, ClickEvent.getType());
	}

	@Override
	public HandlerRegistration addBlurHandler(BlurHandler handler) {
		return addDomHandler(handler, BlurEvent.getType());
	}

	@Override
	public HandlerRegistration addFocusHandler(FocusHandler handler) {
		return addDomHandler(handler, FocusEvent.getType());
	}

	@Override
	public HandlerRegistration addKeyDownHandler(KeyDownHandler handler) {
		return addDomHandler(handler, KeyDownEvent.getType());
	}

	@Override
	public HandlerRegistration addKeyPressHandler(KeyPressHandler handler) {
		return addDomHandler(handler, KeyPressEvent.getType());
	}

	@Override
	public HandlerRegistration addKeyUpHandler(KeyUpHandler handler) {
		return addDomHandler(handler, KeyUpEvent.getType());
	}

	@Override
	public HandlerRegistration addMouseDownHandler(MouseDownHandler handler) {
		return addDomHandler(handler, MouseDownEvent.getType());
	}

	@Override
	public HandlerRegistration addMouseMoveHandler(MouseMoveHandler handler) {
		return addDomHandler(handler, MouseMoveEvent.getType());
	}

	@Override
	public HandlerRegistration addMouseOutHandler(MouseOutHandler handler) {
		return addDomHandler(handler, MouseOutEvent.getType());
	}

	@Override
	public HandlerRegistration addMouseOverHandler(MouseOverHandler handler) {
		return addDomHandler(handler, MouseOverEvent.getType());
	}

	@Override
	public HandlerRegistration addMouseUpHandler(MouseUpHandler handler) {
		return addDomHandler(handler, MouseUpEvent.getType());
	}

	@Override
	public HandlerRegistration addMouseWheelHandler(MouseWheelHandler handler) {
		return addDomHandler(handler, MouseWheelEvent.getType());
	}

	// ========================================================================
	// ===================== FOCUSABLE
	// ========================================================================

	@Override
	public int getTabIndex() {
		return listBox.getTabIndex();
	}

	@Override
	public void setTabIndex(int index) {
		listBox.setTabIndex(index);
	}

	@Override
	public void setAccessKey(char key) {
		listBox.setAccessKey(key);
	}

	@Override
	public void setFocus(boolean focused) {
		listBox.setFocus(focused);
	}

	// ========================================================================
	// ===================== HAS VALUE
	// ========================================================================

	@Override
	public DataType getValue() {
		return listBox.getSelectedItem();
	}

	@Override
	public void setValue(DataType value) {
		this.setValue(value, false);
	}

	@Override
	public void setValue(DataType value, boolean fireEvents) {
		DataType oldValue = listBox.getSelectedItem();

		/* Remove any previously added value */
		this.removeAdditionalValue();

		/* Determine if the new value is in the list, else add to list */
		int index = listBox.getIndex(value);
		if (index == -1) {
			/* Add the value to the list */
			String itemName = valueConverter.getStringValue(value);
			putAdditionalValue(itemName, value, false);
			index = listBox.getIndex(value);
		}
		listBox.setSelectedIndex(index);

		if (fireEvents) {
			ValueChangeEvent.fireIfNotEqual(this, oldValue, value);
		}

		this.updateWidgetState();
	}

	/*
	 * Put an additional value as the first element in the list
	 */
	private void putAdditionalValue(String name, DataType value, boolean invalid) {
		/* Remove the previous additional value */
		removeAdditionalValue();

		/* Add the new additional value */
		listBox.insertItem(name, value, 0);
		this.hasAdditionalValue = true;
		this.isAdditionalValueInvalid = invalid;
	}

	/*
	 * Remove any previously added additional value
	 */
	private void removeAdditionalValue() {
		if (this.hasAdditionalValue) {
			listBox.removeItemAtIndex(0);
		}

		this.hasAdditionalValue = false;
		this.isAdditionalValueInvalid = false;
	}

	@Override
	public HandlerRegistration addValueChangeHandler(ValueChangeHandler<DataType> handler) {
		return this.addHandler(handler, ValueChangeEvent.getType());
	}

	// ========================================================================
	// ===================== LIST BOX METHODS
	// ========================================================================

	/**
	 * @see ListBoxWidget#addItem(String, Object)
	 */
	public void addItem(String itemName, DataType item) {
		listBox.addItem(itemName, item);
	}

	/**
	 * @see ListBoxWidget#insertItem(String, Object, int)
	 */
	public void insertItem(String itemName, DataType item, int index) {
		listBox.insertItem(itemName, item, index);
	}

	/**
	 * @see ListBoxWidget#setItem(String, Object, int)
	 */
	public void setItem(String itemName, DataType item, int index) {
		listBox.setItem(itemName, item, index);
	}

	/**
	 * @see ListBoxWidget#removeItem(Object)
	 */
	public void removeItem(DataType item) {
		listBox.removeItem(item);
	}

	/**
	 * @see ListBoxWidget#removeItemAtIndex(int)
	 */
	public void removeItemAtIndex(int index) {
		listBox.removeItemAtIndex(index);
	}

	/**
	 * @see ListBoxWidget#clear()
	 */
	public void clear() {
		listBox.clear();
	}

	/**
	 * @see ListBoxWidget#getItemCount()
	 */
	public int getItemCount() {
		return listBox.getItemCount();
	}

	/**
	 * @see ListBoxWidget#getItemText(int)
	 */
	public String getItemText(int index) {
		return listBox.getItemText(index);
	}

	/**
	 * @see ListBoxWidget#setItemText()
	 */
	public void setItemText(int index, String text) {
		listBox.setItemText(index, text);
	}

	/**
	 * @see ListBoxWidget#getItem(int)
	 */
	public DataType getItem(int index) {
		return listBox.getItem(index);
	}

	/**
	 * @see ListBoxWidget#setItem(int, Object)
	 */
	public void setItem(int index, DataType value) {
		listBox.setItem(index, value);
	}

	/**
	 * @see ListBoxWidget#getIndex(Object)
	 */
	public int getIndex(DataType item) {
		return listBox.getIndex(item);
	}

	/**
	 * @see ListBoxWidget#getSelectedIndex()
	 */
	public int getSelectedIndex() {
		return listBox.getSelectedIndex();
	}

	/**
	 * @see ListBoxWidget#setSelectedIndex(int)
	 */
	public void setSelectedIndex(int index) {
		listBox.setSelectedIndex(index);
	}

	/**
	 * @see ListBoxWidget#getSelectedItem()
	 */
	public DataType getSelectedItem() {
		return listBox.getSelectedItem();
	}

	/**
	 * @see ListBoxWidget#getSelectedItem()
	 */
	public void setSelectedItem(DataType item) {
		listBox.setSelectedItem(item);
	}

	/**
	 * @see ListBoxWidget#isItemSelected(int)
	 */
	public boolean isItemSelected(int index) {
		return listBox.isItemSelected(index);
	}

	/**
	 * @see ListBoxWidget#setItemSelected(int, boolean)
	 */
	public void setItemSelected(int index, boolean selected) {
		listBox.setItemSelected(index, selected);
	}

	/**
	 * @see ListBoxWidget#getVisibleItemCount()
	 */
	public int getVisibleItemCount() {
		return listBox.getVisibleItemCount();
	}

	/**
	 * @see ListBoxWidget#setVisibleItemCount(int)
	 */
	public void setVisibleItemCount(int visibleItems) {
		listBox.setVisibleItemCount(visibleItems);
	}

	/**
	 * @see ListBoxWidget#isMultipleSelect()
	 */
	public boolean isMultipleSelect() {
		return listBox.isMultipleSelect();
	}

	// ========================================================================
	// ===================== DATA TYPE WIDGET METHODS
	// ========================================================================

	@Override
	public boolean isValidValue() {
		boolean invalid = this.hasAdditionalValue && this.isAdditionalValueInvalid;
		return (!(invalid && listBox.getSelectedIndex() == 0));
	}

	@Override
	public String getTextValue() {
		return valueConverter.getStringValue(this.getSelectedItem());
	}

	@Override
	public void setTextValue(String textValue) {
		try {
			DataType value = valueConverter.getTypedValue(textValue);
			this.setValue(value);
		} catch (ValueFormatException vfe) {
			/* Add the invalid text value as an option and keep track of the value */
			putAdditionalValue(textValue, null, true);
			listBox.setSelectedIndex(0);
			this.updateWidgetState();
		}
	}

	@Override
	public boolean isReadOnly() {
		return listBox.isEnabled();
	}

	@Override
	public void setReadOnly(boolean readOnly) {
		listBox.setEnabled(!readOnly);
		
		// TODO: Fix ListBoxWidget: Call GWT's ListBox.setReadOnly()
		if(readOnly) {
			listBox.addStyleName("gwt-ListBox-readonly");
		}
		else {
			listBox.removeStyleName("gwt-ListBox-readonly");
		}
	}

	/**
	 * Sets whether this widget is enabled.
	 * 
	 * @see ListBoxWidget#setEnabled(boolean)
	 */
	@Override
	public void setEnabled(boolean enabled) {
		listBox.setEnabled(enabled);
	}

	/**
	 * Gets whether this widget is enabled.
	 * 
	 * @see ListBoxWidget#isEnabled()
	 */
	@Override
	public boolean isEnabled() {
		return listBox.isEnabled();
	}

	@Override
	protected void updateStyle() {
		if (isInErrorState())
			listBox.addStyleDependentName(ERROR_STYLE);
		else
			listBox.removeStyleDependentName(ERROR_STYLE);
	}

	@Override
	public String getCurrentToolTip() {
		return listBox.getTitle();
	}

	@Override
	protected void setCurrentToolTip(String toolTip) {
		listBox.setTitle(toolTip);
	}

	// ========================================================================
	// ===================== INTERNAL EVENT HANDLING
	// ========================================================================

	/*
	 * Event listener for change in value
	 */
	private final class ListBoxEventHandler implements ChangeHandler {

		@Override
		public void onChange(ChangeEvent event) {
			updateWidgetState();

			if (hasAdditionalValue && getSelectedIndex() != 0) {
				removeAdditionalValue();
			}

			ValueChangeEvent.fire(TypedListBoxWidget.this, getSelectedItem());
		}
	}
}
