/*
 * Created on Sep 10, 2009
 * 
 * (C) Copyright TANDBERG Television Ltd.
 */

package com.tandbergtv.neptune.widgettoolkit.client.widget.composite.table.impl;

import static com.google.gwt.user.client.ui.HasHorizontalAlignment.ALIGN_CENTER;
import static com.google.gwt.user.client.ui.HasHorizontalAlignment.ALIGN_LEFT;
import static com.google.gwt.user.client.ui.HasHorizontalAlignment.ALIGN_RIGHT;
import static com.google.gwt.user.client.ui.HasVerticalAlignment.ALIGN_MIDDLE;

import com.google.gwt.core.client.GWT;
import com.google.gwt.event.dom.client.ClickEvent;
import com.google.gwt.event.dom.client.ClickHandler;
import com.google.gwt.event.dom.client.KeyCodes;
import com.google.gwt.event.dom.client.KeyPressEvent;
import com.google.gwt.event.dom.client.KeyPressHandler;
import com.google.gwt.event.shared.HandlerRegistration;
import com.google.gwt.resources.client.ImageResource;
import com.google.gwt.user.client.Window;
import com.google.gwt.user.client.ui.Composite;
import com.google.gwt.user.client.ui.Grid;
import com.tandbergtv.neptune.widgettoolkit.client.widget.basic.ImageWidget;
import com.tandbergtv.neptune.widgettoolkit.client.widget.basic.LabelWidget;
import com.tandbergtv.neptune.widgettoolkit.client.widget.basic.TextBoxWidget;
import com.tandbergtv.neptune.widgettoolkit.client.widget.composite.table.TableConstants;
import com.tandbergtv.neptune.widgettoolkit.client.widget.composite.table.feature.PageFeature;
import com.tandbergtv.neptune.widgettoolkit.client.widget.composite.table.feature.event.HasPageNavigationHandler;
import com.tandbergtv.neptune.widgettoolkit.client.widget.composite.table.feature.event.HasPageResizeHandler;
import com.tandbergtv.neptune.widgettoolkit.client.widget.composite.table.feature.event.PageNavigationEvent;
import com.tandbergtv.neptune.widgettoolkit.client.widget.composite.table.feature.event.PageNavigationHandler;
import com.tandbergtv.neptune.widgettoolkit.client.widget.composite.table.feature.event.PageResizeEvent;
import com.tandbergtv.neptune.widgettoolkit.client.widget.composite.table.feature.event.PageResizeHandler;
import com.tandbergtv.neptune.widgettoolkit.client.widget.composite.table.impl.images.PageFeatureImages;
import com.tandbergtv.neptune.widgettoolkit.client.widget.container.HorizontalContainer;

/**
 * The widget that contains all paging information used by the page feature handler for the table
 * 
 * @author Vijay Silva
 */
public class PageFeatureWidget extends Composite implements HasPageResizeHandler,
        HasPageNavigationHandler {

	/* Input */
	private PageFeature input;

	/* The Widgets */
	private final HorizontalContainer container;
	private final PageFeatureImages imageBundle;

	/* 'Page Size' widgets */
	private HorizontalContainer pageSizePanel;
	private Grid pageSizeGrid;

	/* 'Page Navigation' widgets */
	private HorizontalContainer pageNavigationPanel;
	private LabelWidget pageCountWidget;
	private ImageWidget leftLeftWidget;
	private ImageWidget leftWidget;
	private ImageWidget rightWidget;
	private ImageWidget rightRightWidget;

	/* 'Goto Page' widgets */
	private HorizontalContainer gotoPagePanel;
	private TextBoxWidget pageNumberWidget;

	/* Styles */
	private static final String STYLE_NAME = "nwt-PageFeatureWidget";

	/* Constants */
	private static final int DEFAULT_PAGE_NUMBER = 1;

	/**
	 * Constructor
	 */
	public PageFeatureWidget() {
		this.container = new HorizontalContainer();
		container.setStylePrimaryName(STYLE_NAME);
		this.imageBundle = GWT.create(PageFeatureImages.class);

		initialize();
		initWidget(container);
	}

	private void initialize() {
		/* Initialize the container */
		container.setSpacing(15);
		container.setHorizontalAlignment(ALIGN_CENTER);
		container.setVerticalAlignment(ALIGN_MIDDLE);

		/* Build and add the 'Page Size' panel */
		pageSizePanel = buildPageSizePanel();
		container.add(pageSizePanel);
		container.setCellHorizontalAlignment(pageSizePanel, ALIGN_LEFT);

		/* Build and add the 'Page Navigation' panel */
		pageNavigationPanel = buildPageNavigationPanel();
		container.add(pageNavigationPanel);
		container.setCellHorizontalAlignment(pageNavigationPanel, ALIGN_LEFT);

		/* Build and add the 'Goto Page' panel */
		gotoPagePanel = buildGotoPagePanel();
		container.add(gotoPagePanel);
		container.setCellHorizontalAlignment(gotoPagePanel, ALIGN_RIGHT);
	}

	/* Build the 'Page Size' panel */
	private HorizontalContainer buildPageSizePanel() {
		HorizontalContainer panel = new HorizontalContainer();
		panel.setVerticalAlignment(ALIGN_MIDDLE);

		pageSizeGrid = new Grid(1, 0);
		pageSizeGrid.setStyleName(TableConstants.STYLE_PAGE_SIZE_TABLE);
		pageSizeGrid.addClickHandler(new ClickHandler() {
			@Override
			public void onClick(ClickEvent event) {
				handlePageSizeGridClick(event);
			}
		});
		panel.add(pageSizeGrid);
		panel.setSpacing(TableConstants.PAGESIZE_TEXT_SPACING);

		/* add label after page options */
		LabelWidget suffixLabel = new LabelWidget("rows per page");
		suffixLabel.setStyleName(TableConstants.STYLE_GENERAL_TEXT);
		panel.add(suffixLabel);

		return panel;
	}

	/* Build the 'Page Navigation' panel */
	private HorizontalContainer buildPageNavigationPanel() {
		HorizontalContainer panel = new HorizontalContainer();
		panel.setVerticalAlignment(ALIGN_MIDDLE);
		panel.setSpacing(TableConstants.NAV_SPACING);

		leftLeftWidget = new ImageWidget();
		leftLeftWidget.setStyleName(TableConstants.STYLE_NAV_BUTTON);
		leftLeftWidget.addClickHandler(new ClickHandler() {
			@Override
			public void onClick(ClickEvent event) {
				handleGoToFirstPageClick(event);
			}
		});
		leftLeftWidget.setResource(imageBundle.firstPageDisabled());
		panel.add(leftLeftWidget);

		leftWidget = new ImageWidget();
		leftWidget.setStyleName(TableConstants.STYLE_NAV_BUTTON);
		leftWidget.addClickHandler(new ClickHandler() {
			@Override
			public void onClick(ClickEvent event) {
				handleGoToPreviousPageClick(event);
			}
		});
		leftWidget.setResource(imageBundle.previousPageDisabled());
		panel.add(leftWidget);

		pageCountWidget = new LabelWidget("");
		pageCountWidget.setStyleName(TableConstants.STYLE_GENERAL_TEXT);
		panel.add(pageCountWidget);

		rightWidget = new ImageWidget();
		rightWidget.setStyleName(TableConstants.STYLE_NAV_BUTTON);
		rightWidget.addClickHandler(new ClickHandler() {
			@Override
			public void onClick(ClickEvent event) {
				handleGoToNextPageClick(event);
			}
		});
		rightWidget.setResource(imageBundle.nextPageDisabled());
		panel.add(rightWidget);

		rightRightWidget = new ImageWidget();
		rightRightWidget.setStyleName(TableConstants.STYLE_NAV_BUTTON);
		rightRightWidget.addClickHandler(new ClickHandler() {
			@Override
			public void onClick(ClickEvent event) {
				handleGoToLastPageClick(event);
			}
		});
		rightRightWidget.setResource(imageBundle.lastPageDisabled());
		panel.add(rightRightWidget);

		return panel;
	}

	private HorizontalContainer buildGotoPagePanel() {
		HorizontalContainer panel = new HorizontalContainer();
		panel.setVerticalAlignment(ALIGN_MIDDLE);

		/* Add the 'goto' label */
		LabelWidget gotoPageLabelWidget = new LabelWidget("go to page:");
		gotoPageLabelWidget.setStyleName(TableConstants.STYLE_GENERAL_TEXT);
		panel.add(gotoPageLabelWidget);

		/* Add the page number text box */
		pageNumberWidget = new TextBoxWidget();
		pageNumberWidget.setMaxLength(5);
		pageNumberWidget.setVisibleLength(2);
		pageNumberWidget.addKeyPressHandler(new KeyPressHandler() {
			@Override
			public void onKeyPress(KeyPressEvent event) {
				handlePageNumberKeyPress(event);
			}
		});
		panel.add(pageNumberWidget);

		return panel;
	}

	// ========================================================================
	// ===================== INPUT
	// ========================================================================

	/**
	 * Get the input used by the widget
	 * 
	 * @return The feature input
	 */
	public PageFeature getInput() {
		return input;
	}

	/**
	 * Set the input to use for this widget
	 * 
	 * @param input The feature input
	 */
	public void setInput(PageFeature input) {
		this.input = input;
	}

	// ========================================================================
	// ===================== EVENT REGISTRATION
	// ========================================================================

	@Override
	public HandlerRegistration addPageNavigationHandler(PageNavigationHandler handler) {
		return addHandler(handler, PageNavigationEvent.getType());
	}

	@Override
	public HandlerRegistration addPageResizeHandler(PageResizeHandler handler) {
		return addHandler(handler, PageResizeEvent.getType());
	}

	// ========================================================================
	// ===================== EVENT HANDLING
	// ========================================================================

	/* The 'page size' grid click */
	private void handlePageSizeGridClick(ClickEvent event) {
		/* If there is no input, no event is fired */
		if (getInput() == null)
			return;

		int clickedCellIndex = pageSizeGrid.getCellForEvent(event).getCellIndex();
		int size = getInput().getPageSizes()[clickedCellIndex];

		/* Ignore if page size does not change */
		if (size == getInput().getPageSize())
			return;

		/* Fire event indicating that the page size has changed */
		fireEvent(new PageResizeEvent(size, getInput()));
	}

	/* The 'first page' button click */
	private void handleGoToFirstPageClick(ClickEvent event) {
		/* If there is no input, no event is fired */
		if (this.getInput() == null)
			return;

		/* If at first page, ignore */
		if (getInput().getPageNumber() == 1)
			return;

		int pageNumber = 1;

		/* Fire event indicating that the page navigation required */
		fireEvent(new PageNavigationEvent(pageNumber, getInput()));
	}

	/* The 'previous page' button click */
	private void handleGoToPreviousPageClick(ClickEvent event) {
		/* If there is no input, no event is fired */
		if (this.getInput() == null)
			return;

		/* If at first page, ignore */
		if (getInput().getPageNumber() == 1)
			return;

		int pageCount = getTotalPages();
		int pageNumber = getInput().getPageNumber() - 1;
		if (pageNumber <= 0)
			pageNumber = 1;
		else if (pageCount > 0 && pageNumber > pageCount)
			pageNumber = pageCount;

		/* Fire event indicating that the page navigation required */
		fireEvent(new PageNavigationEvent(pageNumber, getInput()));
	}

	/* The 'next page' button click */
	private void handleGoToNextPageClick(ClickEvent event) {
		/* If there is no input, no event is fired */
		if (this.getInput() == null)
			return;

		/* If at last page, ignore */
		int pageCount = this.getTotalPages();
		if (pageCount > 0 && getInput().getPageNumber() == pageCount)
			return;

		int pageNumber = getInput().getPageNumber() + 1;
		if (pageCount > 0 && pageNumber > pageCount)
			pageNumber = pageCount;

		/* Fire event indicating that the page navigation required */
		fireEvent(new PageNavigationEvent(pageNumber, getInput()));
	}

	/* The 'last page' button click */
	private void handleGoToLastPageClick(ClickEvent event) {
		/* If there is no input, no event is fired */
		if (this.getInput() == null)
			return;

		/* If at last page, ignore */
		int pageCount = this.getTotalPages();
		if (pageCount > 0 && getInput().getPageNumber() == pageCount)
			return;

		if (pageCount <= 0)
			pageCount = 1;

		/* Fire event indicating that the page navigation required */
		fireEvent(new PageNavigationEvent(pageCount, getInput()));
	}

	/* The key press event for the page number widget */
	private void handlePageNumberKeyPress(KeyPressEvent event) {
		/* If there is no input, no event is fired */
		if (getInput() == null)
			return;

		/* Determine if a complete page number has been entered */
		if (event.getCharCode() == KeyCodes.KEY_ENTER) {
			int pageNumber = 0;
			try {
				pageNumber = Integer.parseInt(pageNumberWidget.getText());
			} catch (NumberFormatException e) {
				Window.alert("Please enter a valid page number");
				return;
			}

			if (pageNumber <= 0) {
				Window.alert("Please enter a valid page number");
				return;
			}

			// find total pages
			int totalPages = getTotalPages();

			// bug fix #3836: if the entered page number is later than last page, goto last page
			if (totalPages > 0 && pageNumber > totalPages) {
				pageNumber = totalPages;
				setPageNumberText(pageNumber);
			}

			/* Fire event indicating that the page navigation required */
			fireEvent(new PageNavigationEvent(pageNumber, getInput()));
		}
	}

	// ========================================================================
	// ===================== REFRESH VIEW
	// ========================================================================

	public void refresh() {
		/* Update the style for the page size grid cells */
		updatePageSizePanel();

		/* Update the buttons for the page navigation */
		updatePageNavigationPanel();

		/* Clear the page number widget */
		updatePageNumberText();
	}

	/**
	 * Update the page number shown in the page number widget
	 * 
	 * @param pageNumber The page number, or null to clear the widget
	 */
	public void setPageNumberText(Integer pageNumber) {
		String text = pageNumber == null ? "" : pageNumber.toString();
		this.pageNumberWidget.setText(text);
	}

	/*
	 * Update the page number text box
	 */
	private void updatePageNumberText() {
		int pageNumber = (getInput() != null) ? getInput().getPageNumber() : DEFAULT_PAGE_NUMBER;
		setPageNumberText(pageNumber);
	}

	/*
	 * Update the panel containing the page size options based on the current feature state
	 */
	private void updatePageSizePanel() {
		int[] pageSizes = (getInput() != null) ? getInput().getPageSizes() : new int[0];
		int pageSizesCount = (pageSizes != null) ? pageSizes.length : 0;
		int size = (getInput() != null) ? getInput().getPageSize() : 0;
		
		/* Redraw the page size grid */
		pageSizeGrid.resizeColumns(pageSizes.length);
		for (int i = 0; i < pageSizesCount; ++i) {
			pageSizeGrid.setWidget(0, i, new LabelWidget(String.valueOf(pageSizes[i])));
			String style = (size == pageSizes[i]) ? TableConstants.STYLE_CURRENT_PAGE_SIZE_CELL
			        : TableConstants.STYLE_PAGE_SIZE_CELL;
			pageSizeGrid.getCellFormatter().setStyleName(0, i, style);
		}
	}

	/*
	 * Update the panel containing the page navigation options based on the current feature state
	 */
	private void updatePageNavigationPanel() {
		/* Update the page count */
		int totalPages = this.getTotalPages();
		int currentPage = (getInput() != null) ? getInput().getPageNumber() : DEFAULT_PAGE_NUMBER;
		String pageCount = (totalPages <= 0) ? "-" : Integer.toString(totalPages);
		pageCountWidget.setText("page " + currentPage + " of " + pageCount);

		/* Update the page navigation images */
		ImageResource image = null;
		boolean isLeftEnabled = (currentPage > 1);
		image = isLeftEnabled ? imageBundle.firstPageEnabled() : imageBundle.firstPageDisabled();
		leftLeftWidget.setResource(image);
		image = isLeftEnabled ? imageBundle.previousPageEnabled() : imageBundle
		        .previousPageDisabled();
		leftWidget.setResource(image);

		boolean isRightEnabled = (totalPages <= 0 || currentPage < totalPages);
		image = isRightEnabled ? imageBundle.lastPageEnabled() : imageBundle.lastPageDisabled();
		rightRightWidget.setResource(image);
		image = isRightEnabled ? imageBundle.nextPageEnabled() : imageBundle.nextPageDisabled();
		rightWidget.setResource(image);
	}

	/*
	 * Get the total number of pages or -1 if the record count is not initialized.
	 */
	private int getTotalPages() {
		/* No input present, return -1 */
		if (getInput() == null)
			return -1;

		/* If no records present, return a page count of -1 */
		int recordCount = getInput().getRecordCount();
		if (recordCount < 0)
			return -1;

		int recordsPerPage = getInput().getPageSize();
		int totalPages = recordCount / recordsPerPage;
		if (recordCount % recordsPerPage > 0)
			totalPages++;

		if (totalPages <= 0)
			totalPages = 1;

		return totalPages;
	}
}
