package com.tandbergtv.neptune.widgettoolkit.client.widget.basic;

import java.util.Date;

import com.google.gwt.event.dom.client.BlurEvent;
import com.google.gwt.event.dom.client.BlurHandler;
import com.google.gwt.event.dom.client.ClickEvent;
import com.google.gwt.event.dom.client.ClickHandler;
import com.google.gwt.event.logical.shared.CloseEvent;
import com.google.gwt.event.logical.shared.CloseHandler;
import com.google.gwt.event.logical.shared.HasValueChangeHandlers;
import com.google.gwt.event.logical.shared.ValueChangeEvent;
import com.google.gwt.event.logical.shared.ValueChangeHandler;
import com.google.gwt.event.shared.HandlerRegistration;
import com.google.gwt.event.shared.HasHandlers;
import com.google.gwt.user.client.ui.Composite;
import com.google.gwt.user.client.ui.HasValue;
import com.google.gwt.user.client.ui.HorizontalPanel;
import com.google.gwt.user.client.ui.PopupPanel;
import com.google.gwt.user.client.ui.TextBox;
import com.google.gwt.user.datepicker.client.CalendarUtil;
import com.google.gwt.user.datepicker.client.DatePicker;
import com.tandbergtv.neptune.widgettoolkit.client.widget.INeptuneWidget;

public class DateTimeInputWidget extends Composite implements HasValue<Date> ,INeptuneWidget {
	  private static final String DATEPICKER_BUTTON_ICON = "neptune_widget_toolkit/js/ext/resources/images/default/shared/calendar.gif";
	  private static final String DATEPICKER_BUTTON_STYLENAME = "datePickerButton";
	  private static final String DATETIME_TEXTBOX_STYLENAME = "dateTimeField";
	  private static final String POPUP_STYLENAME = "dateBoxPopup";
	  private final PopupPanel popup;
	  private final TextBox dateTimeBox = new TextBox();
	  private final DatePicker picker;
	  private final PushButtonWidget datePickerButton;
	  private final HorizontalPanel dateTimePanel;
	  private Format format;

	  public DateTimeInputWidget (DatePicker picker, Date date, Format format) {
		    this.picker = picker;
		    this.popup = new PopupPanel(true);
		    assert format != null : "Error:You cannot construct a DateTimeInputWidget with a null format";
		    this.format = format;

		    popup.addAutoHidePartner(dateTimeBox.getElement());
		    popup.setWidget(picker);
		    popup.setStyleName(POPUP_STYLENAME);

			ImageWidget image = new ImageWidget(DATEPICKER_BUTTON_ICON);
		    datePickerButton = new PushButtonWidget(image);
		    datePickerButton.addStyleDependentName(DATEPICKER_BUTTON_STYLENAME);
		    
		    datePickerButton.addClickHandler(new ClickHandler() {
			    public void onClick(ClickEvent event) {
			    	if (isDatePickerShowing() == false)			    	
			    		dateTimeBox.setReadOnly(true);
			    		showDatePicker();
			    }
			});
		    
			dateTimePanel  = new HorizontalPanel();
			dateTimePanel.add(dateTimeBox);
			dateTimePanel.add(datePickerButton);
			initWidget(dateTimePanel);
		    
		    dateTimeBox.addStyleDependentName(DATETIME_TEXTBOX_STYLENAME);
		    
		    DateTimeInputWidgetHandler handler = new DateTimeInputWidgetHandler();
		    picker.addValueChangeHandler(handler);
		    dateTimeBox.addBlurHandler(handler);
		    popup.addCloseHandler(handler);
		    setValue(date);

	  }

	  @Override
	  public void addStyleDependentName(String styleSuffix) {
		  dateTimeBox.addStyleDependentName(styleSuffix);
	  }

	  @Override
	  public void removeStyleDependentName(String styleSuffix) {
		  dateTimeBox.removeStyleDependentName(styleSuffix);
	  }
	  
	  public TextBox getTextBox() {
		    return dateTimeBox;
	  }
	  
	  public int getTabIndex() {
		    return dateTimeBox.getTabIndex();
	  }
	  
	  public Format getFormat() {
		    return this.format;
	  }
	  
	  public int getCursorPos() {
		    return dateTimeBox.getCursorPos();
	  }
	  
	  public DatePicker getDatePicker() {
		    return picker;
	  }
	  
	  private void showDatePicker() {
		    Date current = parseDate(false);
		    if (current == null) {
		      current = new Date();
		    }
		    picker.setCurrentMonth(current);
		    popup.showRelativeTo(datePickerButton);
	  }
	  
	  private void hideDatePicker() {
		    popup.hide();
	  }

	  public boolean isDatePickerShowing() {
		    return popup.isShowing();
	  }
	  
	  private Date parseDate(boolean reportError) {
		    if (reportError) {
		      getFormat().reset(this, false);
		    }
		    String text = dateTimeBox.getText().trim();
		    return getFormat().parse(this, text, reportError);
	  }
	  
	  public void setValue(Date date) {
		    setValue(date, false);
	  }

	  public void setValue(Date date, boolean fireEvents) {
		    setValue(picker.getValue(), date, fireEvents);
	  }
	  
	  @Override
	  public Date getValue() {
		    return parseDate(true);
	   }

	  private void setValue(Date oldDate, Date date, boolean fireEvents) {
		    if (date != null) {
		      picker.setCurrentMonth(date);
		    }
		    picker.setValue(date, false);
		    format.reset(this, false);
		    dateTimeBox.setText(getFormat().format(this, date));

		    if (fireEvents) {
		      DateChangeEvent.fireIfNotEqualDates(this, oldDate, date);
		    }
	  }
	  
	  /**
	   * Sets the datetime textbox's 'access key'. This key is used (in conjunction with a
	   * browser-specific modifier key) to automatically focus the widget.
	   * 
	   * @param key the datetime textbox's access key
	   */
	  public void setAccessKey(char key) {
	    dateTimeBox.setAccessKey(key);
	  }

	  /**
	   * Sets whether the datetime box is enabled.
	   * 
	   * @param enabled is the box enabled
	   */
	  public void setEnabled(boolean enabled) {
	    dateTimeBox.setEnabled(enabled);
	  }

	  /**
	   * Explicitly focus/unfocus this widget. Only one widget can have focus at a
	   * time, and the widget that does will receive all keyboard events.
	   * 
	   * @param focused whether this widget should take focus or release it
	   */
	  public void setFocus(boolean focused) {
	    dateTimeBox.setFocus(focused);
	  }

	  /**
	   * Sets the datetime textbox's position in the tab index. If more than one widget has
	   * the same tab index, each such widget will receive focus in an arbitrary
	   * order. Setting the tab index to <code>-1</code> will cause this widget to
	   * be removed from the tab order.
	   * 
	   * @param index the datetime textbox's tab index
	   */
	  public void setTabIndex(int index) {
	    dateTimeBox.setTabIndex(index);
	  }

	  /**
	   * Sets the format used to control formatting and parsing of datetime in the
	   * textbox of this widget. If textbox is not empty, the contents of the textbox
	   * will be replaced with current contents in the new format.
	   * 
	   * @param format the new date format
	   */
	  public void setFormat(Format format) {
	    assert format != null : "A DateTimeInputWidget cannot have a null format";
	    if (this.format != format) {
	      Date date = getValue();

	      // This call lets the formatter do whatever other clean up is required to
	      // switch formatters.
	      //
	      this.format.reset(this, true);

	      // Now update the format and show the current date using the new format.
	      this.format = format;
	      setValue(date);
	    }
	  }
	
	 
	  private void updateDateFromTextBox() {
		    Date parsedDate = parseDate(true);
		    if (parsedDate != null) {
		      setValue(picker.getValue(), parsedDate,true);
		    }
	  }
	  
	  
	  @Override
	  public HandlerRegistration addValueChangeHandler(
				ValueChangeHandler<Date> handler) {
		    return addHandler(handler, ValueChangeEvent.getType());
	  }  		  
	  
	  public interface Format {

		    /**
		     * Formats the provided date. Note, a null date is a possible input.
		     * 
		     * @param dateTimeInput the dateTimeInput widget
		     * @param date the date to format
		     * @return the formatted date as a string
		     */
		    String format(DateTimeInputWidget dateTimeInput, Date date);

		    /**
		     * Parses the provided string as a date.
		     * 
		     * @param dateTimeInput the dateTimeInput widget
		     * @param text the string representing a date
		     * @param reportError should the formatter indicate a parse error to the
		     *          user?
		     * @return the date created, or null if there was a parse error
		     */
		    Date parse(DateTimeInputWidget dateTimeInput, String text, boolean reportError);

		    /**
		     * If the format did any modifications to the datetime box's styling, reset them
		     * now.
		     * 
		     * @param abandon true when the current format is being replaced by another
		     * @param dateTimeInput the dateTimeInput widget
		     */
		    void reset(DateTimeInputWidget dateBox, boolean abandon);
	  }

	  private class DateTimeInputWidgetHandler implements ValueChangeHandler<Date>, 
    		BlurHandler, CloseHandler<PopupPanel> {

		@Override
		public void onValueChange(ValueChangeEvent<Date> event) {
		      Date oldDate =  parseDate(false);
		      Date newDate = event.getValue();
		      if (oldDate != null) {
		    	  newDate.setHours(oldDate.getHours());
		    	  newDate.setMinutes(oldDate.getMinutes());
		    	  newDate.setSeconds(oldDate.getSeconds());
		      }
		      else {
		    	  newDate.setHours(0);
		    	  newDate.setMinutes(0);
		    	  newDate.setSeconds(0);		    	  
		      }
	    	  dateTimeBox.setReadOnly(false);
	    	  setValue(oldDate, newDate,true);
		      hideDatePicker();
		      dateTimeBox.setFocus(true);
		}
	
	    public void onClose(CloseEvent<PopupPanel> event) {
	    	  dateTimeBox.setReadOnly(false);
		      updateDateFromTextBox();
		}
			
	    public void onBlur(BlurEvent event) {
		      if (isDatePickerShowing() == false) {
		        updateDateFromTextBox();
		      }
		}
	    		
	  }
	  
	  static class DateChangeEvent extends ValueChangeEvent<Date> { 
		  public static <S extends HasValueChangeHandlers<Date> & HasHandlers> void fireIfNotEqualDates(S source, Date oldValue, Date newValue) {
			  if (ValueChangeEvent.shouldFire(source, oldValue, newValue)) {        
				  source.fireEvent(new DateChangeEvent(newValue));      
			  }    
		  }  
		  protected DateChangeEvent(Date value) {      
			  // The date must be copied in case one handler causes it to change.      
			  super(CalendarUtil.copyDate(value));    
		  }      
		  
		  @Override    
		  public Date getValue() {      
			  return CalendarUtil.copyDate(super.getValue());    
		  }  
	  }	  

}
