/*
 * Created on May 13, 2010
 * 
 * (C) Copyright TANDBERG Television Inc.
 */

package com.tandbergtv.neptune.widgettoolkit.client.application;

import java.util.Date;
import java.util.Map;

import com.google.gwt.i18n.client.DateTimeFormat;

/**
 * Provides access to all user preferences for all components. These methods should only be called
 * by the client-side code.
 * 
 * @author Vijay Silva
 */
public class ClientPreferenceManager {

	/* Cannot instantiate */
	private ClientPreferenceManager() {
	}

	public static Map<String, String> getAllPreferences() {
		return NeptuneApplication.getApplication().getComponentProperties();
	}

	/**
	 * Get the preference value
	 * 
	 * @param preferenceName The preference name
	 * @return The preference value
	 */
	public static String getPreference(String preferenceName) {
		Map<String, String> preferences = getAllPreferences();
		return (preferences != null) ? preferences.get(preferenceName) : null;
	}

	/**
	 * Get the string value for a preference, or get default value if no value is defined
	 * 
	 * @param preferenceName The preference name
	 * @param defaultValue The default value (can be null)
	 * @return The preference value
	 */
	public static String getStringPreference(String preferenceName, String defaultValue) {
		String value = getPreference(preferenceName);
		return (value != null) ? value : defaultValue;
	}

	/**
	 * Get the boolean value for a preference, or get default value if no value is defined. If a
	 * value is defined, uses {@link Boolean#parseBoolean(String)} to parse the value.
	 * 
	 * @param preferenceName The preference name
	 * @param defaultValue The default value (can be null)
	 * @return The preference value
	 */
	public static boolean getBooleanPreference(String preferenceName, Boolean defaultValue) {
		String value = getPreference(preferenceName);
		return (value != null) ? Boolean.parseBoolean(value) : defaultValue;
	}

	/**
	 * Get the date value for a preference, or return default value if no value is present or if the
	 * value is invalid.
	 * 
	 * @param preferenceName The preference name
	 * @param pattern The date string pattern
	 * @param defaultValue The default date value
	 * @return
	 */
	public static Date getDatePreference(String preferenceName, String pattern, Date defaultValue) {
		String value = getPreference(preferenceName);
		Date dateValue = null;
		if (value != null) {
			try {
				DateTimeFormat format = DateTimeFormat.getFormat(pattern);
				dateValue = format.parseStrict(value);
			} catch (RuntimeException e) {
			}
		}
		return (dateValue != null) ? dateValue : defaultValue;
	}

	/**
	 * Get integer value for a preference. If no value or invalid value is specified, replace with
	 * default value.
	 * 
	 * @param preferenceName The preference name
	 * @param defaultValue The default value to return if no value or invalid value is defined
	 * @return The preference value
	 */
	public static Integer getIntegerPreference(String preferenceName, Integer defaultValue) {
		return getIntegerPreference(preferenceName, defaultValue, null, null);
	}

	/**
	 * Get integer value for a preference. If no value or invalid value is specified, replace with
	 * default value. If value violates maximum or minimum limits, fix with the limit value.
	 * 
	 * @param preferenceName The preference name
	 * @param defaultValue The default value to return if no value or invalid value is defined
	 * @param minimumValue The minimum value limit to apply (or null to ignore)
	 * @param maximumValue The maximum value limit to apply (or null to ignore)
	 * @return The preference value
	 */
	public static Integer getIntegerPreference(String preferenceName, Integer defaultValue,
	        Integer minimumValue, Integer maximumValue) {
		String value = getPreference(preferenceName);
		Integer integerValue = null;
		if (value != null) {
			try {
				integerValue = new Integer(value);
			} catch (RuntimeException e) {
			}
		}

		return getNumberValue(integerValue, defaultValue, minimumValue, maximumValue);
	}

	/**
	 * Get long value for a preference. If no value or invalid value is specified, replace with
	 * default value.
	 * 
	 * @param preferenceName The preference name
	 * @param defaultValue The default value to return if no value or invalid value is defined
	 * @return The preference value
	 */
	public static Long getLongPreference(String preferenceName, Long defaultValue) {
		return getLongPreference(preferenceName, defaultValue, null, null);
	}

	/**
	 * Get long value for a preference. If no value or invalid value is specified, replace with
	 * default value. If value violates maximum or minimum limits, fix with the limit value.
	 * 
	 * @param preferenceName The preference name
	 * @param defaultValue The default value to return if no value or invalid value is defined
	 * @param minimumValue The minimum value limit to apply (or null to ignore)
	 * @param maximumValue The maximum value limit to apply (or null to ignore)
	 * @return The preference value
	 */
	public static Long getLongPreference(String preferenceName, Long defaultValue,
	        Long minimumValue, Long maximumValue) {
		Long longValue = null;
		String value = getPreference(preferenceName);
		if (value != null) {
			try {
				longValue = new Long(value);
			} catch (RuntimeException e) {
			}
		}

		return getNumberValue(longValue, defaultValue, minimumValue, maximumValue);
	}

	/**
	 * Get float value for a preference. If no value or invalid value is specified, replace with
	 * default value.
	 * 
	 * @param preferenceName The preference name
	 * @param defaultValue The default value to return if no value or invalid value is defined
	 * @return The preference value
	 */
	public static Float getFloatPreference(String preferenceName, Float defaultValue) {
		return getFloatPreference(preferenceName, defaultValue, null, null);
	}

	/**
	 * Get float value for a preference. If no value or invalid value is specified, replace with
	 * default value. If value violates maximum or minimum limits, fix with the limit value.
	 * 
	 * @param preferenceName The preference name
	 * @param defaultValue The default value to return if no value or invalid value is defined
	 * @param minimumValue The minimum value limit to apply (or null to ignore)
	 * @param maximumValue The maximum value limit to apply (or null to ignore)
	 * @return The preference value
	 */
	public static Float getFloatPreference(String preferenceName, Float defaultValue,
	        Float minimumValue, Float maximumValue) {
		Float floatValue = null;
		String value = getPreference(preferenceName);
		if (value != null) {
			try {
				floatValue = new Float(value);
			} catch (RuntimeException e) {
			}
		}

		return getNumberValue(floatValue, defaultValue, minimumValue, maximumValue);
	}

	/**
	 * Get double value for a preference. If no value or invalid value is specified, replace with
	 * default value.
	 * 
	 * @param preferenceName The preference name
	 * @param defaultValue The default value to return if no value or invalid value is defined
	 * @return The preference value
	 */
	public static Double getDoublePreference(String preferenceName, Double defaultValue) {
		return getDoublePreference(preferenceName, defaultValue, null, null);
	}

	/**
	 * Get double value for a preference. If no value or invalid value is specified, replace with
	 * default value. If value violates maximum or minimum limits, fix with the limit value.
	 * 
	 * @param preferenceName The preference name
	 * @param defaultValue The default value to return if no value or invalid value is defined
	 * @param minimumValue The minimum value limit to apply (or null to ignore)
	 * @param maximumValue The maximum value limit to apply (or null to ignore)
	 * @return The preference value
	 */
	public static Double getDoublePreference(String preferenceName, Double defaultValue,
	        Double minimumValue, Double maximumValue) {
		Double doubleValue = null;
		String value = getPreference(preferenceName);
		if (value != null) {
			try {
				doubleValue = new Double(value);
			} catch (RuntimeException e) {
			}
		}

		return getNumberValue(doubleValue, defaultValue, minimumValue, maximumValue);
	}

	/*
	 * Given the value, a default value, and a maximum and minimum limit, get the value to use
	 */
	private static <N extends Comparable<N>> N getNumberValue(N value, N defaultValue,
	        N minimumValue, N maximumValue) {
		if (value == null) {
			/* Use default value */
			value = defaultValue;
		} else if (minimumValue != null && value.compareTo(minimumValue) < 0) {
			/* Use minimum value */
			value = minimumValue;
		} else if (maximumValue != null && value.compareTo(maximumValue) > 0) {
			/* Use maximum value */
			value = maximumValue;
		}

		return value;
	}
}
