/*
 * Created on Aug 15, 2011
 * 
 * (C) Copyright Ericsson Television Inc.
 */
package com.tandbergtv.neptune.widgettoolkit.client.widget.servertime;

import java.util.Date;

import com.google.gwt.core.client.GWT;
import com.google.gwt.event.dom.client.ClickEvent;
import com.google.gwt.event.dom.client.ClickHandler;
import com.google.gwt.event.dom.client.DoubleClickEvent;
import com.google.gwt.event.dom.client.DoubleClickHandler;
import com.google.gwt.event.dom.client.GestureChangeEvent;
import com.google.gwt.event.dom.client.GestureChangeHandler;
import com.google.gwt.event.dom.client.GestureEndEvent;
import com.google.gwt.event.dom.client.GestureEndHandler;
import com.google.gwt.event.dom.client.GestureStartEvent;
import com.google.gwt.event.dom.client.GestureStartHandler;
import com.google.gwt.event.dom.client.HasClickHandlers;
import com.google.gwt.event.dom.client.HasDoubleClickHandlers;
import com.google.gwt.event.dom.client.HasGestureChangeHandlers;
import com.google.gwt.event.dom.client.HasGestureEndHandlers;
import com.google.gwt.event.dom.client.HasGestureStartHandlers;
import com.google.gwt.event.dom.client.HasMouseDownHandlers;
import com.google.gwt.event.dom.client.HasMouseMoveHandlers;
import com.google.gwt.event.dom.client.HasMouseOutHandlers;
import com.google.gwt.event.dom.client.HasMouseOverHandlers;
import com.google.gwt.event.dom.client.HasMouseUpHandlers;
import com.google.gwt.event.dom.client.HasMouseWheelHandlers;
import com.google.gwt.event.dom.client.HasTouchCancelHandlers;
import com.google.gwt.event.dom.client.HasTouchEndHandlers;
import com.google.gwt.event.dom.client.HasTouchMoveHandlers;
import com.google.gwt.event.dom.client.HasTouchStartHandlers;
import com.google.gwt.event.dom.client.MouseDownEvent;
import com.google.gwt.event.dom.client.MouseDownHandler;
import com.google.gwt.event.dom.client.MouseMoveEvent;
import com.google.gwt.event.dom.client.MouseMoveHandler;
import com.google.gwt.event.dom.client.MouseOutEvent;
import com.google.gwt.event.dom.client.MouseOutHandler;
import com.google.gwt.event.dom.client.MouseOverEvent;
import com.google.gwt.event.dom.client.MouseOverHandler;
import com.google.gwt.event.dom.client.MouseUpEvent;
import com.google.gwt.event.dom.client.MouseUpHandler;
import com.google.gwt.event.dom.client.MouseWheelEvent;
import com.google.gwt.event.dom.client.MouseWheelHandler;
import com.google.gwt.event.dom.client.TouchCancelEvent;
import com.google.gwt.event.dom.client.TouchCancelHandler;
import com.google.gwt.event.dom.client.TouchEndEvent;
import com.google.gwt.event.dom.client.TouchEndHandler;
import com.google.gwt.event.dom.client.TouchMoveEvent;
import com.google.gwt.event.dom.client.TouchMoveHandler;
import com.google.gwt.event.dom.client.TouchStartEvent;
import com.google.gwt.event.dom.client.TouchStartHandler;
import com.google.gwt.event.shared.HandlerRegistration;
import com.google.gwt.i18n.client.DateTimeFormat;
import com.google.gwt.i18n.client.HasDirection.Direction;
import com.google.gwt.i18n.client.TimeZone;
import com.google.gwt.i18n.shared.DirectionEstimator;
import com.google.gwt.i18n.shared.HasDirectionEstimator;
import com.google.gwt.user.client.Timer;
import com.google.gwt.user.client.ui.Composite;
import com.google.gwt.user.client.ui.HasAutoHorizontalAlignment;
import com.google.gwt.user.client.ui.HasWordWrap;
import com.tandbergtv.neptune.widgettoolkit.client.remote.NeptuneAsyncCallback;
import com.tandbergtv.neptune.widgettoolkit.client.widget.INeptuneWidget;
import com.tandbergtv.neptune.widgettoolkit.client.widget.basic.LabelWidget;
import com.tandbergtv.neptune.widgettoolkit.client.widget.servertime.internal.IServerTimeService;
import com.tandbergtv.neptune.widgettoolkit.client.widget.servertime.internal.IServerTimeServiceAsync;
import com.tandbergtv.neptune.widgettoolkit.client.widget.servertime.internal.ServerTime;

/**
 * Displays the current application server time. It goes to the server every hour to get the
 * application server time, and then keeps incrementing the time every second on the client itself,
 * until it's time to go the server next hour.
 * 
 * @author Vijay Silva
 */
public class ServerTimeLabel extends Composite implements HasClickHandlers, HasDoubleClickHandlers,
        HasMouseDownHandlers, HasMouseUpHandlers, HasMouseOutHandlers, HasMouseOverHandlers,
        HasMouseMoveHandlers, HasMouseWheelHandlers, HasTouchStartHandlers, HasTouchMoveHandlers,
        HasTouchEndHandlers, HasTouchCancelHandlers, HasGestureStartHandlers,
        HasGestureChangeHandlers, HasGestureEndHandlers, HasWordWrap, HasDirectionEstimator,
        HasAutoHorizontalAlignment, INeptuneWidget {

	private static final int TICK_PERIOD = 1000;
	private static final String STYLE_NAME = "nwt-ServerTimeLabel";

	/**
	 * The default pattern used when building the date time string to show in the label
	 */
	public static final String DEFAULT_PATTERN = "yyyy-MM-dd hh:mm:ss a";

	/* The RPC service */
	private final IServerTimeServiceAsync service = GWT.create(IServerTimeService.class);

	/* The format string */
	private DateTimeFormat format;

	/* The timer to trigger updates of the widget state */
	private final Timer timer;

	/* The widget and its required state */
	private final LabelWidget label;

	/* The current server time */
	private ServerTime serverTime;
	private Date tickingTime;

	/**
	 * Constructor. Use the default pattern for display
	 */
	public ServerTimeLabel() {
		this(DEFAULT_PATTERN);
	}

	/**
	 * Constructor. Use the specified pattern for display.
	 * 
	 * @param pattern The date time pattern to use for display
	 */
	public ServerTimeLabel(String pattern) {
		this(DateTimeFormat.getFormat(pattern));
	}

	/**
	 * Constructor. Use the specified format to build the string for display
	 * 
	 * @param format The date time format to use for building the display string
	 */
	public ServerTimeLabel(DateTimeFormat format) {
		/* Set the format */
		this.format = format;

		/* Initialize the widget */
		this.label = new LabelWidget();
		this.initWidget(this.label);
		this.addStyleName(STYLE_NAME);

		/* Initialize the timer */
		this.timer = new Timer() {
			@Override
			public void run() {
				updateServerTime();
			}
		};
	}

	// ========================================================================
	// ===================== WIDGET LOAD / UNLOAD AND ACCESS
	// ========================================================================

	/**
	 * Get the label widget hidden by this widget
	 * 
	 * @return The hidden label widget
	 */
	protected LabelWidget getLabel() {
		return this.label;
	}

	/*
	 * Start the timer when the widget is attached to the DOM to make server calls
	 */
	@Override
	protected void onLoad() {
		super.onLoad();

		/* Update every second */
		updateServerTime();
		this.timer.scheduleRepeating(TICK_PERIOD);
	}

	/*
	 * Stop the timer (ensuring that no server calls are made)
	 */
	@Override
	protected void onUnload() {
		/* Stop the timer and clear the timer state */
		this.timer.cancel();
		reset();

		super.onUnload();
	}

	// ========================================================================
	// ===================== HAS WORD WRAP
	// ========================================================================

	@Override
	public boolean getWordWrap() {
		return label.getWordWrap();
	}

	@Override
	public void setWordWrap(boolean wrap) {
		label.setWordWrap(wrap);
	}

	// ========================================================================
	// ===================== HAS DIRECTION ESTIMATOR
	// ========================================================================

	@Override
	public DirectionEstimator getDirectionEstimator() {
		return this.label.getDirectionEstimator();
	}

	@Override
	public void setDirectionEstimator(boolean enabled) {
		this.label.setDirectionEstimator(enabled);
	}

	@Override
	public void setDirectionEstimator(DirectionEstimator directionEstimator) {
		this.label.setDirectionEstimator(directionEstimator);
	}

	public Direction getTextDirection() {
		return this.label.getTextDirection();
	}

	// ========================================================================
	// ===================== HAS AUTO HORIZONTAL ALIGNMENT
	// ========================================================================

	@Override
	public HorizontalAlignmentConstant getHorizontalAlignment() {
		return this.label.getHorizontalAlignment();
	}

	@Override
	public void setHorizontalAlignment(HorizontalAlignmentConstant align) {
		this.label.setHorizontalAlignment(align);
	}

	@Override
	public AutoHorizontalAlignmentConstant getAutoHorizontalAlignment() {
		return this.label.getAutoHorizontalAlignment();
	}

	@Override
	public void setAutoHorizontalAlignment(AutoHorizontalAlignmentConstant autoAlignment) {
		this.label.setAutoHorizontalAlignment(autoAlignment);
	}

	// ========================================================================
	// ===================== EVENT HANDLING
	// ========================================================================

	@Override
	public HandlerRegistration addClickHandler(ClickHandler handler) {
		return addDomHandler(handler, ClickEvent.getType());
	}

	@Override
	public HandlerRegistration addDoubleClickHandler(DoubleClickHandler handler) {
		return addDomHandler(handler, DoubleClickEvent.getType());
	}

	@Override
	public HandlerRegistration addMouseDownHandler(MouseDownHandler handler) {
		return addDomHandler(handler, MouseDownEvent.getType());
	}

	@Override
	public HandlerRegistration addMouseUpHandler(MouseUpHandler handler) {
		return addDomHandler(handler, MouseUpEvent.getType());
	}

	@Override
	public HandlerRegistration addMouseOutHandler(MouseOutHandler handler) {
		return addDomHandler(handler, MouseOutEvent.getType());
	}

	@Override
	public HandlerRegistration addMouseOverHandler(MouseOverHandler handler) {
		return addDomHandler(handler, MouseOverEvent.getType());
	}

	@Override
	public HandlerRegistration addMouseMoveHandler(MouseMoveHandler handler) {
		return addDomHandler(handler, MouseMoveEvent.getType());
	}

	@Override
	public HandlerRegistration addMouseWheelHandler(MouseWheelHandler handler) {
		return addDomHandler(handler, MouseWheelEvent.getType());
	}

	@Override
	public HandlerRegistration addTouchStartHandler(TouchStartHandler handler) {
		return addDomHandler(handler, TouchStartEvent.getType());
	}

	@Override
	public HandlerRegistration addTouchMoveHandler(TouchMoveHandler handler) {
		return addDomHandler(handler, TouchMoveEvent.getType());
	}

	@Override
	public HandlerRegistration addTouchEndHandler(TouchEndHandler handler) {
		return addDomHandler(handler, TouchEndEvent.getType());
	}

	@Override
	public HandlerRegistration addTouchCancelHandler(TouchCancelHandler handler) {
		return addDomHandler(handler, TouchCancelEvent.getType());
	}

	@Override
	public HandlerRegistration addGestureStartHandler(GestureStartHandler handler) {
		return addDomHandler(handler, GestureStartEvent.getType());
	}

	@Override
	public HandlerRegistration addGestureChangeHandler(GestureChangeHandler handler) {
		return addDomHandler(handler, GestureChangeEvent.getType());
	}

	@Override
	public HandlerRegistration addGestureEndHandler(GestureEndHandler handler) {
		return addDomHandler(handler, GestureEndEvent.getType());
	}

	// ========================================================================
	// ===================== DATE / TIME HANDLING
	// ========================================================================

	/**
	 * Set the date time pattern to use when displaying the server time.
	 * 
	 * @param pattern The pattern or null to use the default pattern
	 */
	public void setDisplayPattern(String pattern) {
		if (pattern == null || pattern.isEmpty()) {
			pattern = DEFAULT_PATTERN;
		}

		setDisplayFormat(DateTimeFormat.getFormat(pattern));
	}

	/**
	 * Set the date time format to use when displaying the server time
	 * 
	 * @param format The date time format
	 */
	public void setDisplayFormat(DateTimeFormat format) {
		this.format = format;
		this.updateText();
	}

	/**
	 * Reset the time and view for this widget. The next tick is forced to be read from the server.
	 */
	public void reset() {
		this.serverTime = null;
		this.tickingTime = null;
		updateText();
	}

	/**
	 * Get the text displayed by this widget
	 * 
	 * @return The text displayed by the widget
	 */
	public String getText() {
		return this.label.getText();
	}

	/**
	 * Get the time zone for the server
	 * 
	 * @return The server time zone
	 */
	public TimeZone getServerTimeZone() {
		return (this.serverTime != null) ? this.serverTime.getTimeZone() : null;
	}

	/**
	 * Get the current time on the server
	 * 
	 * @return The date object with the current server time
	 */
	public Date getServerTime() {
		return (this.tickingTime != null) ? new Date(this.tickingTime.getTime()) : null;
	}

	/*
	 * Update the state with the current time, and update the view
	 */
	private void updateServerTime() {
		if (this.tickingTime != null) {
			tickTime();
		} else {
			updateTimeFromServer();
		}
	}

	/*
	 * Get the current date and time from the server
	 */
	private void updateTimeFromServer() {
		service.getCurrentTime(new NeptuneAsyncCallback<ServerTime>() {
			@Override
			public void onNeptuneFailure(Throwable caught) {
				reset();
			}

			@Override
			public void onNeptuneSuccess(ServerTime result) {
				setServerTime(result);
			}
		});
	}

	/*
	 * Set the current time
	 */
	private void setServerTime(ServerTime serverTime) {
		this.serverTime = serverTime;
		this.tickingTime = this.serverTime.getDate();
		updateText();
	}

	/*
	 * Server time is made to tick by a second
	 */
	private void tickTime() {
		/* Increase the ticking time by one second */
		long time = this.tickingTime.getTime() + TICK_PERIOD;
		this.tickingTime.setTime(time);

		/* If the new time is the start of a new hour, re-sync the clock with the server */
		@SuppressWarnings("deprecation")
		int seconds = this.tickingTime.getSeconds();
		@SuppressWarnings("deprecation")
		int minutes = this.tickingTime.getMinutes();
		if (seconds == 0 && minutes == 0) {
			updateTimeFromServer();
		} else {
			updateText();
		}
	}

	/*
	 * Update the text in this label to show the current time
	 */
	private void updateText() {
		String text = null;
		try {
			text = (this.tickingTime != null) ? this.format.format(tickingTime) : "";
		} catch (Exception e) {
			text = "Pattern: " + this.format.getPattern() + " is invalid.";
		}

		this.label.setText(text);
	}
}
