/*
 * Created on Sep 4, 2009
 * 
 * (C) Copyright TANDBERG Television Ltd.
 */

package com.tandbergtv.neptune.widgettoolkit.client.widget.datatype;

import com.google.gwt.event.dom.client.BlurEvent;
import com.google.gwt.event.dom.client.BlurHandler;
import com.google.gwt.event.dom.client.ChangeEvent;
import com.google.gwt.event.dom.client.ChangeHandler;
import com.google.gwt.event.dom.client.FocusEvent;
import com.google.gwt.event.dom.client.FocusHandler;
import com.google.gwt.event.logical.shared.SelectionEvent;
import com.google.gwt.event.logical.shared.SelectionHandler;
import com.google.gwt.event.logical.shared.ValueChangeEvent;
import com.google.gwt.event.logical.shared.ValueChangeHandler;
import com.google.gwt.event.shared.HandlerRegistration;
import com.google.gwt.user.client.ui.HasName;
import com.google.gwt.user.client.ui.HasText;
import com.tandbergtv.neptune.widgettoolkit.client.widget.basic.ComboBoxWidget;
import com.tandbergtv.neptune.widgettoolkit.client.widget.datatype.converter.IValueConverter;
import com.tandbergtv.neptune.widgettoolkit.client.widget.datatype.converter.ValueFormatException;

/**
 * The Combo Box widget which performs data type validation
 * 
 * @author Vijay Silva
 */
public class TypedComboBoxWidget<DataType> extends DataTypeWidget<DataType> implements HasName,
        HasText {

	/* State */
	private final ComboBoxWidget<DataType> widget;
	private final IValueConverter<DataType> valueConverter;
	private final EventHandler eventHandler = new EventHandler();

	/* Styles */
	private static final String COMBO_ERROR_STYLE = "TypedComboBoxWidget-DataTypeError";

	/**
	 * Create a new typed combo box widget where the user can enter values outside of the combo box
	 * options.
	 * 
	 * @param converter The value converter
	 */
	public TypedComboBoxWidget(IValueConverter<DataType> converter) {
		this.valueConverter = converter;
		this.widget = new ComboBoxWidget<DataType>();
		this.widget.addChangeHandler(eventHandler);
		this.widget.addSelectionHandler(eventHandler);
		this.initWidget(this.widget);
	}

	// ========================================================================
	// ===================== WIDGET ACCESS
	// ========================================================================

	/**
	 * Get the combo box
	 * 
	 * @return The combo box
	 */
	protected ComboBoxWidget<DataType> getComboBox() {
		return this.widget;
	}

	// ========================================================================
	// ===================== HAS NAME
	// ========================================================================

	@Override
	public String getName() {
		return widget.getName();
	}

	@Override
	public void setName(String name) {
		widget.setName(name);
	}

	// ========================================================================
	// ===================== HAS TEXT
	// ========================================================================

	@Override
	public String getText() {
		return widget.getText();
	}

	@Override
	public void setText(String text) {
		widget.setText(text);
	}

	// ========================================================================
	// ===================== HAS VALUE
	// ========================================================================

	/**
	 * Gets the typed value, or null if there was an error parsing the value
	 */
	@Override
	public DataType getValue() {
		DataType value = null;

		if (widget.isItemSelected()) {
			value = widget.getSelectedItem();
		} else {
			String textValue = widget.getText();
			try {
				/* Try getting the converted value */
				value = valueConverter.getTypedValue(textValue);
			} catch (ValueFormatException vfe) {
			}
		}

		return value;
	}

	@Override
	public void setValue(DataType value) {
		this.setValue(value, false);
	}

	@Override
	public void setValue(DataType value, boolean fireEvents) {
		DataType oldValue = widget.getSelectedItem();

		int index = widget.getIndex(value);
		if (index == -1) {
			String textValue = valueConverter.getStringValue(value);
			widget.setText(textValue);
		} else {
			widget.setSelectedItem(value);
		}

		if (fireEvents) {
			if (oldValue == null)
				ValueChangeEvent.fire(this, value);
			else
				ValueChangeEvent.fireIfNotEqual(this, oldValue, value);
		}

		this.updateWidgetState();
	}

	@Override
	public HandlerRegistration addValueChangeHandler(ValueChangeHandler<DataType> handler) {
		return this.addHandler(handler, ValueChangeEvent.getType());
	}

	// ========================================================================
	// ===================== FOCUSABLE
	// ========================================================================

	@Override
	public int getTabIndex() {
		return widget.getTabIndex();
	}

	@Override
	public void setTabIndex(int index) {
		widget.setTabIndex(index);
	}

	@Override
	public void setAccessKey(char key) {
		throw new RuntimeException("Operation: 'setAccessKey' is not supported.");
	}

	@Override
	public void setFocus(boolean focused) {
		if (focused) {
			widget.focus();
		} else {
			throw new RuntimeException("Operation: setFocus(false) is not supported.");
		}
	}

	// ========================================================================
	// ===================== HAS ALL FOCUS HANDLERS
	// ========================================================================

	@Override
	public HandlerRegistration addBlurHandler(BlurHandler handler) {
		return addDomHandler(handler, BlurEvent.getType());
	}

	@Override
	public HandlerRegistration addFocusHandler(FocusHandler handler) {
		return addDomHandler(handler, FocusEvent.getType());
	}

	// ========================================================================
	// ===================== DATA TYPE WIDGET ABSTRACT METHODS
	// ========================================================================

	@Override
	public boolean isValidValue() {
		/* If no item is selected, parse the text */
		if (!widget.isItemSelected()) {
			String textValue = widget.getText();
			try {
				/* Try getting the converted value */
				valueConverter.getTypedValue(textValue);
			} catch (ValueFormatException vfe) {
				return false;
			}
		}

		return true;
	}

	@Override
	public String getTextValue() {
		return widget.getText();
	}

	@Override
	public void setTextValue(String textValue) {
		/* Try and convert to the correct data type */
		try {
			DataType value = valueConverter.getTypedValue(textValue);
			setValue(value);
		} catch (ValueFormatException vfe) {
			widget.setText(textValue);
			this.updateWidgetState();
		}
	}

	@Override
	public boolean isReadOnly() {
		return !widget.isEnabled();
	}

	@Override
	public void setReadOnly(boolean readOnly) {
		widget.setReadOnly(readOnly);
		widget.setEnabled(!readOnly);
	}

	@Override
	public boolean isEnabled() {
		return widget.isEnabled();
	}

	@Override
	public void setEnabled(boolean enabled) {
		widget.setEnabled(enabled);
	}

	@Override
	protected void updateStyle() {
		if (isInErrorState())
			widget.addClass(COMBO_ERROR_STYLE);
		else
			widget.removeClass(COMBO_ERROR_STYLE);
	}

	@Override
	public String getCurrentToolTip() {
		return widget.getTitle();
	}

	@Override
	protected void setCurrentToolTip(String toolTip) {
		widget.setTitle(toolTip);
	}

	// ========================================================================
	// ===================== COMBO BOX WIDGET METHODS
	// ========================================================================

	/**
	 * @see ComboBoxWidget#expand()
	 */
	public void expand() {
		widget.expand();
	}

	/**
	 * @see ComboBoxWidget#collapse()
	 */
	public void collapse() {
		widget.collapse();
	}

	/**
	 * @see ComboBoxWidget#isExpanded()
	 */
	public boolean isExpanded() {
		return widget.isExpanded();
	}

	/**
	 * @see ComboBoxWidget#setEditable(boolean)
	 */
	public void setEditable(boolean editable) {
		widget.setEditable(editable);
	}

	/**
	 * @see ComboBoxWidget#getMinHeight()
	 */
	public int getMinHeight() {
		return widget.getMinHeight();
	}

	/**
	 * @see ComboBoxWidget#setMinHeight(int)
	 */
	public void setMinHeight(int minHeight) {
		widget.setMinHeight(minHeight);
	}

	/**
	 * @see ComboBoxWidget#setMinChars(int)
	 */
	public void setMinChars(int minChars) {
		widget.setMinChars(minChars);
	}

	/**
	 * @see ComboBoxWidget#getCaretPosition()
	 */
	public int[] getCaretPosition() {
		return widget.getCaretPosition();
	}

	/**
	 * @see ComboBoxWidget#setCaretPosition(int, int)
	 */
	public void setCaretPosition(int caretStart, int numToSelect) {
		widget.setCaretPosition(caretStart, numToSelect);
	}

	/**
	 * @see ComboBoxWidget#insertAtCaret(String)
	 */
	public void insertAtCaret(String text) {
		widget.insertAtCaret(text);
	}

	/**
	 * @see ComboBoxWidget#selectText()
	 */
	public void selectText() {
		widget.selectText();
	}

	/**
	 * @see ComboBoxWidget#selectText(int, int)
	 */
	public void selectText(int start, int end) {
		widget.selectText(start, end);
	}

	/**
	 * @see ComboBoxWidget#setMaxLength(int)
	 */
	public void setMaxLength(int maxLength) {
		widget.setMaxLength(maxLength);
	}

	/**
	 * @see ComboBoxWidget#setMaxLengthText(String)
	 */
	public void setMaxLengthText(String maxLengthText) {
		widget.setMaxLengthText(maxLengthText);
	}

	/**
	 * @see ComboBoxWidget#setMinLength(int)
	 */
	public void setMinLength(int minLength) {
		widget.setMinLength(minLength);
	}

	/**
	 * @see ComboBoxWidget#setMinLengthText(String)
	 */
	public void setMinLengthText(String minLengthText) {
		widget.setMinLengthText(minLengthText);
	}

	/**
	 * @see ComboBoxWidget#getHeight()
	 */
	public int getHeight() {
		return widget.getHeight();
	}

	/**
	 * @see ComboBoxWidget#setHeight(int)
	 */
	public void setHeight(int height) {
		widget.setHeight(height);
	}

	/**
	 * @see ComboBoxWidget#setHeight(String)
	 */
	public void setHeight(String height) {
		widget.setHeight(height);
	}

	/**
	 * @see ComboBoxWidget#getWidth()
	 */
	public int getWidth() {
		return widget.getWidth();
	}

	/**
	 * @see ComboBoxWidget#setWidth(int)
	 */
	public void setWidth(int width) {
		widget.setWidth(width);
	}

	/**
	 * @see ComboBoxWidget#setWidth(String)
	 */
	public void setWidth(String width) {
		widget.setWidth(width);
	}

	/**
	 * @see ComboBoxWidget#setSize(int, int)
	 */
	public void setSize(int width, int height) {
		widget.setSize(width, height);
	}

	/**
	 * @see ComboBoxWidget#setSize(String, String)
	 */
	public void setSize(String width, String height) {
		widget.setSize(width, height);
	}

	/**
	 * @see ComboBoxWidget#addClass(String)
	 */
	public void addClass(String cls) {
		widget.addClass(cls);
	}

	/**
	 * @see ComboBoxWidget#removeClass(String)
	 */
	public void removeClass(String cls) {
		widget.removeClass(cls);
	}

	/**
	 * @see ComboBoxWidget#setListClass(String)
	 */
	public void setListClass(String listClass) {
		widget.setListClass(listClass);
	}

	// ========================================================================
	// ===================== ITEM MANAGEMENT
	// ========================================================================

	/**
	 * @see ComboBoxWidget#addItem(String, Object)
	 */
	public void addItem(String itemName, DataType item) {
		widget.addItem(itemName, item);
	}

	/**
	 * @see ComboBoxWidget#insertItem(String, Object, int)
	 */
	public void insertItem(String itemName, DataType item, int index) {
		widget.insertItem(itemName, item, index);
	}

	/**
	 * @see ComboBoxWidget#setItem(String, Object, int)
	 */
	public void setItem(String itemName, DataType item, int index) {
		widget.setItem(itemName, item, index);
	}

	/**
	 * @see ComboBoxWidget#removeItem(Object)
	 */
	public void removeItem(DataType item) {
		widget.removeItem(item);
	}

	/**
	 * @see ComboBoxWidget#removeItemAtIndex(int)
	 */
	public void removeItemAtIndex(int index) {
		widget.removeItemAtIndex(index);
	}

	/**
	 * @see ComboBoxWidget#clear()
	 */
	public void clear() {
		widget.clear();
	}

	/**
	 * @see ComboBoxWidget#getItemCount()
	 */
	public int getItemCount() {
		return widget.getItemCount();
	}

	/**
	 * @see ComboBoxWidget#getItemText(int)
	 */
	public String getItemText(int index) {
		return widget.getItemText(index);
	}

	/**
	 * @see ComboBoxWidget#setItemText(int, String)
	 */
	public void setItemText(int index, String text) {
		widget.setItemText(index, text);
	}

	/**
	 * @see ComboBoxWidget#getItem(int)
	 */
	public DataType getItem(int index) {
		return widget.getItem(index);
	}

	/**
	 * @see ComboBoxWidget#setItem(int, Object)
	 */
	public void setItem(int index, DataType value) {
		widget.setItem(index, value);
	}

	/**
	 * @see ComboBoxWidget#getIndex(Object)
	 */
	public int getIndex(DataType item) {
		return widget.getIndex(item);
	}

	/**
	 * @see ComboBoxWidget#getSelectedIndex()
	 */
	public int getSelectedIndex() {
		return widget.getSelectedIndex();
	}

	/**
	 * @see ComboBoxWidget#setSelectedIndex(int)
	 */
	public void setSelectedIndex(int index) {
		widget.setSelectedIndex(index);
	}

	/**
	 * @see ComboBoxWidget#getSelectedItem()
	 */
	public DataType getSelectedItem() {
		return widget.getSelectedItem();
	}

	/**
	 * @see ComboBoxWidget#setSelectedItem(Object)
	 */
	public void setSelectedItem(DataType item) {
		widget.setSelectedItem(item);
	}

	/**
	 * @see ComboBoxWidget#isItemSelected()
	 */
	public boolean isItemSelected() {
		return widget.isItemSelected();
	}

	/**
	 * @see ComboBoxWidget#clearText()
	 */
	public void clearText() {
		widget.clearText();
	}

	// ========================================================================
	// ===================== INTERNAL EVENT HANDLING
	// ========================================================================

	/*
	 * Event listener for change in value
	 */
	private final class EventHandler implements ChangeHandler, SelectionHandler<DataType> {

		@Override
		public void onChange(ChangeEvent event) {
			updateWidgetState();
			DataType value = null;

			if (widget.isItemSelected()) {
				value = widget.getSelectedItem();
			} else {
				String textValue = widget.getText();
				try {
					/* Try getting the converted value */
					value = valueConverter.getTypedValue(textValue);
					String formattedValue = valueConverter.getStringValue(value);
					widget.setText(formattedValue);
				} catch (ValueFormatException vfe) {
				}
			}

			ValueChangeEvent.fire(TypedComboBoxWidget.this, value);
		}

		@Override
		public void onSelection(SelectionEvent<DataType> event) {
			updateWidgetState();
			DataType value = event.getSelectedItem();
			ValueChangeEvent.fire(TypedComboBoxWidget.this, value);
		}
	}
}
