/**
 * File Name	:	ResizableContainer.java
 * Created By	:	Sanjay Khattar (sanjay.khattar@ericsson.com)
 * Date Created	:	Aug 27, 2012 1:01:15 PM
 * Purpose		:	
 *
 * (c) Ericsson Television Inc. 
 */
package com.tandbergtv.neptune.widgettoolkit.client.widget.composite.resizablecontainer;

import java.util.ArrayList;
import java.util.List;

import com.google.gwt.core.client.Scheduler;
import com.google.gwt.core.client.Scheduler.RepeatingCommand;
import com.google.gwt.event.logical.shared.ResizeEvent;
import com.google.gwt.event.logical.shared.ResizeHandler;
import com.google.gwt.event.logical.shared.SelectionEvent;
import com.google.gwt.event.logical.shared.SelectionHandler;
import com.google.gwt.event.shared.HandlerRegistration;
import com.google.gwt.user.client.History;
import com.google.gwt.user.client.Window;
import com.google.gwt.user.client.ui.Widget;
import com.google.gwt.user.client.ui.Composite;
import com.tandbergtv.neptune.widgettoolkit.client.widget.basic.ButtonWidget;
import com.tandbergtv.neptune.widgettoolkit.client.widget.composite.messagearea.MessageArea;
import com.tandbergtv.neptune.widgettoolkit.client.widget.container.HorizontalContainer;
import com.tandbergtv.neptune.widgettoolkit.client.widget.container.ScrollContainer;
import com.tandbergtv.neptune.widgettoolkit.client.widget.container.TabContainer;
import com.tandbergtv.neptune.widgettoolkit.client.widget.container.VerticalContainer;

/**
 * A <code>ResizableContainer</code> is a re-sizable composite widget with a {@link MessageArea} 
 * on top of a Tabs container (@link TabContainer}. Multiple tabs can be added to the
 * Tab container. Each tab has a {@link Widget} <code>content</code> within a
 * {@link ScrollContainer} and a {@link HorizontalContainer} at the bottom, 
 * to which any number of widgets can be added. Typically, the latter will be used 
 * as a Buttons Panel to perform operations on the <code>content</code>.
 * <p>
 * The <code>ResizableContainer</code> resizes itself and it's children upon 
 * resizing of the container browser Window. When the size of the 
 * {@link MessageArea} or the bottom {@link HorizontalContainer} changes, 
 * because of messages or widgets becoming visible/invisible, the 
 * {@link ScrollContainer} size changes to accommodate the change keeping
 * the bottom panel fully visible and pinned at the bottom of the browser 
 * window.  
 * 
 * @author esakhat (sanjay.khattar@ericsson.com)
 *
 */
public class ResizableContainer extends Composite implements IResizable, IMessageListener {

	public static final String ANCHOR_TAB_PORTION_PARAMETER_NAME = "tab";
	
	protected static final int CMS_HEADER_MENU_SIZE_PX = 115;
	protected static final int BOTTOM_OFFSET_PX = 15;
	
	protected static final String STYLE_NAME_MAIN_CONTAINER = "nwt-reszc";
	protected static final String STYLE_NAME_MESSAGE_AREA = "nwt-reszc-messageArea";
	protected static final String STYLE_NAME_TAB_CONTAINER = "nwt-reszc-tabContainer";
	protected static final String STYLE_NAME_SCROLL_CONTAINER = "nwt-reszc-scrollContainer";
	protected static final String STYLE_NAME_BUTTONS_CONTAINER = "nwt-reszc-buttonsContainer";
	
	protected VerticalContainer mainContainer;
	protected MessageArea messageArea;
	protected TabContainer tabContainer;
	protected Widget content;
	protected ScrollContainer nonTabScrollContainer;
	protected HorizontalContainer nonTabButtonsContainer;
	
	protected HandlerRegistration windowRegistration = null;

	protected String tabTitle;
	protected String anchorPrefix;
	protected int offsetHeight = 0;
	protected boolean hasTabs = false;
	
	protected List<ResizableTab> resizableTabs  = new ArrayList<ResizableTab>();
	protected int selectedTabIndex = 0;
	
	protected List<Widget> heightOffsettingWidgets = new ArrayList<Widget>();

	/**
	 * Create a <code>ResizableContainer</code> given a {@link Widget} 
	 * <code>content</code>, a <code>tabTitle</code>, and an 
	 * <code>anchorPrefix</code>.
	 * <p>
	 * If <code>null</code> or empty <code>tabTitle</code> is provided, 
	 * a non-tabbed container is created, which is re-sizable just like 
	 * the tabbed one.
	 * <p>
	 * This constructor lets the <code>ResizableContainer</code> use it's 
	 * own internal {@link MessageArea} rather than requiring the client to 
	 * provide it.
	 * 
	 * @param content
	 * @param tabTitle
	 * @param anchorPrefix
	 */
	public ResizableContainer(Widget content, String tabTitle, String anchorPrefix) {	
		this(content, tabTitle, anchorPrefix, null);
	}
	
	/**
	 * Create a <code>ResizableContainer</code> given a {@link Widget} 
	 * <code>content</code>, a <code>tabTitle</code>, an 
	 * <code>anchorPrefix</code> and a {@link MessageArea}.
	 * <p>
	 * If <code>null</code> or empty <code>tabTitle</code> is provided, 
	 * a non-tabbed container is created, which is re-sizable just like 
	 * the tabbed one.
	 * <p>
	 * This constructor lets the <code>ResizableContainer</code> use a  
	 * client provided {@link MessageArea} rather than the default one.  
	 * 
	 * @param content
	 * @param tabTitle
	 * @param anchorPrefix
	 * @param extMessageArea
	 */
	public ResizableContainer(Widget content, String tabTitle, 
			String anchorPrefix, MessageArea extMessageArea) {	
		this.content = content;
		this.tabTitle = tabTitle;
		this.anchorPrefix = anchorPrefix;
		this.messageArea = extMessageArea;
		
		if (content == null) {
			throw new RuntimeException("Widget content cannot be null");
		}
		
		if (content instanceof IMessageEmitter) {
			IMessageEmitter messageEmitter = (IMessageEmitter) content;
			messageEmitter.registerMessageListener(this);
		}

		if ((tabTitle != null) && !tabTitle.trim().isEmpty()) {
			hasTabs = true;
		}
		
		mainContainer = new VerticalContainer();
		if (messageArea == null) {
			messageArea = new MessageArea();
		}
		mainContainer.add(messageArea);
		
		mainContainer.addStyleName(STYLE_NAME_MAIN_CONTAINER);
		messageArea.addStyleName(STYLE_NAME_MESSAGE_AREA);
		
		if (hasTabs) {
			tabContainer = new TabContainer();
			mainContainer.add(tabContainer);
			
			tabContainer.addStyleName(STYLE_NAME_TAB_CONTAINER);

			tabContainer.addSelectionHandler(new SelectionHandler<Integer>() {
				@Override
				public void onSelection(SelectionEvent<Integer> event) {
					selectedTabIndex = event.getSelectedItem();
					for (int i = 0; i < resizableTabs.size(); ++i) {
						resizableTabs.get(i).setSelected(
								i == event.getSelectedItem());
					}
				}
			});
			
			addTab(content, tabTitle);
			tabContainer.selectTab(0);
		}
		else {
			nonTabScrollContainer = new ScrollContainer();
			nonTabScrollContainer.setWidget(content);
			mainContainer.add(nonTabScrollContainer);
			
			nonTabButtonsContainer = new HorizontalContainer();
			mainContainer.add(nonTabButtonsContainer);
			
			nonTabButtonsContainer.addStyleName(STYLE_NAME_BUTTONS_CONTAINER);
		}
		
		initWidget(mainContainer);
		
		messageArea.reset();
		
		Scheduler.get().scheduleFixedDelay(new RepeatingCommand() {

			@Override
			public boolean execute() {
				updateSize();
				return false;
			}
			
		}, 500);
	}

	/**
	 * @return the messageArea
	 */
	public MessageArea getMessageArea() {
		return messageArea;
	}

	/**
	 * @return the content
	 */
	public Widget getContent() {
		return content;
	}

	/**
	 * @return the anchorPrefix
	 */
	public String getAnchorPrefix() {
		return anchorPrefix;
	}

	/**
	 * @param anchorPrefix the anchorPrefix to set
	 */
	public void setAnchorPrefix(String anchorPrefix) {
		this.anchorPrefix = anchorPrefix;
	}

	/**
	 * @return the offsetHeight
	 */
	public int getOffsetHeight() {
		return offsetHeight;
	}

	/**
	 * @param offsetHeight the offsetHeight to set
	 */
	public void setOffsetHeight(int offsetHeight) {
		this.offsetHeight = offsetHeight;
	}

	/**
	 * Add a new Tab to the re-sizable container. 
	 * <p>
	 * Both <code>content</code> and <code>tabTitle</code> are required. 
	 * <code>tabTitle</code> should in addition have at least one non-space character.
	 * 
	 * @param content
	 * @param tabTitle
	 */
	public void addTab(Widget content, String tabTitle) {
		
		if (content == null) {
			throw new RuntimeException("Widget content cannot be null");
		}
		
		if ((tabTitle == null) || (tabTitle.trim().isEmpty())) {
			throw new RuntimeException("String tabTitle cannot be null or empty");
		}
		
		if (hasTabs) {
			ResizableTab resizableTab = new ResizableTab(content, tabTitle.trim());
			addResizableTab(resizableTab);
		}
		else {
			throw new RuntimeException("This ResizableContainer does not have tabs");
		}
	}
	
	/**
	 * A convenience method that adds a button to the bottom container.
	 * <p>
	 * The added buttons are placed in same order in which they are added.
	 * 
	 * @param tabTitle
	 * @param buttonWidget
	 * 
	 * @see #addBottomWidget(String, Widget)
	 */
	public void addButton(String tabTitle, ButtonWidget buttonWidget) {
		addBottomWidget(tabTitle, buttonWidget);
	}
	
	/**
	 * Adds a widget to the bottom container.
	 * <p>
	 * The added widgets are placed in same order in which they are added.
	 * 
	 * @param tabTitle
	 * @param Widget
	 */
	public void addBottomWidget(String tabTitle, Widget Widget) {
		if (hasTabs) {
			ResizableTab resizableTab = getResizableTab(tabTitle);
			if (resizableTab != null) {
				resizableTab.buttonsContainer.add(Widget);
			} else {
				throw new RuntimeException("No tab with tabTitle '" + tabTitle
						+ "' exists");
			}
		}
		else {
			nonTabButtonsContainer.add(Widget);
		}
	}
	
	/**
	 * Select the tab with the given <code>tabTitle</code>.
	 * 
	 * @param tabTitle
	 */
	public void selectTab(String tabTitle) {
		if (hasTabs) {
			int tabIndex = getTabIndex(tabTitle);
			if (tabIndex >= 0) {
				tabContainer.selectTab(tabIndex);
			} else {
				throw new RuntimeException("No tab with tabTitle '" + tabTitle
						+ "' exists");
			}
		}
		else {
			throw new RuntimeException("This ResizableContainer does not have tabs");
		}
	}
	
	@Override
	public void updateSize() {
		if (hasTabs) {
			resizableTabs.get(selectedTabIndex).updateScrollPanelSize();
		}
		else {
			updateNonTabScrollContainerSize();
		}
	}
	
	/**
	 * Add the given widget's height to offsetHeight while calculating scrollContainer height.
	 * 
	 * @param w widget to be added when calculating offset height
	 */
	public void addHeightOffsettingWidget(Widget w) {
		if (w != null) {
			heightOffsettingWidgets.add(w);
		}
	}

	@Override
	public void setInfoMessage(String infoMessage) {
		messageArea.setInfoMessage(infoMessage);
		updateSize();
	}

	@Override
	public void setErrorMessage(String errorMessage) {
		messageArea.setErrorMessage(errorMessage);
		updateSize();
	}

	@Override
	public void setErrorHTML(String errorHTML) {
		messageArea.setErrorHTML(errorHTML);
		updateSize();
	}
	
	/**
	 * 
	 * @param resizableTab
	 */
	protected void addResizableTab(ResizableTab resizableTab) {
		if (hasTabs) {
			resizableTabs.add(resizableTab);
			tabContainer.add(resizableTab.getVerticalContainer(),
					resizableTab.getTabTitle());
		}
		else {
			throw new RuntimeException("This ResizableContainer does not have tabs");
		}
	}
	
	/**
	 * 
	 * @param tabTitle
	 * @return
	 */
	protected ResizableTab getResizableTab(String tabTitle) {
		ResizableTab resizableTab = null;
		
		if (hasTabs) {
			int tabIndex = getTabIndex(tabTitle);
			if (tabIndex >= 0) {
				resizableTab = resizableTabs.get(tabIndex);
			}
		}
		else {
			throw new RuntimeException("This ResizableContainer does not have tabs");
		}

		return resizableTab;
	}
	
	/**
	 * 
	 * @param tabTitle
	 * @return
	 */
	protected int getTabIndex(String tabTitle) {
		int tabIndex = -1;
		
		if (hasTabs) {
			if ((tabTitle != null) && !tabTitle.trim().isEmpty()) {
				for (int i = 0; i < resizableTabs.size(); i++) {
					ResizableTab resizableTab = resizableTabs.get(i);
					String resizableTabTitle = resizableTab.getTabTitle();
					if (tabTitle.equals(resizableTabTitle)) {
						tabIndex = i;
						break;
					}
				}
			}
		}
		else {
			throw new RuntimeException("This ResizableContainer does not have tabs");
		}
		
		return tabIndex;
	}

	/**
	 * 
	 */
	protected void updateNonTabScrollContainerSize() {
		int height = Window.getClientHeight()
				- CMS_HEADER_MENU_SIZE_PX
				- offsetHeight
				- messageArea.getOffsetHeight()
				- nonTabButtonsContainer.getOffsetHeight()
				- BOTTOM_OFFSET_PX;
			
			for (Widget w : heightOffsettingWidgets) {
				height -= w.getOffsetHeight(); 
			}
			
			nonTabScrollContainer.setHeight(height + "px");
	}
	
	// ==============================================================
	// ===================== WIDGET OVERRIDES
	// ==============================================================

	@Override
	protected void onLoad() {
		super.onLoad();

		windowRegistration = Window.addResizeHandler(new ResizeHandler() {
			public void onResize(ResizeEvent event) {
				updateSize();
			}
		});
	};

	@Override
	protected void onUnload() {
		windowRegistration.removeHandler();
		windowRegistration = null;

		super.onUnload();
	}
	
	/**
	 * Inner class which contains the widgets comprising a 
	 * re-sizable tab.
	 * 
	 * @author esakhat (sanjay.khattar@ericsson.com)
	 *
	 */
	protected class ResizableTab {
		
		private String tabTitle;
		private Widget tabContent;
		
		private VerticalContainer verticalContainer;
		private ScrollContainer scrollContainer;
		private HorizontalContainer buttonsContainer;
		
		/**
		 * Create a <code>ResizableTab</code> given the {@link Widget} 
		 * <code>content</content> and <code>String tabTitle</code>.
		 * <p>
		 * If either <code>content</code> is <code>null</code> or 
		 * <code>tabTitle</code> is <code>null</code> or empty, a 
		 * <code>ResizableTab</code> cannot be created, and a 
		 * {@link NullPointerException} is thrown.  
		 * 
		 * @param tabContent
		 * @param tabTitle
		 */
		public ResizableTab(Widget tabContent, String tabTitle) {
			this.tabContent = tabContent;
			this.tabTitle = tabTitle;
			
			if (tabContent == null)  {
				throw new NullPointerException("Widget tabContent cannot be null");
			}
			
			if ((tabTitle == null) || (tabTitle.trim().isEmpty())) {
				throw new NullPointerException("String tabTitle cannot be null or empty");
			}
			
			verticalContainer = new VerticalContainer();
			scrollContainer = new ScrollContainer();
			buttonsContainer = new HorizontalContainer();
			
			verticalContainer.add(scrollContainer);
			verticalContainer.add(buttonsContainer);
			
			scrollContainer.setWidget(tabContent);

			scrollContainer.addStyleName(STYLE_NAME_SCROLL_CONTAINER);
			buttonsContainer.addStyleName(STYLE_NAME_BUTTONS_CONTAINER);
		}

		/**
		 * @return the tabTitle
		 */
		public String getTabTitle() {
			return tabTitle;
		}

		/**
		 * @param tabTitle the tabTitle to set
		 */
		public void setTabTitle(String tabTitle) {
			this.tabTitle = tabTitle;
		}

		/**
		 * @return the tabContent
		 */
		public Widget getTabContent() {
			return tabContent;
		}

		/**
		 * @param tabContent the tabContent to set
		 */
		public void setTabContent(Widget tabContent) {
			this.tabContent = tabContent;
		}		

		/**
		 * @return the verticalContainer
		 */
		public VerticalContainer getVerticalContainer() {
			return verticalContainer;
		}
		
		/**
		 * 
		 * @param selected
		 */
		public void setSelected(boolean selected) {
			if (selected) {
				String anchor = getAnchor(tabTitle, anchorPrefix);
				if (anchor != null) {
					History.newItem(anchor, false);
				}
				messageArea.reset();
				updateScrollPanelSize();
			}
		}
		
		/**
		 * 
		 */
		protected void updateScrollPanelSize() {
			int height = Window.getClientHeight()
				- CMS_HEADER_MENU_SIZE_PX
				- offsetHeight
				- messageArea.getOffsetHeight()
				- tabContainer.getTabBar().getOffsetHeight()
				- buttonsContainer.getOffsetHeight()
				- BOTTOM_OFFSET_PX;
			
			for (Widget w : heightOffsettingWidgets) {
				height -= w.getOffsetHeight(); 
			}
			
			scrollContainer.setHeight(height + "px");
		}
		
		/**
		 * 
		 * @param tabTitle
		 * @param anchorPrefix
		 * @return
		 */
		protected String getAnchor(String tabTitle, String anchorPrefix) {
			String anchor = anchorPrefix;
			
			if ((tabTitle != null) && (anchorPrefix != null)) {
				String tabSuffix = ANCHOR_TAB_PORTION_PARAMETER_NAME + "=" + tabTitle;
				if (anchorPrefix.contains("?")) {	
					tabSuffix = "&" + tabSuffix;
				}
				else {
					tabSuffix = "?" + tabSuffix;
				}
				anchor = anchorPrefix + tabSuffix;
			}
			
			return anchor;
		}
		
	}

}
