/*
 * Created on May 15, 2009
 * 
 * (C) Copyright TANDBERG Television Ltd.
 */

package com.tandbergtv.neptune.widgettoolkit.client.widget.basic;

import java.util.HashMap;
import java.util.Map;

import com.google.gwt.event.dom.client.BlurEvent;
import com.google.gwt.event.dom.client.BlurHandler;
import com.google.gwt.event.dom.client.ChangeEvent;
import com.google.gwt.event.dom.client.ChangeHandler;
import com.google.gwt.event.dom.client.ClickEvent;
import com.google.gwt.event.dom.client.ClickHandler;
import com.google.gwt.event.dom.client.FocusEvent;
import com.google.gwt.event.dom.client.FocusHandler;
import com.google.gwt.event.dom.client.HasAllFocusHandlers;
import com.google.gwt.event.dom.client.HasAllKeyHandlers;
import com.google.gwt.event.dom.client.HasAllMouseHandlers;
import com.google.gwt.event.dom.client.HasChangeHandlers;
import com.google.gwt.event.dom.client.HasClickHandlers;
import com.google.gwt.event.dom.client.KeyDownEvent;
import com.google.gwt.event.dom.client.KeyDownHandler;
import com.google.gwt.event.dom.client.KeyPressEvent;
import com.google.gwt.event.dom.client.KeyPressHandler;
import com.google.gwt.event.dom.client.KeyUpEvent;
import com.google.gwt.event.dom.client.KeyUpHandler;
import com.google.gwt.event.dom.client.MouseDownEvent;
import com.google.gwt.event.dom.client.MouseDownHandler;
import com.google.gwt.event.dom.client.MouseMoveEvent;
import com.google.gwt.event.dom.client.MouseMoveHandler;
import com.google.gwt.event.dom.client.MouseOutEvent;
import com.google.gwt.event.dom.client.MouseOutHandler;
import com.google.gwt.event.dom.client.MouseOverEvent;
import com.google.gwt.event.dom.client.MouseOverHandler;
import com.google.gwt.event.dom.client.MouseUpEvent;
import com.google.gwt.event.dom.client.MouseUpHandler;
import com.google.gwt.event.dom.client.MouseWheelEvent;
import com.google.gwt.event.dom.client.MouseWheelHandler;
import com.google.gwt.event.shared.HandlerRegistration;
import com.google.gwt.user.client.ui.Composite;
import com.google.gwt.user.client.ui.Focusable;
import com.google.gwt.user.client.ui.HasName;
import com.google.gwt.user.client.ui.ListBox;
import com.tandbergtv.neptune.widgettoolkit.client.widget.INeptuneWidget;

/**
 * The toolkit widget that is a List Box which allows for values that are not Strings. If using this
 * widget in an HTML Form that needs to be submitted, use the {@link SimpleListBoxWidget} instead.
 * This widget generates unique keys for the 'option' values that are submitted.
 * 
 * @param <ValueType> The type of the values the list box contains.
 * @author Vijay Silva
 */
public class ListBoxWidget<ValueType> extends Composite implements HasName, Focusable,
        HasChangeHandlers, HasClickHandlers, HasAllFocusHandlers, HasAllKeyHandlers,
        HasAllMouseHandlers, INeptuneWidget {

	/* State */
	private ListBox listBox;
	private Map<String, ValueType> items = new HashMap<String, ValueType>();
	private ItemKeyGenerator keyGenerator = new ItemKeyGenerator();

	/**
	 * Constructor
	 * 
	 * @see SimpleListBoxWidget#SimpleListBoxWidget()
	 */
	public ListBoxWidget() {
		this(false);
	}

	/**
	 * Constructor
	 * 
	 * @see SimpleListBoxWidget#SimpleListBoxWidget(boolean)
	 */
	public ListBoxWidget(boolean isMultipleSelect) {
		super();
		listBox = new SimpleListBoxWidget(isMultipleSelect);
		initializeWidget();
	}

	/*
	 * Perform any initialization of the widget at the end of construction.
	 */
	private void initializeWidget() {
		initWidget(listBox);
	}

	// ========================================================================
	// ===================== HAS NAME
	// ========================================================================

	@Override
	public String getName() {
		return listBox.getName();
	}

	@Override
	public void setName(String name) {
		listBox.setName(name);
	}

	// ========================================================================
	// ===================== EVENT HANDLERS
	// ========================================================================

	@Override
	public HandlerRegistration addChangeHandler(ChangeHandler handler) {
		return addDomHandler(handler, ChangeEvent.getType());
	}

	@Override
	public HandlerRegistration addClickHandler(ClickHandler handler) {
		return addDomHandler(handler, ClickEvent.getType());
	}

	@Override
	public HandlerRegistration addBlurHandler(BlurHandler handler) {
		return addDomHandler(handler, BlurEvent.getType());
	}

	@Override
	public HandlerRegistration addFocusHandler(FocusHandler handler) {
		return addDomHandler(handler, FocusEvent.getType());
	}

	@Override
	public HandlerRegistration addKeyDownHandler(KeyDownHandler handler) {
		return addDomHandler(handler, KeyDownEvent.getType());
	}

	@Override
	public HandlerRegistration addKeyPressHandler(KeyPressHandler handler) {
		return addDomHandler(handler, KeyPressEvent.getType());
	}

	@Override
	public HandlerRegistration addKeyUpHandler(KeyUpHandler handler) {
		return addDomHandler(handler, KeyUpEvent.getType());
	}

	@Override
	public HandlerRegistration addMouseDownHandler(MouseDownHandler handler) {
		return addDomHandler(handler, MouseDownEvent.getType());
	}

	@Override
	public HandlerRegistration addMouseMoveHandler(MouseMoveHandler handler) {
		return addDomHandler(handler, MouseMoveEvent.getType());
	}

	@Override
	public HandlerRegistration addMouseOutHandler(MouseOutHandler handler) {
		return addDomHandler(handler, MouseOutEvent.getType());
	}

	@Override
	public HandlerRegistration addMouseOverHandler(MouseOverHandler handler) {
		return addDomHandler(handler, MouseOverEvent.getType());
	}

	@Override
	public HandlerRegistration addMouseUpHandler(MouseUpHandler handler) {
		return addDomHandler(handler, MouseUpEvent.getType());
	}

	@Override
	public HandlerRegistration addMouseWheelHandler(MouseWheelHandler handler) {
		return addDomHandler(handler, MouseWheelEvent.getType());
	}

	// ========================================================================
	// ===================== FOCUSABLE
	// ========================================================================

	@Override
	public int getTabIndex() {
		return listBox.getTabIndex();
	}

	@Override
	public void setTabIndex(int index) {
		listBox.setTabIndex(index);
	}

	@Override
	public void setAccessKey(char key) {
		listBox.setAccessKey(key);
	}

	@Override
	public void setFocus(boolean focused) {
		listBox.setFocus(focused);
	}

	/**
	 * Sets whether this widget is enabled.
	 * 
	 * @see ListBox#setEnabled(boolean)
	 */
	public void setEnabled(boolean enabled) {
		listBox.setEnabled(enabled);
	}

	/**
	 * Gets whether this widget is enabled.
	 * 
	 * @see ListBox#isEnabled()
	 */
	public boolean isEnabled() {
		return listBox.isEnabled();
	}

	// ========================================================================
	// ===================== ITEM MANAGEMENT
	// ========================================================================

	/**
	 * Adds an item using the specified name and value.
	 * 
	 * @param itemName The display name for the item
	 * @param item The value for the item
	 */
	public void addItem(String itemName, ValueType item) {
		String itemKey = keyGenerator.generate();
		listBox.addItem(itemName, itemKey);
		items.put(itemKey, item);
	}

	/**
	 * Inserts an item at the specified index.
	 * 
	 * @param itemName The display name for the item
	 * @param item The value for the item
	 * @param index the index at which to insert it
	 */
	public void insertItem(String itemName, ValueType item, int index) {
		String itemKey = keyGenerator.generate();
		listBox.insertItem(itemName, itemKey, index);
		items.put(itemKey, item);
	}

	/**
	 * Sets the item (both display name and value) at the specified index overwriting the previous
	 * item at that index.
	 * 
	 * @param index The index of the item being overwritten
	 * @param itemName The new item name
	 * @param item The new item value
	 */
	public void setItem(String itemName, ValueType item, int index) {
		setItemText(index, itemName);
		setItem(index, item);
	}

	/**
	 * Removes the first item in the list box with matching item value, or does nothing if the item
	 * is not present.
	 * 
	 * @param item The item value to match
	 */
	public void removeItem(ValueType item) {
		int index = this.getIndex(item);
		if (index > -1)
			removeItemAtIndex(index);
	}

	/**
	 * Removes the item at the specified index.
	 * 
	 * @param index the (0-based) index of the item to be removed
	 * @throws IndexOutOfBoundsException if the index is out of range
	 */
	public void removeItemAtIndex(int index) {
		String itemKey = this.getItemKey(index);
		listBox.removeItem(index);
		items.remove(itemKey);
	}

	/**
	 * Removes all items from the list box.
	 */
	public void clear() {
		/* Reset the key generator and clear all items */
		listBox.clear();
		items.clear();
		keyGenerator.reset();
	}

	/**
	 * Gets the number of items present in the list box.
	 * 
	 * @return the number of items
	 * @see ListBox#getItemCount()
	 */
	public int getItemCount() {
		return listBox.getItemCount();
	}

	/**
	 * Gets the text associated with the item at the specified index.
	 * 
	 * @param index the index of the item whose text is to be retrieved
	 * @return the text associated with the item
	 * @see ListBox#getItemText(int)
	 */
	public String getItemText(int index) {
		return listBox.getItemText(index);
	}

	/**
	 * Sets the text associated with the item at a given index.
	 * 
	 * @param index the index of the item to be set
	 * @param text the item's new text
	 * @see ListBox#setItemText(int, String)
	 */
	public void setItemText(int index, String text) {
		listBox.setItemText(index, text);
	}

	/**
	 * Gets the value associated with the item at a given index.
	 * 
	 * @param index the index of the item to be retrieved
	 * @return the item's associated value
	 */
	public ValueType getItem(int index) {
		String itemKey = this.getItemKey(index);
		return items.get(itemKey);
	}

	/**
	 * Sets the value associated with the item at a given index. The display text for this item is
	 * not updated when the item is updated.
	 * 
	 * @param index the index of the item to be set
	 * @param value the new item value
	 */
	public void setItem(int index, ValueType value) {
		String itemKey = this.getItemKey(index);
		items.put(itemKey, value);
	}

	/**
	 * Find the index of the first matching item in the list of list box items.
	 * 
	 * @param item The item to match
	 * @return The index of the first matching item, or -1 if the item is not present
	 */
	public int getIndex(ValueType item) {
		int itemIndex = -1;
		for (int index = 0; index < getItemCount(); index++) {
			ValueType currentItemValue = getItem(index);

			/* The item could be null */
			if ((item == currentItemValue) || (item != null && item.equals(currentItemValue))) {
				itemIndex = index;
				break;
			}
		}
		return itemIndex;
	}

	/**
	 * Gets the currently selected item index.
	 * 
	 * @return the selected index, or -1 if none is selected
	 * @see ListBox#getSelectedIndex()
	 */
	public int getSelectedIndex() {
		return listBox.getSelectedIndex();
	}

	/**
	 * Sets the currently selected item index.
	 * 
	 * @param index the index of the item to be selected
	 * @see ListBox#setSelectedIndex(int)
	 */
	public void setSelectedIndex(int index) {
		listBox.setSelectedIndex(index);
	}

	/**
	 * Gets the currently selected item
	 * 
	 * @return the selected item, or null if no item is selected
	 */
	public ValueType getSelectedItem() {
		int index = listBox.getSelectedIndex();
		return (index != -1) ? getItem(index) : null;
	}

	/**
	 * Set the item as selected. If the item is not found in the list, the selection is cleared.
	 * 
	 * @param item The item to select
	 */
	public void setSelectedItem(ValueType item) {
		this.setSelectedIndex(getIndex(item));
	}

	/**
	 * Determines whether an item is selected.
	 * 
	 * @param index the index of the item to be tested
	 * @return true if the item is selected, false otherwise
	 * @see ListBox#isItemSelected(int)
	 */
	public boolean isItemSelected(int index) {
		return listBox.isItemSelected(index);
	}

	/**
	 * Sets whether an individual list item is selected.
	 * 
	 * @param index the index of the item to be selected or unselected
	 * @param selected true to select the item
	 * @see ListBox#setItemSelected(int, boolean)
	 */
	public void setItemSelected(int index, boolean selected) {
		listBox.setItemSelected(index, selected);
	}

	/**
	 * Gets the number of items that are visible.
	 * 
	 * @return the visible item count
	 * @see ListBox#getVisibleItemCount()
	 */
	public int getVisibleItemCount() {
		return listBox.getVisibleItemCount();
	}

	/**
	 * Sets the number of items that are visible.
	 * 
	 * @param visibleItems the visible item count
	 * @see ListBox#setVisibleItemCount(int)
	 */
	public void setVisibleItemCount(int visibleItems) {
		listBox.setVisibleItemCount(visibleItems);
	}

	/**
	 * Gets whether this list allows multiple selection.
	 * 
	 * @return true if multiple selection is allowed, false otherwise
	 * @see ListBox#isMultipleSelect()
	 */
	public boolean isMultipleSelect() {
		return listBox.isMultipleSelect();
	}

	/*
	 * Get the Item Key for an item at a specified index
	 */
	private String getItemKey(int index) {
		return listBox.getValue(index);
	}

	// ========================================================================
	// ===================== ITEM KEY MANAGEMENT
	// ========================================================================

	/*
	 * Internal class that maintains the state required for generating keys for items added to the
	 * list box
	 */
	private final class ItemKeyGenerator {
		/* Properties */
		private long itemCounter;

		/* The prefix to use for the item keys */
		private static final String ITEM_KEY_PREFIX = "option-";

		/*
		 * Constructs a new generator
		 */
		public ItemKeyGenerator() {
			reset();
		}

		/*
		 * Generate a new item key
		 */
		String generate() {
			return ITEM_KEY_PREFIX + (++itemCounter);
		}

		/*
		 * Reset the item key generator
		 */
		void reset() {
			itemCounter = 0;
		}
	}
}
