package com.tandbergtv.neptune.ui.componentmgr.client.tab.component.view;

import static com.tandbergtv.neptune.ui.componentmgr.client.ComponentManagerComponentPermissions.COMPONENT_CREATE;
import static com.tandbergtv.neptune.ui.componentmgr.client.ComponentManagerComponentPermissions.COMPONENT_DELETE;
import static com.tandbergtv.neptune.ui.componentmgr.client.ComponentManagerComponentPermissions.COMPONENT_MODIFY;
import static com.tandbergtv.neptune.ui.componentmgr.client.ComponentManagerComponentPermissions.COMPONENT_VIEW;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import com.google.gwt.core.client.GWT;
import com.google.gwt.user.client.Window;
import com.google.gwt.user.client.rpc.AsyncCallback;
import com.google.gwt.user.client.ui.Label;
import com.google.gwt.user.client.ui.SimplePanel;
import com.google.gwt.user.client.ui.VerticalPanel;
import com.google.gwt.user.client.ui.Widget;
import com.tandbergtv.neptune.ui.componentmgr.client.ComponentRegistrationDataService;
import com.tandbergtv.neptune.ui.componentmgr.client.ComponentRegistrationDataServiceAsync;
import com.tandbergtv.neptune.ui.componentmgr.client.i18n.ComponentManagerConstants;
import com.tandbergtv.neptune.ui.componentmgr.client.i18n.ComponentManagerMessages;
import com.tandbergtv.neptune.ui.componentmgr.client.tab.component.model.UiComponent;
import com.tandbergtv.neptune.ui.componentmgr.client.tab.component.model.UiComponentKey;
import com.tandbergtv.neptune.ui.componentmgr.client.tab.component.service.ComponentUiService;
import com.tandbergtv.neptune.ui.componentmgr.client.tab.component.service.ComponentUiServiceAsync;
import com.tandbergtv.neptune.ui.componentmgr.client.tab.component.view.ComponentForm;
import com.tandbergtv.neptune.ui.componentmgr.client.tab.component.view.ComponentRecord;
import com.tandbergtv.neptune.ui.componentmgr.client.tab.component.view.ComponentTableView;
import com.tandbergtv.neptune.ui.componentmgr.client.tab.component.view.ComponentTableViewEvent;
import com.tandbergtv.neptune.ui.componentmgr.client.tab.component.view.ComponentTableViewEventHandler;
import com.tandbergtv.neptune.widgettoolkit.client.application.ClientAuthorizationManager;
import com.tandbergtv.neptune.widgettoolkit.client.application.NeptuneApplication;
import com.tandbergtv.neptune.widgettoolkit.client.remote.NeptuneAsyncCallback;
import com.tandbergtv.neptune.widgettoolkit.client.widget.basic.LabelWidget;
import com.tandbergtv.neptune.widgettoolkit.client.widget.composite.BusyIndicator;
import com.tandbergtv.neptune.widgettoolkit.client.widget.composite.HeaderPanel;
import com.tandbergtv.neptune.widgettoolkit.client.widget.composite.LazyView;
import com.tandbergtv.neptune.widgettoolkit.client.widget.composite.table.feature.impl.AnchorTokenizer;
import com.tandbergtv.neptune.widgettoolkit.client.widget.container.SimpleContainer;
import com.tandbergtv.neptune.widgettoolkit.client.widget.event.RecordClickEvent;
import com.tandbergtv.neptune.widgettoolkit.client.widget.event.RecordClickHandler;
import com.tandbergtv.neptune.widgettoolkit.client.widget.event.ViewAnchorChangeEvent;
import com.tandbergtv.neptune.widgettoolkit.client.widget.event.ViewAnchorChangeHandler;
import com.tandbergtv.neptune.widgettoolkit.client.widget.event.ViewCancelEvent;
import com.tandbergtv.neptune.widgettoolkit.client.widget.event.ViewCancelHandler;
import com.tandbergtv.neptune.widgettoolkit.client.widget.event.ViewCommitEvent;
import com.tandbergtv.neptune.widgettoolkit.client.widget.event.ViewCommitHandler;

public class ComponentView extends LazyView {
	private VerticalPanel mainPanel;
	private HeaderPanel headerPanel;
	private Label errorLabel;
	private SimplePanel widgetPanel;
	private ComponentTableView tableView;
	private BusyIndicator busyIndicator = new BusyIndicator(this);
	private ComponentManagerMessages messages;
	private ComponentManagerConstants constants;
	private ComponentUiServiceAsync service = null;
	private ComponentRegistrationDataServiceAsync registrationService = null;

	/* Styles / constants */
	private static final String STYLE_NAME = "component-manager-ComponentView";
	private static final String STYLE_LIST_VIEW = "listView";
	private static final String STYLE_DETAIL_VIEW = "detailView";
	private static final String STYLE_HEADER_PANEL = STYLE_NAME + "-headerPanel";
	private static final String STYLE_ERROR_LABEL = STYLE_NAME + "-errorLabel";
	private static final String STYLE_WIDGET_PANEL = STYLE_NAME + "-widgetPanel";
	private static final String CREATE_VIEW_TOKEN = "Create";
	private static final String EDIT_VIEW_TOKEN = "Edit";
	private static final String VIEW_TOKEN_SEPARATOR = "?";
	private static final String RECORD_KEY_TOKEN = "id";
	private static final String COMPONENT_NAME_TOKEN = "componentName";

	/**
	 * Constructor
	 */
	public ComponentView() {
		addStyleName(STYLE_NAME);
	}

	/**
	 * Reset the view to the initial state. Does not fire an anchor change event.
	 */
	public void reset() {
		/* Still lazy, do nothing */
		if (getWidget() == null)
			return;

		/* Hide the header panel and error message */
		hideHeaderPanel();
		hideErrorMessage();

		/* Reset the displayed widget to a blank widget */
		setWidgetInPanel(null);
		if (tableView != null)
			tableView.reset();
	}

	/**
	 * Show table view, keeping the current state of the features. The records are re-fetched from
	 * the server. Does not fire an anchor change event.
	 * 
	 * @param showFirstPage Flag indicating the table should change to show the first page or
	 *        continue to show the current page of records
	 */
	public void showTable(final boolean showFirstPage) {
		/* Ensure the widget is not lazy */
		ensureWidget();

		/* Hide the error message and error panel */
		hideErrorMessage();
		hideHeaderPanel();
		
		initTableView();

		showTableView();
		tableView.refresh(showFirstPage);
	}

	/**
	 * Check if this view is showing the table of components. If the view is lazy and is not
	 * initialized, this method will return true
	 * 
	 * @return true if the table view is showing, false otherwise
	 */
	public boolean isShowingTable() {
		Widget widget = getWidgetInPanel();
		return (widget == null || widget == tableView);
	}

	/*
	 * Create the widget
	 */
	@Override
	protected Widget createWidget() {
		mainPanel = new VerticalPanel();
		mainPanel.setSpacing(5);
		mainPanel.setStyleName(STYLE_LIST_VIEW);

		/* At the very top is the header panel */
		headerPanel = new HeaderPanel("");
		headerPanel.setVisible(false);
		headerPanel.addStyleName(STYLE_HEADER_PANEL);
		mainPanel.add(headerPanel);

		/* Next is a blank (invisible) error label */
		errorLabel = new LabelWidget();
		errorLabel.setVisible(false);
		errorLabel.addStyleName(STYLE_ERROR_LABEL);
		mainPanel.add(errorLabel);

		/* Add the widget panel */
		widgetPanel = new SimpleContainer();
		widgetPanel.addStyleName(STYLE_WIDGET_PANEL);
		mainPanel.add(widgetPanel);

		return mainPanel;
	}

	// ========================================================================
	// ===================== INTERNAL STUFF
	// ========================================================================

	/*
	 * Initialize the Table View
	 */
	private void initTableView() {
		tableView = new ComponentTableView(busyIndicator);
		TableEventHandler eventHandler = new TableEventHandler();
		tableView.addRecordClickHandler(eventHandler);
		tableView.addComponentTableViewEventHandler(eventHandler);
		tableView.addViewAnchorChangeHandler(eventHandler);
		setWidgetInPanel(tableView);
	}
	
	/*
	 * If the widget is attached to the DOM, show the busy indicator
	 */
	private void showBusyIndicator() {
		busyIndicator.center();
	}

	/*
	 * Hide the busy indicator
	 */
	private void hideBusyIndicator() {
		busyIndicator.hide();
	}

	/*
	 * Show the error message in the view
	 */
	private void showErrorMessage(String message) {
		if (message == null)
			message = "";

		errorLabel.setText(message);
		errorLabel.setVisible(true);
	}

	/*
	 * Clear the error message displayed in the view
	 */
	private void hideErrorMessage() {
		errorLabel.setText("");
		errorLabel.setVisible(false);
	}

	/*
	 * Show the header panel
	 */
	private void showHeaderPanel(String heading, String styleName) {
		headerPanel.setLabel(heading);
		headerPanel.setVisible(true);
		mainPanel.setStyleName(styleName);
	}

	/*
	 * Hide the header panel
	 */
	private void hideHeaderPanel() {
		headerPanel.setLabel("");
		headerPanel.setVisible(false);
		mainPanel.setStyleName(STYLE_LIST_VIEW);
	}

	/*
	 * Get the constants
	 */
	private ComponentManagerConstants getConstants() {
		if (constants == null)
			constants = GWT.create(ComponentManagerConstants.class);

		return constants;
	}

	/*
	 * Get the messages
	 */
	private ComponentManagerMessages getMessages() {
		if (messages == null)
			messages = GWT.create(ComponentManagerMessages.class);

		return messages;
	}
	
	/*
	 * Get the component management service
	 */
	private ComponentUiServiceAsync getService() {
		if (service == null) {
			service = GWT.create(ComponentUiService.class);
		}
		return service;
	}
	
	/*
	 * Get the Component Registration Data Service
	 */
	private ComponentRegistrationDataServiceAsync getRegistrationService() {
		if (registrationService == null) {
			registrationService = GWT.create(ComponentRegistrationDataService.class);
		}
		return registrationService;
	}

	// ========================================================================
	// ===================== ANCHOR MANAGEMENT
	// ========================================================================

	/*
	 * Get the widget anchor
	 */
	@Override
	public String getAnchor() {
		String anchor = null;
		Widget widget = getWidgetInPanel();

		if (tableView != null && widget == tableView) {
			/* Showing the list view, get table anchor */
			anchor = tableView.getAnchor();
		} else if (widget instanceof ComponentForm) {
			/* This is either the create or edit screen */
			ComponentForm form = (ComponentForm) widget;
			UiComponentKey key = form.getComponent().getKey();
			if (key.isValueAssigned()) {
				Map<String, String> tokens = new HashMap<String, String>();
				tokens.put(RECORD_KEY_TOKEN, key.toString());
				anchor = EDIT_VIEW_TOKEN + VIEW_TOKEN_SEPARATOR
				        + new AnchorTokenizer().buildAnchor(tokens);
			} else {
				anchor = CREATE_VIEW_TOKEN;
			}
		} else {
			anchor = "";
		}

		return anchor;
	}

	/*
	 * Set the widget anchor
	 */
	@Override
	protected void setWidgetAnchor(final String anchor) {
		/* Reset the widget state / view */
		reset();
		initTableView();
		
		/* Test the view token in the anchor */
		if (isAnchorForView(CREATE_VIEW_TOKEN, anchor)) {
			/* Create Anchor */
			showWidgetForCreateAnchor(anchor);
		} else if (isAnchorForView(EDIT_VIEW_TOKEN, anchor)) {
			showWidgetForEditAnchor(anchor);
		} else {
			/* Default to table view */
			showTableView();
			tableView.setAnchor(anchor);
		}
	}

	/*
	 * Determine if the anchor is for a particular view
	 */
	private boolean isAnchorForView(String viewToken, String anchor) {
		return (anchor != null && (anchor.equals(viewToken) || anchor.startsWith(viewToken
		        + VIEW_TOKEN_SEPARATOR)));
	}

	/*
	 * Show the widget for the 'Create' anchor
	 */
	private void showWidgetForCreateAnchor(String anchor) {
		/* Creating component - verify permissions */
		if (!verifyAuthorization(COMPONENT_VIEW, COMPONENT_CREATE))
			return;

		/* Show the widget */
		showComponentFormForCreate();
	}

	/*
	 * Show the widget for the 'Edit' anchor
	 */
	private void showWidgetForEditAnchor(String anchor) {
		/* Editing / viewing user - verify permissions */
		if (!verifyAuthorization(COMPONENT_VIEW))
			return;

		/* Parse the anchor */
		anchor = getAnchorForView(EDIT_VIEW_TOKEN, anchor);
		Map<String, String> tokens = new AnchorTokenizer().parseAnchor(anchor);
		if (tokens.containsKey(RECORD_KEY_TOKEN)) {
			try {
				String keyValue = tokens.get(RECORD_KEY_TOKEN);
				UiComponentKey key = UiComponentKey.parse(keyValue);
				showComponentByKey(key);
			} catch (RuntimeException e) {
				showErrorMessage(getConstants().viewComponentInvalidAnchorError());
			}
		} else if (tokens.containsKey(COMPONENT_NAME_TOKEN)) {
			String name = tokens.get(COMPONENT_NAME_TOKEN);
			showComponentByName(name);
		} else {
			showErrorMessage(getConstants().viewComponentInvalidAnchorError());
		}
	}

	/*
	 * Show the component given the key
	 */
	private void showComponentByKey(final UiComponentKey key) {
		showBusyIndicator();
		getService().getComponent(key.getId(), new NeptuneAsyncCallback<UiComponent>() {
			@Override
			public void onNeptuneFailure(Throwable caught) {
				hideBusyIndicator();
				String idValue = (key.getId() != null) ? key.getId().toString() : "";
				String errorMessage = caught.getLocalizedMessage();
				showErrorMessage(getMessages().getComponentFailure(idValue, errorMessage));
			}

			@Override
			public void onNeptuneSuccess(UiComponent result) {
				hideBusyIndicator();
				showComponentFormForEdit(result);
			}
		});
	}

	/*
	 * Show the component given the component name
	 */
	private void showComponentByName(final String name) {
		showBusyIndicator();
		getService().getComponent(name, new NeptuneAsyncCallback<UiComponent>() {
			@Override
			public void onNeptuneFailure(Throwable caught) {
				hideBusyIndicator();
				String errorMessage = caught.getLocalizedMessage();
				showErrorMessage("No component with name " + name + " found.  " + errorMessage);
			}

			@Override
			public void onNeptuneSuccess(UiComponent result) {
				hideBusyIndicator();
				showComponentFormForEdit(result);
			}
		});
	}

	/*
	 * Verify that the user is authorized based on permissions, updating the error message if not
	 * authorized.
	 */
	private boolean verifyAuthorization(String... permissions) {
		/* Verify permission to access this view */
		if (!ClientAuthorizationManager.isAuthorized(permissions)) {
			showErrorMessage(getConstants().unauthorizedAccessError());
			return false;
		}

		return true;
	}

	/*
	 * Get the anchor without the view token
	 */
	private String getAnchorForView(String viewToken, String anchor) {
		anchor = anchor.substring(viewToken.length());
		if (anchor.startsWith(VIEW_TOKEN_SEPARATOR))
			anchor = anchor.substring(VIEW_TOKEN_SEPARATOR.length());

		return anchor;
	}

	/*
	 * Fire the view anchor change event
	 */
	private void fireViewAnchorChange() {
		fireViewAnchorChangeEvent(getAnchor());
	}

	// ========================================================================
	// ===================== WIDGET PANEL DISPLAY
	// ========================================================================

	/*
	 * Get the widget being displayed in the widget panel
	 */
	private Widget getWidgetInPanel() {
		if (widgetPanel == null)
			return null;

		return widgetPanel.getWidget();
	}

	/*
	 * Set the panel widget
	 */
	private void setWidgetInPanel(Widget widget) {
		widgetPanel.setWidget(widget);
	}

	/*
	 * Show the table of user records as a result of an event
	 */
	private void returnToTableView(boolean showFirstPage) {
		/* Hide the error message */
		hideErrorMessage();

		/* Return to table and refresh the table records */
		showTableView();
		tableView.refresh(showFirstPage);

		/* Update the widget anchor */
		fireViewAnchorChange();
	}

	/*
	 * Show the table view in the widget panel
	 */
	private void showTableView() {
		setWidgetInPanel(tableView);
		showHeaderPanel(getConstants().componentListViewHeading(), STYLE_LIST_VIEW);
	}

	/*
	 * Create and show a component form for a newly created component
	 */
	private ComponentForm showComponentFormForCreate() {
		/* Create the component to create */
		UiComponent componentToCreate = new UiComponent();

		/* Build the form */
		ComponentForm form = new ComponentForm(componentToCreate, false);
		FormEventHandler eventHandler = new FormEventHandler();
		form.addViewCancelHandler(eventHandler);
		form.addViewCommitHandler(eventHandler);
		setWidgetInPanel(form);
		showHeaderPanel(getConstants().componentCreateHeading(), STYLE_DETAIL_VIEW);
		return form;
	}

	/*
	 * Create and show the user form for the user being edited / viewed
	 */
	private ComponentForm showComponentFormForEdit(UiComponent component) {
		boolean readOnly = isReadOnly(component);
		ComponentForm form = new ComponentForm(component, readOnly);
		FormEventHandler eventHandler = new FormEventHandler();
		form.addViewCancelHandler(eventHandler);
		form.addViewCommitHandler(eventHandler);
		setWidgetInPanel(form);
		showHeaderPanel(getMessages().componentEditHeading(component.getName()), STYLE_DETAIL_VIEW);
		return form;
	}

	/*
	 * Determine if the user must be shown as read-only
	 */
	private boolean isReadOnly(UiComponent component) {
		if (!ClientAuthorizationManager.isAuthorized(COMPONENT_VIEW, COMPONENT_MODIFY)) {
			/* No permission to edit component */
			return true;
		}

		return false;
	}

	// ========================================================================
	// ===================== TABLE VIEW EVENTS
	// ========================================================================

	/*
	 * Internal event handler for the table
	 */
	private final class TableEventHandler implements RecordClickHandler<ComponentRecord>,
	        ComponentTableViewEventHandler, ViewAnchorChangeHandler {

		/*
		 * Handle the table record click
		 */
		@Override
		public void onRecordClick(RecordClickEvent<ComponentRecord> event) {
			/* Hide the error message */
			hideErrorMessage();

			/* Verify that the user permissions authorize this action */
			if (!ClientAuthorizationManager.isAuthorized(COMPONENT_VIEW))
				return;

			/* Get the complete record from the server */
			final ComponentRecord record = event.getRecord();
			showBusyIndicator();
			getService().getComponent(record.getKey().getId(), new NeptuneAsyncCallback<UiComponent>() {
				@Override
				public void onNeptuneFailure(Throwable caught) {
					hideBusyIndicator();
					UiComponentKey key = record.getKey();
					String idValue = (key.getId() != null) ? key.getId().toString() : "";
					String msg = caught.getLocalizedMessage();
					Window.alert(getMessages().getComponentFailure(idValue, msg));
				}

				@Override
				public void onNeptuneSuccess(UiComponent result) {
					hideBusyIndicator();

					/* Show the form */
					showComponentFormForEdit(result);

					/* Update the anchor */
					fireViewAnchorChange();
				}
			});
		}

		/*
		 * Handle the table create button click
		 */
		@Override
		public void onCreateButtonClick(ComponentTableViewEvent event) {
			/* Hide the error message */
			hideErrorMessage();

			/* Show the form */
			showComponentFormForCreate();

			/* Update the anchor */
			fireViewAnchorChange();
		}

		/*
		 * Handle the table delete button click
		 */
		@Override
		public void onDeleteButtonClick(ComponentTableViewEvent event) {
			if (!verifyAuthorization(COMPONENT_DELETE))
				return;

			/* Hide the error message */
			hideErrorMessage();

			/* Check if records are selected */
			List<ComponentRecord> records = tableView.getSelectedRecords();
			if (records.size() == 0) {
				Window.alert(getConstants().noComponentCheckBoxSelected());
				return;
			}

			/* Request confirmation from the user */
			if (!Window.confirm(getConstants().componentDeleteConfirmation())) {
				return;
			}

			/* Collect the record keys and names */
			List<UiComponentKey> componentKeys = new ArrayList<UiComponentKey>();
			final List<String> componentNames = new ArrayList<String>();
			for (ComponentRecord record : records) {
				componentKeys.add(record.getKey());
				componentNames.add(record.getComponent().getName());
			}

			/* Delete the selected records */
			showBusyIndicator();
			getService().deleteComponents(componentKeys, new NeptuneAsyncCallback<Void>() {
				@Override
				public void onNeptuneFailure(Throwable caught) {
					/* Hide the busy indicator and report error to user */
					hideBusyIndicator();
					Window.alert(getMessages().deleteComponentsFailure(caught.getLocalizedMessage()));
				}

				@Override
				public void onNeptuneSuccess(Void result) {
					deleteRegistrationData(componentNames);
					/* Hide the busy indicator */
					hideBusyIndicator();

					/* Refresh the table records */
					tableView.refresh(false);
				}
			});
		}

		/*
		 * Handle the table anchor change
		 */
		@Override
		public void onAnchorChange(ViewAnchorChangeEvent event) {
			/* Hide the error message */
			hideErrorMessage();

			/* If the table view is being displayed, fire the new anchor change */
			if (getWidgetInPanel() == tableView)
				fireViewAnchorChange();
		}
		
		private void deleteRegistrationData(List<String> componentNames) {
			getRegistrationService().deleteComponentRegistration(componentNames, new AsyncCallback<Void>() {
				@Override
				public void onFailure(Throwable caught) {
				}
				@Override
				public void onSuccess(Void result) {
					NeptuneApplication.getApplication().handleUiComponentChange(null);					
				}
			});
		}
		
	}

	// ========================================================================
	// ===================== USER FORM EVENTS
	// ========================================================================

	/*
	 * Internal event handler for the user form
	 */
	private final class FormEventHandler implements ViewCommitHandler<UiComponent>, ViewCancelHandler {

		/* User saved, return to first page of the table */
		@Override
		public void onCommit(ViewCommitEvent<UiComponent> event) {
			returnToTableView(false);
			UiComponent uComponent = event.getData();
			NeptuneApplication.getApplication().handleUiComponentChange(uComponent.getName());
		}

		/* Action canceled, return to the table */
		@Override
		public void onCancel(ViewCancelEvent event) {
			returnToTableView(false);
		}
	}


}
