/*
 * Created on Jan 3, 2012
 * 
 * (C) Copyright Ericsson Television Inc.
 */

package com.tandbergtv.neptune.ui.realm.client.tab.user.view.external;

import java.util.HashSet;
import java.util.List;
import java.util.Set;

import com.google.gwt.core.client.GWT;
import com.google.gwt.event.dom.client.ClickEvent;
import com.google.gwt.event.dom.client.ClickHandler;
import com.google.gwt.event.dom.client.KeyCodes;
import com.google.gwt.event.dom.client.KeyPressEvent;
import com.google.gwt.event.dom.client.KeyPressHandler;
import com.google.gwt.event.shared.HandlerRegistration;
import com.google.gwt.user.client.rpc.AsyncCallback;
import com.google.gwt.user.client.ui.Button;
import com.google.gwt.user.client.ui.HasVerticalAlignment;
import com.google.gwt.user.client.ui.HorizontalPanel;
import com.google.gwt.user.client.ui.SimplePanel;
import com.google.gwt.user.client.ui.TextBox;
import com.google.gwt.user.client.ui.VerticalPanel;
import com.google.gwt.user.client.ui.Widget;
import com.tandbergtv.neptune.ui.realm.client.i18n.RealmConstants;
import com.tandbergtv.neptune.ui.realm.client.tab.user.model.UiUserConfiguration;
import com.tandbergtv.neptune.widgettoolkit.client.remote.NeptuneAsyncCallback;
import com.tandbergtv.neptune.widgettoolkit.client.widget.basic.ButtonWidget;
import com.tandbergtv.neptune.widgettoolkit.client.widget.basic.LabelWidget;
import com.tandbergtv.neptune.widgettoolkit.client.widget.basic.TextBoxWidget;
import com.tandbergtv.neptune.widgettoolkit.client.widget.composite.BusyIndicator;
import com.tandbergtv.neptune.widgettoolkit.client.widget.composite.LazyView;
import com.tandbergtv.neptune.widgettoolkit.client.widget.composite.resizablecontainer.ResizableContainer;
import com.tandbergtv.neptune.widgettoolkit.client.widget.composite.table.Column;
import com.tandbergtv.neptune.widgettoolkit.client.widget.composite.table.SortOrder;
import com.tandbergtv.neptune.widgettoolkit.client.widget.composite.table.Table;
import com.tandbergtv.neptune.widgettoolkit.client.widget.composite.table.anchor.TableAnchor;
import com.tandbergtv.neptune.widgettoolkit.client.widget.composite.table.anchor.TableAnchorChangeEvent;
import com.tandbergtv.neptune.widgettoolkit.client.widget.composite.table.anchor.TableAnchorChangeHandler;
import com.tandbergtv.neptune.widgettoolkit.client.widget.composite.table.feature.BookmarkFeature;
import com.tandbergtv.neptune.widgettoolkit.client.widget.composite.table.feature.PageFeature;
import com.tandbergtv.neptune.widgettoolkit.client.widget.composite.table.feature.SortFeature;
import com.tandbergtv.neptune.widgettoolkit.client.widget.composite.table.feature.impl.BookmarkFeatureImpl;
import com.tandbergtv.neptune.widgettoolkit.client.widget.composite.table.feature.impl.CookieStoreBasedPageFeatureImpl;
import com.tandbergtv.neptune.widgettoolkit.client.widget.composite.table.feature.impl.SortFeatureImpl;
import com.tandbergtv.neptune.widgettoolkit.client.widget.container.HorizontalContainer;
import com.tandbergtv.neptune.widgettoolkit.client.widget.container.VerticalContainer;
import com.tandbergtv.neptune.widgettoolkit.client.widget.event.HasRecordClickHandlers;
import com.tandbergtv.neptune.widgettoolkit.client.widget.event.HasViewCancelHandlers;
import com.tandbergtv.neptune.widgettoolkit.client.widget.event.RecordClickEvent;
import com.tandbergtv.neptune.widgettoolkit.client.widget.event.RecordClickHandler;
import com.tandbergtv.neptune.widgettoolkit.client.widget.event.ViewCancelEvent;
import com.tandbergtv.neptune.widgettoolkit.client.widget.event.ViewCancelHandler;
import com.tandbergtv.neptune.widgettoolkit.client.widget.style.StyleNames;

/**
 * View for the external users table and search / filter criteria
 * 
 * @author Vijay Silva
 */
public final class ExternalUserTableView extends LazyView implements HasViewCancelHandlers,
        HasRecordClickHandlers<ExternalUserRecord> {

	private final UiUserConfiguration configuration;
	private final BusyIndicator busyIndicator;
	private VerticalPanel tableContainer;
	private TextBox criteriaTextBox;
	private Button searchButton;
	private ExternalUserDataProvider dataProvider;
	private Table<String, ExternalUserRecord> table;
	private RealmConstants constants;

	/* Styles / Constants */
	private static final String STYLE_NAME = "realm-ExternalUserTableView";
	private static final String STYLE_CRITERIA_PANEL = STYLE_NAME + "-criteriaPanel";
	private static final String STYLE_CRITERIA_LABEL = STYLE_NAME + "-criteriaLabel";
	private static final String STYLE_CRITERIA_TEXTBOX = STYLE_NAME + "-criteriaTextBox";
	private static final String STYLE_CRITERIA_BUTTON = STYLE_NAME + "-criteriaButton";
	private static final String STYLE_SECTION_SEPARATOR = STYLE_NAME + "-sectionSeparator";
	private static final String STYLE_TABLE = STYLE_NAME + "-table";
	private static final String COOKIE_PREFIX = "realm_externalUserTable";
	private static final String CRITERIA_TOKEN = "searchCriteria";
	
	/* view */
	private ResizableContainer container;
	private ButtonWidget cancelButton;

	/**
	 * Constructor
	 */
	public ExternalUserTableView(UiUserConfiguration configuration, BusyIndicator indicator, ResizableContainer container) {
		this.configuration = configuration;
		this.container = container;
		this.busyIndicator = indicator;
		addStyleName(STYLE_NAME);
	}

	/**
	 * Reset the view, clearing the criteria and reseting the table.
	 */
	public void reset() {
		if (getWidget() == null)
			return;

		setSearchCriteria(null);
		if (table.isInitialized()) {
			table.reset();
			table.showListView();
		}
	}

	/**
	 * Refresh the view with new state.
	 */
	public void refresh(final boolean showFirstPage) {
		ensureWidget();
		
		/* If showing first page, set the state in the feature */
		if (showFirstPage)
			table.getPageFeature().setPageNumber(1);

		/* Initialize and refresh the table */
		initializeTable(new NeptuneAsyncCallback<Void>() {
			@Override
			public void onNeptuneFailure(Throwable caught) {
				/* Should not happen */
			}

			@Override
			public void onNeptuneSuccess(Void result) {
				table.reset(false);
				table.showListView();
				table.refresh();
			}
		});
	}

	/**
	 * Get the search criteria value displayed in the widget.
	 * 
	 * @return The search criteria input value
	 */
	public String getSearchCriteria() {
		if (getWidget() == null)
			return null;

		return criteriaTextBox.getText();
	}

	/**
	 * Set the search criteria to display in the widget
	 * 
	 * @param criteria The search criteria value
	 */
	public void setSearchCriteria(String criteria) {
		ensureWidget();
		criteriaTextBox.setText((criteria != null) ? criteria : "");
	}

	/**
	 * Get the selected records in the table
	 * 
	 * @return The selected records in the table
	 */
	public List<ExternalUserRecord> getSelectedRecords() {
		return table.getSelectedRecords();
	}

	/**
	 * Check if the table is showing the list view of table records
	 * 
	 * @return true if showing the list view, false otherwise
	 */
	public boolean isShowingListView() {
		return (table != null && table.isShowingListView());
	}

	/*
	 * Add handler for the view cancel event
	 */
	@Override
	public HandlerRegistration addViewCancelHandler(ViewCancelHandler handler) {
		return addHandler(handler, ViewCancelEvent.getType());
	}

	/*
	 * Add handler for the record click event
	 */
	@Override
	public HandlerRegistration addRecordClickHandler(RecordClickHandler<ExternalUserRecord> handler) {
		return addHandler(handler, RecordClickEvent.getType());
	}

	/*
	 * Get the anchor for this widget
	 */
	@Override
	public String getAnchor() {
		/* Lazy view, widget is not ready. The anchor is empty */
		if (getWidget() == null)
			return "";

		return buildAnchor(table.getTableAnchor());
	}

	/*
	 * Set the anchor on this widget
	 */
	@Override
	protected void setWidgetAnchor(String anchor) {
		/* Reset the view */
		reset();

		/* Parse the anchor */
		final TableAnchor tableAnchor = table.getBookmarkFeature().parseAnchor(anchor);

		/* Update the criteria in the view */
		setSearchCriteria(tableAnchor.getTokens().get(CRITERIA_TOKEN));

		/* Update the table */
		initializeTable(new NeptuneAsyncCallback<Void>() {
			@Override
			public void onNeptuneFailure(Throwable caught) {
				/* Should not happen */
			}

			@Override
			public void onNeptuneSuccess(Void result) {
				/* The table is initialized, set the table anchor */
				table.setTableAnchor(tableAnchor);
			}
		});
	}

	/*
	 * Create the lazy widget
	 */
	@Override
	protected Widget createWidget() {
		RealmConstants constants = getRealmConstants();
		tableContainer = new VerticalContainer();

		/* Build the criteria form entry */
		HorizontalPanel criteriaPanel = new HorizontalContainer();
		criteriaPanel.addStyleName(STYLE_CRITERIA_PANEL);
		criteriaPanel.setVerticalAlignment(HasVerticalAlignment.ALIGN_MIDDLE);
		tableContainer.add(criteriaPanel);

		/* Criteria label */
		LabelWidget label = new LabelWidget(constants.externalUserCriteriaLabel());
		label.addStyleName(STYLE_CRITERIA_LABEL);
		criteriaPanel.add(label);

		/* Criteria text box */
		criteriaTextBox = new TextBoxWidget();
		criteriaTextBox.addStyleName(STYLE_CRITERIA_TEXTBOX);
		criteriaTextBox.addKeyPressHandler(new KeyPressHandler() {
			@Override
			public void onKeyPress(KeyPressEvent event) {
				handleCriteriaTextBoxKeyPress(event);
			}
		});
		criteriaPanel.add(criteriaTextBox);

		/* Criteria search button */
		searchButton = new ButtonWidget(constants.externalUserSearchButton());
		searchButton.addClickHandler(new ClickHandler() {
			@Override
			public void onClick(ClickEvent event) {
				handleSearchButtonClick();
			}
		});
		searchButton.addStyleDependentName(StyleNames.COMMIT_BUTTON_STYLE);
		searchButton.addStyleName(STYLE_CRITERIA_BUTTON);
		criteriaPanel.add(searchButton);

		/* Add a section separator */
		SimplePanel sectionSeparator = new SimplePanel();
		VerticalContainer filler = new VerticalContainer();
		filler.add(new LabelWidget());
		sectionSeparator.setWidget(filler);
		sectionSeparator.addStyleName(STYLE_SECTION_SEPARATOR);
		tableContainer.add(sectionSeparator);

		/* Build the data provider and the table widget */
		dataProvider = new ExternalUserDataProvider(this);
		table = new Table<String, ExternalUserRecord>(dataProvider);
		table.addStyleName(STYLE_TABLE);
		table.addAnchorChangeHandler(new TableAnchorChangeHandler() {
			@Override
			public void onAnchorChange(TableAnchorChangeEvent event) {
				handleTableAnchorChange(event.getAnchor());
			}
		});
		tableContainer.add(table);

		/* Add paging feature */
		PageFeature pageFeature = new CookieStoreBasedPageFeatureImpl(COOKIE_PREFIX);
		pageFeature.setShowTotalCount(true);
		table.addPageFeature(pageFeature);

		/* Add sorting feature */
		if (isTableSortingSupported()) {
			Set<Column<?, ExternalUserRecord>> columns = new HashSet<Column<?, ExternalUserRecord>>();
			Column<?, ExternalUserRecord> column = null;
			Column<?, ExternalUserRecord> defaultColumn = null;

			/* Check the user name column */
			if (configuration.isUserNameSortable()) {
				column = dataProvider.getColumn(ExternalUserDataProvider.USERNAME_COLUMN);
				if (defaultColumn == null)
					defaultColumn = column;
				columns.add(column);
			}

			/* Check the first name column */
			if (configuration.isFirstNameSortable()) {
				column = dataProvider.getColumn(ExternalUserDataProvider.FIRSTNAME_COLUMN);
				if (defaultColumn == null)
					defaultColumn = column;
				columns.add(column);
			}

			/* Check the last name column */
			if (configuration.isLastNameSortable()) {
				column = dataProvider.getColumn(ExternalUserDataProvider.LASTNAME_COLUMN);
				if (defaultColumn == null)
					defaultColumn = column;
				columns.add(column);
			}

			/* Add the sort feature */
			SortFeature<String, ExternalUserRecord> sortFeature = new SortFeatureImpl<String, ExternalUserRecord>(
			        defaultColumn, SortOrder.ASCENDING);
			sortFeature.getSortableColumns().addAll(columns);
			table.addSortFeature(sortFeature);
		}

		/* Add the bookmark feature */
		BookmarkFeature bookmarkFeature = new BookmarkFeatureImpl();
		table.addBookmarkFeature(bookmarkFeature);

		/* Add a cancel button to container and not the panel */
		cancelButton = new ButtonWidget(constants.cancelButton());
		cancelButton.addClickHandler(new ClickHandler() {
			@Override
			public void onClick(ClickEvent event) {
				handleCancelButtonClick();
			}
		});
		cancelButton.addStyleDependentName(StyleNames.DATALOSS_BUTTON_STYLE);
		container.addButton(constants.userTab(), cancelButton);
		showCancelButton();

		/* Do not set the table in the container, leave it empty / blank */
		return tableContainer;
	}

	// ========================================================================
	// ===================== INTERNAL STUFF
	// ========================================================================

	/**
	 * Get the constants
	 * 
	 * @return The constants
	 */
	RealmConstants getRealmConstants() {
		if (constants == null)
			constants = GWT.create(RealmConstants.class);

		return constants;
	}

	/**
	 * Get the table
	 * 
	 * @return The table
	 */
	Table<String, ExternalUserRecord> getTable() {
		return table;
	}

	/**
	 * Get the busy indicator
	 */
	BusyIndicator getBusyIndicator() {
		return busyIndicator;
	}

	/*
	 * Build the complete anchor given the table anchor built only from the table. Adds the search
	 * criteria to the anchor and serializes the anchor to a string.
	 */
	private String buildAnchor(TableAnchor tableAnchor) {
		/* Add criteria to the anchor */
		String criteria = getSearchCriteria();
		if (criteria != null && !criteria.trim().isEmpty()) {
			tableAnchor.getTokens().put(CRITERIA_TOKEN, criteria);
		}

		return table.getBookmarkFeature().buildAnchor(tableAnchor);
	}

	/*
	 * Initialize the table with the option of refreshing the contents after initialization
	 */
	private void initializeTable(final AsyncCallback<Void> callback) {
		if (table.isInitialized()) {
			callback.onSuccess(null);
			return;
		}

		table.initialize(new NeptuneAsyncCallback<Void>() {
			@Override
			public void onNeptuneFailure(Throwable caught) {
				/* Will never happen */
				callback.onFailure(caught);
			}

			@Override
			public void onNeptuneSuccess(Void result) {
				callback.onSuccess(result);
			}
		});
	}

	/*
	 * Determine if the table should support sorting
	 */
	private boolean isTableSortingSupported() {
		return (configuration != null && (configuration.isUserNameSortable()
		        || configuration.isFirstNameSortable() || configuration.isLastNameSortable()));
	}

	// ========================================================================
	// ===================== EVENT HANDLING
	// ========================================================================

	/*
	 * Handle the key press in the criteria text box
	 */
	private void handleCriteriaTextBoxKeyPress(KeyPressEvent event) {
		/* Pressing "Enter" key triggers the search */
		if (event.getNativeEvent().getKeyCode() == KeyCodes.KEY_ENTER) {
			searchButton.click();
		}
	}

	/*
	 * Handle the search button click by performing the search with the new criteria
	 */
	private void handleSearchButtonClick() {
		/* Return to the first page and refresh the table */
		table.getPageFeature().setPageNumber(1);
		initializeTable(new NeptuneAsyncCallback<Void>() {
			@Override
			public void onNeptuneFailure(Throwable caught) {
				/* Should never happen */
			}

			@Override
			public void onNeptuneSuccess(Void result) {
				/* Fire an anchor change event */
				table.reset(false);
				table.showListView();
				table.refresh();
				handleTableAnchorChange(table.getTableAnchor());
			}
		});
	}

	/*
	 * Handle the table anchor change
	 */
	private void handleTableAnchorChange(TableAnchor tableAnchor) {
		fireViewAnchorChangeEvent(buildAnchor(tableAnchor));
	}

	/*
	 * Handle the cancel button click
	 */
	private void handleCancelButtonClick() {
		container.reset();
		cancelButton.setVisible(false);
		fireEvent(new ViewCancelEvent());
	}

	/*
	 * Handle the record click
	 */
	void handleTableRecordClick(ExternalUserRecord record) {
		fireEvent(new RecordClickEvent<ExternalUserRecord>(record));
	}
	
	private void showCancelButton(){
		cancelButton.setVisible(true);
	}
	
	public ButtonWidget getCancelButton(){
		return cancelButton;
	}
}
