/*
 * Created on Feb 9, 2010
 * 
 * (C) Copyright TANDBERG Television Inc.
 */

package com.tandbergtv.neptune.widgettoolkit.client.widget.composite;

import com.google.gwt.event.shared.HandlerRegistration;
import com.google.gwt.user.client.Timer;
import com.google.gwt.user.client.ui.Composite;
import com.google.gwt.user.client.ui.Widget;
import com.tandbergtv.neptune.widgettoolkit.client.widget.event.HasRefreshHandlers;
import com.tandbergtv.neptune.widgettoolkit.client.widget.event.RefreshEvent;
import com.tandbergtv.neptune.widgettoolkit.client.widget.event.RefreshHandler;

/**
 * A widget wrapper that adds auto-refresh functionality. Auto-refresh is triggered periodically
 * when the widget is attached to the HTML document, or when the widget is first added to the
 * document. Removing the widget from the document will cease all calls to refresh. If the widget is
 * invisible and attached to the document, the refresh events are still fired.
 * 
 * @author Vijay Silva
 */
public class RefreshWidget<W extends Widget> extends Composite implements HasRefreshHandlers {

	/** The default refresh period is set to 30 milliseconds */
	public static final int DEFAULT_REFRESH_PERIOD = 30000;

	/* Properties */
	private final Timer refreshTimer;
	private int refreshPeriod = DEFAULT_REFRESH_PERIOD;
	private boolean refreshOnLoad = true;
	private boolean refreshPeriodically = true;

	/**
	 * Constructor. Sets the refresh period to be 30 seconds {@link #DEFAULT_REFRESH_PERIOD}, and
	 * ensures that the widget is refreshed on load and is refreshed periodically.
	 * 
	 * @param widget The widget for which auto-refresh functionality is being added
	 */
	public RefreshWidget(W widget) {
		this.initWidget(widget);

		this.refreshTimer = new Timer() {
			@Override
			public void run() {
				refresh();
			}
		};
	}

	// ========================================================================
	// ===================== ACCESSORS
	// ========================================================================

	@SuppressWarnings("unchecked")
	@Override
	public W getWidget() {
		return (W) super.getWidget();
	}

	/**
	 * Get the time period (in milliseconds) after which to trigger the widget refresh
	 * 
	 * @return the refresh periodic rate (in milliseconds)
	 */
	public int getRefreshPeriod() {
		return refreshPeriod;
	}

	/**
	 * Set the time period (in milliseconds) after which to trigger the widget refresh. The refresh
	 * period must be value greater than 0.
	 * 
	 * @param refreshPeriod the refresh periodic rate (in milliseconds)
	 */
	public void setRefreshPeriod(int refreshPeriod) {
		if (refreshPeriod <= 0)
			throw new IllegalArgumentException("The refresh period must be greater than 0.");

		this.refreshPeriod = refreshPeriod;
		updateTimer();
	}

	/**
	 * Determine if the widget is instantly refreshed on loading into HTML DOM
	 * 
	 * @return true if refreshing on load, false otherwise
	 */
	public boolean isRefreshOnLoad() {
		return refreshOnLoad;
	}

	/**
	 * Set if the widget should be instantly refreshed on loading into HTML DOM
	 * 
	 * @param refreshOnLoad flag for if refreshing on load (true to refresh, false otherwise)
	 */
	public void setRefreshOnLoad(boolean refreshOnLoad) {
		this.refreshOnLoad = refreshOnLoad;
	}

	/**
	 * Determine if the widget is refreshed periodically (with the refresh period set)
	 * 
	 * @return true if refreshed periodically, false otherwise
	 */
	public boolean isRefreshPeriodically() {
		return refreshPeriodically;
	}

	/**
	 * Set if the widget should refreshed periodically (with the refresh period set)
	 * 
	 * @param refreshPeriodically flag for periodic refresh (true for refresh, false otherwise)
	 */
	public void setRefreshPeriodically(boolean refreshPeriodically) {
		this.refreshPeriodically = refreshPeriodically;
		updateTimer();
	}

	// ========================================================================
	// ===================== EVENT REGISTRATION
	// ========================================================================

	@Override
	public HandlerRegistration addRefreshHandler(RefreshHandler handler) {
		return this.addHandler(handler, RefreshEvent.getType());
	}

	// ========================================================================
	// ===================== OVERRIDDEN METHODS
	// ========================================================================

	@Override
	protected void onLoad() {
		super.onLoad();

		/* Start the timer, and refresh the widget if required */
		updateTimer();
		if (isRefreshOnLoad())
			refresh();
	}

	@Override
	protected void onUnload() {
		/* Stop the timer */
		refreshTimer.cancel();
		super.onUnload();
	}

	// ========================================================================
	// ===================== PRIVATE METHODS
	// ========================================================================

	/*
	 * Update the timer by canceling previous scheduled tasks, and re-starting the timer is the
	 * widget is attached to the HTML DOM.
	 */
	private void updateTimer() {
		/* Cancel any previously scheduled tasks */
		refreshTimer.cancel();
		if (isAttached() && isRefreshPeriodically()) {
			refreshTimer.scheduleRepeating(refreshPeriod);
		}
	}

	/*
	 * The widget has to be 'refreshed'. Fire the refresh event to all listeners.
	 */
	private void refresh() {
		this.fireEvent(new RefreshEvent());
	}
}
